import java.util.*;

public class Perceptron
{
    Vector           layers;
    Vector           inputSamples;
    Vector           outputSamples;
    public Layer     inputLayer;
    public Layer     outputLayer;
    public int       algorithm;
    public double    error;

    final public static int PERCEPTRON = 1;
    final public static int ADALINE    = 2;
    final public static int BACKPROP   = 3;
    final public static int POCKET     = 4;

    public Perceptron(int i,int o,int algo)
    {
	layers          = new Vector();
	inputSamples    = new Vector();
	outputSamples   = new Vector();
	inputLayer      = new Layer("I",i+1); // bias
	outputLayer     = new Layer("O",o);
	layers.addElement(inputLayer);
	layers.addElement(outputLayer);
	algorithm       = algo;
	error = 0.0;
    }
    public void addLayer(int n)
    {
	layers.insertElementAt(new Layer("H",n),layers.size()-1);
    }
    public Layer getLayer(int i)
    {
	int         j=0;
	boolean     found=false;
	Layer       layer=null;
	Enumeration e = layers.elements();
	while(e.hasMoreElements())
	    {
		layer = (Layer)e.nextElement();
		if (i==j)
		    {
			found = true;
			break;
		    } else j++;
	    }
	if (found==false) layer = null;
	return layer;
    }
    public void connect(int sourceLayer,int sourceNeuron,
			int destLayer,int destNeuron)
    {
	new Synapse(getLayer(sourceLayer).getNeuron(sourceNeuron),
		    getLayer(destLayer).getNeuron(destNeuron));
    }
    public void biasConnect(int destLayer,int destNeuron)
    {
	Synapse s = new Synapse(inputLayer.getNeuron(inputLayer.size-1),
				getLayer(destLayer).getNeuron(destNeuron));
	s.weight = 0.5;
    }
    public void removeSamples()
    {
	inputSamples.removeAllElements();
	outputSamples.removeAllElements();
    }
    public void addSample(Vector i,Vector o)
    {
	inputSamples.addElement(i);
	outputSamples.addElement(o);
    }
    public void printSamples()
    {
	System.out.println(inputSamples+"->"+outputSamples);
    }
    public Vector recognize(Vector iS)
    {
	initInputs(iS);
	propagate();
	Vector oS = getOutput();
	return oS;
    }
    public void learn(int iterations)
    {
	Enumeration iS;
	Enumeration oS;
	for(int i = 0; i<iterations; i++)
	    {
		iS = inputSamples.elements();
		oS = outputSamples.elements();
		while(iS.hasMoreElements()) {
		    Vector temp = (Vector) oS.nextElement ();
		    learnPattern((Vector)iS.nextElement(),temp);
		}
	    }
	error = 0.0;
	iS = inputSamples.elements();
	oS = outputSamples.elements();
	while(iS.hasMoreElements()) {
	    Vector temp = (Vector) oS.nextElement ();
	    recognize((Vector) iS.nextElement());
	    error += computeError (temp);
	}
    }
    void learnPattern(Vector iS, Vector oS)
    {
	initInputs(iS);
	propagate();
	switch(algorithm)
	    {
	    case BACKPROP:
		bpAdjustWeights(oS);
		break;
	    case ADALINE:
		outputLayer.computeAdaline(oS);
		break;
	    case PERCEPTRON:
		outputLayer.computePerceptron(oS);
		break;
	    case POCKET:
		outputLayer.computePocket(oS);
		break;
	    }
    }
    void initInputs(Vector iS)
    {
	Neuron neuron;
	Enumeration e = inputLayer.neurons.elements();
	Enumeration eS = iS.elements();
	while (eS.hasMoreElements())
	    {
		neuron = (Neuron)e.nextElement();
		neuron.output = ((Double)eS.nextElement()).doubleValue();
	    }
	neuron = (Neuron)e.nextElement(); // bias;
	neuron.output = 1.0;
    }
    void propagate()
    {
	Layer layer;
	Enumeration e = layers.elements();
	e.nextElement(); // skip the input layer
	while(e.hasMoreElements())
	    {
		layer = (Layer)e.nextElement();
		layer.computeOutputs(algorithm);
	    }
    }
    public Vector getOutput()
    {
	Vector oS = new Vector();
	Neuron neuron;
	Enumeration e = outputLayer.neurons.elements();
	while(e.hasMoreElements())
	    {
		neuron = (Neuron) e.nextElement();
		switch(algorithm) {
		case BACKPROP:	// sigmoid
		    oS.addElement(new Double(neuron.getOutput()));
		    break;
		case PERCEPTRON:
		case POCKET:
		    oS.addElement(new Double(neuron.getOutput()));
		    break;
		case ADALINE:	// weighted sum
		    oS.addElement(new Double(neuron.getLinearOutput()+0.5));
		default:
		}
	    }
	return oS;
    }
    double computeError(Vector oS)
    {
	Neuron neuron;
	double sum=0.0;
	double tmp;
	Enumeration e = outputLayer.neurons.elements();
	Enumeration eS = oS.elements();
	while (e.hasMoreElements())
	    {
		neuron = (Neuron)e.nextElement();
		switch(algorithm) {
		case BACKPROP:	// sigmoid
		    tmp = ((Double)eS.nextElement()).doubleValue() - neuron.getOutput();	
		    sum += tmp * tmp / 2.0;
		    break;
		case ADALINE:	// weighted sum; convert from {-1,1}
		    tmp = ((Double)eS.nextElement()).doubleValue() - 0.5*neuron.getLinearOutput() - 0.5;	
		    sum += 0.5*tmp * tmp;
		    break;
		case PERCEPTRON:
		case POCKET:
		default:
		    tmp = ((Double)eS.nextElement()).doubleValue() - neuron.getThresholdedOutput();	
		    sum += tmp * tmp;
		}
	    }
	return sum;
    }
    double currentError () {
	return error;
    }
    void bpAdjustWeights(Vector oS)
    {
	outputLayer.computeBackpropDeltas(oS);
	for(int i=layers.size()-2; i>=1; i--)
	    ((Layer)layers.elementAt(i)).computeBackpropDeltas();
	outputLayer.computeWeights();
	for(int i=layers.size()-2; i>=1; i--)
	    ((Layer)layers.elementAt(i)).computeWeights();
    }
    void print()
    {
	Layer layer;
	Enumeration e = layers.elements();
	while(e.hasMoreElements())
	    {
		layer = (Layer)e.nextElement();
		layer.print();
	    }
    }
    public void restorePocket()
    {
	outputLayer.restorePocket();
    }
    public void initPocket()
    {
	outputLayer.initPocket();
    }
}
