// -========-
// DynGraph2D
// -========-

// This class is an extension of class Graph2D and allows dynamic drawing
// of new points that haven't been drawn yet.

// Use this class in conjuntion with class DynDataSet


import java.awt.*;
import graph.*;


public class DynGraph2D extends Graph2D {

	protected Rectangle rect;
	protected Graphics ongG = null;
	protected int height = 0, width = 0;
	protected int steps = 0;
	protected final int maxsteps = 30;

	// Constructors
	// ------------

	public DynGraph2D() {}

	public DynGraph2D(int width, int height) {
		this.width = width;
		this.height = height;
	}

	// Destructor
	// ----------

	// In order to free unneeded system resources call this method.
	public void dispose() {
		if (ongG != null) ongG.dispose();
		ongG = null;
	}

	// Public members
	// --------------

	// Override preffered size methods to our suits.
	public Dimension getPreferredSize() {
		return new Dimension(width, height);
	}

	public Dimension getMinimumSize() {
		return getPreferredSize();
	}

	public Dimension getMaximumSize() {
		return getPreferredSize();
	}

	// Overrides paint to adapt internal dimension parameters for
	// dynamic drawing. This internal dimension parmeters hold until
	// a new paint is invoked (so it is most likely that some window
	// dimension changed).
	public void paint(Graphics g) {
		if (!axis.isEmpty()) axisRangeResized();
		super.paint(g);
		updateDrawWindow(g);
		steps = 0;
	}

	// Updates dynamically the ongoing graph while drawing the points
	// that haven't been drawn yet.
	public void dynUpdate() {
		int i;
		Graphics g;

		steps++;

		if (ongG == null || 
				(steps > maxsteps && !axis.isEmpty() && axisRangeResized())) {
			update(g = getGraphics());
			if (g != null) g.dispose();
			steps = 0;
			return;
		}

		if(!paintAll) return;

		if(!dataset.isEmpty()) {
			for (i=0; i<dataset.size(); i++) {
				((DynDataSet)dataset.elementAt(i)).dynDrawData(ongG, rect);
			}
		}
	}


	// Protected members
	// -----------------

	// Tests whether the axis range has changed.
	// Tests only the y-axis which must be the second axis attached to the graph
	protected boolean axisRangeResized() {
		Axis ay = (Axis)axis.elementAt(1);
		double mn = ay.getDataMin(), mx = ay.getDataMax();
		
		// If minimum is the same as maximum, there most likely won't be
		// any data sets attached to the axis, so we leave the ranges unchanged.
		if (Dbl.eq(mn, mx)) return false;

		// Range got bigger.
		if ((ay.maximum - ay.minimum) < (mx - mn)) {
			ay.minimum = mn;
			ay.maximum = mx;
			return true;
		}

		// Range got smaller than a certain threshold (fifth of actual range)
		if ((ay.maximum - ay.minimum)/5 > (mx - mn)) {
			ay.minimum = mn;
			ay.maximum = mx;
			return true;			
		}

		return false;
	}
	

	// Updates the draw window upon a new paint invocation
	protected void updateDrawWindow(Graphics g) {
		int i;

		if (ongG != null) ongG.dispose();
		ongG = g.create();

		rect = getBounds();
		if (ongG == null) return;

		rect.x = 0;
		rect.y = 0;

		rect.x      += borderLeft;
		rect.y      += borderTop;
		rect.width  -= borderLeft+borderRight;
		rect.height -= borderBottom+borderTop;

		if(!axis.isEmpty()) adaptRectByAxis(ongG, rect);
	}

	// Adapts the drawing rectangle by means of the involved graphic axis.
	// Behaves similar like draw_axis but does no drawing.
	protected void adaptRectByAxis(Graphics g, Rectangle r) {
		Axis a;
		int waxis;
		Rectangle dr;
		int x, y, width, height;

		if(square) r = ForceSquare(g,r);
		dr = getDataRectangle(g,r);

		x      = dr.x;
		y      = dr.y;
		width  = dr.width;
		height = dr.height;

		for (int i=0; i<axis.size(); i++) {
			a = ((Axis)axis.elementAt(i));
			waxis = a.getAxisWidth(g);
			
			a.data_window = new Dimension(width, height);
			switch (a.getAxisPos()) {
			case Axis.LEFT:
				r.x += waxis;
				r.width -= waxis;
				break;
			case Axis.RIGHT:
				r.width -= waxis;
				break;
			case Axis.TOP:
				r.y += waxis;
				r.height -= waxis;
			case Axis.BOTTOM:
				r.height -= waxis;
				break;
			}
		}
	}
}
