// -====-
// HHAxon
// -====-

// This class implements the behaviour of an Hodgkin-Huxley axon.

// HHAxon inherits from CableOdeFct in order to use the numerical ODE solvers
// of the classes derived from BaseOde.

// Author:			Pollinger Thomas
// Built:				Feb 26th, 1998


// Class HHDyn for ODE dynamic variables
// -------------------------------------

class HHDyn extends OdeFctRep {
	
	protected double[] im;

	public HHDyn(int nb_comp) {
		im = new double[nb_comp];
	}

	public HHDyn(final HHDyn hd) {
		super(hd);
		im = ArrayDup.dup(hd.im);
	}

	public void set_nb_comp(int nb_comp) {
		im = new double[nb_comp];
	}

	public Object clone() {
		return new HHDyn(this);
	}
}


public class HHAxon extends CableOdeFct {

	protected BaseOde solver;

	protected double i_j, clamp_v;
	protected double ra, cm, gna, gk, gl;
	protected double e_rest, vna, vk, vl;
	protected double clamp_val;
	protected double vm_t;

	protected Simtime sim;
	protected double t_beg, t_dur;

	protected double[] mx, mxt;

	protected int ong_comp;
	protected boolean space_clamp;

	protected final int hhv = 0;
	protected final int hhn = 1;
	protected final int hhm = 2;
	protected final int hhh = 3;

	public HHAxon
	(double length, double comp_length, double diameter,
	 double t_begin, double t_duration,
	 double inject_cur, double clamp_pot, double rest_pot, 
	 double c_m, double r_a, double g_na, double g_k, double g_l, 
	 double na_i, double na_o, double k_i, double k_o,
	 boolean space_clamp) 
	{
		super(length, comp_length, 4, 1, new HHDyn(0));
		((HHDyn)dynamics).set_nb_comp(nb);

		final double R = 8.314, F = 9.65e4, T = 273.15 + 6.3;
		double comp_sect = Math.PI*diameter*diameter/4;
		double comp_surface = Math.PI*diameter*comp_length;
		int k;
		double n_rest, m_rest, h_rest;
		double aln, betn, alm, betm, alh, beth;

		e_rest = rest_pot;
		this.solver = null;

		sim = null;
		t_beg = t_begin;
		t_dur = t_duration;

		vna = - (R * T / F) * 1000 * Math.log(na_i / na_o) + 70;
		vk = - (R * T / F) * 1000 * Math.log(k_i / k_o) + 70;
		vl = 10.613;
		
		ra = r_a * clen / comp_sect;
		cm = c_m * comp_surface;
		gna = g_na * comp_surface;
		gk = g_k * comp_surface;
		gl = g_l * comp_surface;

		clamp_v = clamp_pot - e_rest;
		i_j = inject_cur;
		this.space_clamp = space_clamp;

		mx = new double[nb];
		mxt = new double[nb];

		aln = a_n(0.0); betn = b_n(0.0);
		alm = a_m(0.0); betm = b_m(0.0);
		alh = a_h(0.0); beth = b_h(0.0);

		n_rest = aln / (aln + betn);
		m_rest = alm / (alm + betm);
		h_rest = alh / (alh + beth);

		vm_t = 0.0;
		for (k = 0; k < nb; k++) {
			var[k][hhv] = 0.0;
			var[k][hhn] = n_rest;
			var[k][hhm] = m_rest;
			var[k][hhh] = h_rest;
			mx[k] = 0.0;
			mxt[k] = 0.0;
		}
	}

	public HHAxon(final HHAxon ha) {
		super(ha);

		solver = null;

		i_j = ha.i_j;
		clamp_v = ha.clamp_v;

		ra = ha.ra; 
		cm = ha.cm;
		gna = ha.gna; 
		gk = ha.gk; 
		gl = ha.gl;

		e_rest = ha.e_rest; 
		vna = ha.vna;
		vk = ha.vk;
		vl = ha.vl;
		clamp_val = ha.clamp_val;

		sim = ha.sim;
		t_beg = ha.t_beg;
		t_dur = ha.t_dur;
		
		mx = ArrayDup.dup(ha.mx);
		mxt = ArrayDup.dup(ha.mxt);

		ong_comp = ha.ong_comp;
		space_clamp = ha.space_clamp;
	}

	public void set_simtime(final Simtime sim) {
		this.sim = sim;
	}

	public boolean set_compartment(int comp_nb) {
		if (ong_comp > var.length) return false;
		ong_comp = comp_nb;
		return true;
	}

	public void set_solver(BaseOde solver) {
		this.solver = solver;
		solver.init(this);
	}
	
	public void new_potential() {
		init_output();
		solver.eval(sim.getTimeStep());
		evaluate_output();
	}

	public double get_v_m() {
		return var[ong_comp][hhv] + e_rest;
	}

	public double get_i_m() {
		return ((HHDyn)dynamics).im[ong_comp];
	}

	public double get_i_na() {
		double mval = get_m();
		double t1 = mval*mval;
		return gna*t1*mval*get_h()*(var[ong_comp][hhv]-vna);
	}

	public double get_i_k() {
		double nval = get_n();
		double t1 = nval*nval;
		return gk*t1*t1*(var[ong_comp][hhv]-vk);
	}

	public double get_i_l() {
		return gl*(var[ong_comp][hhv] - vl);
	}

	public double get_i_c() {
		return get_i_m() - get_i_na() - get_i_k()- get_i_l();
	}

	public double get_n() {
		return var[ong_comp][hhn];
	}

	public double get_m() {
		return var[ong_comp][hhm];
	}

	public double get_h() {
		return var[ong_comp][hhh];
	}

	public double f() {
		switch (ong_var) {
		case 0: return f_v();
		case 1: return f_n();
		case 2: return f_m();
		default: return f_h();
		}
	}

	public void update_ode_state(final OdeFct x) {
		clamp_val = ((HHAxon)x).clamp_val;
		space_clamp = ((HHAxon)x).space_clamp;
	}

	public double left(int ong_variable) {
		if (ong_variable > 0) return super.left(ong_variable);
		if (space_clamp) return clamp_val*ra + lat_var[0];
		return lat_var[0];
	}

	public double right(int ong_variable) {
		if (ong_variable > 0) return super.right(ong_variable);
		return lat_var[0];
	}

	public Object clone() {
		return new HHAxon(this);
	}

	// Protected members / friend methods
	// ----------------------------------

	protected void init_output() {
		if (sim.getTime() >= t_beg && sim.getTime() <= t_beg + t_dur)
			if (space_clamp) clamp_val = i_j;
			else clamp_val = clamp_v;
		else
			clamp_val = 0.0;		 
	}
	
	protected boolean evaluate_output() {
		int k;
		
		for (k = 0; k < nb; k++)
			if (var[k][0] > mx[k]) { 
				mx[k] = var[k][0]; 
				mxt[k] = sim.getTime();
			}
		
		return true;
	}

	protected double f_v() {
		double vp = lat_var[0];
		double t1, t8, t7, cur;

		t1 = var[i][hhm]*var[i][hhm];
		t7 = var[i][hhn]*var[i][hhn];
		t8 = t7*t7;

		// when doing voltage clamp
		if (!space_clamp) {
			double ic;
			lat_var[0] = clamp_val;
			ic = (clamp_val - vm_t)/sim.getTimeStep();
		  ((HHDyn)dynamics).im[i] = 
				ic+gna*t1*var[i][hhm]*var[i][hhh]*(vp-vna)+gk*t8*(vp-vk)+gl*(vp-vl);
			vm_t = clamp_val;
			return 0.0;
		}

		// space/current clamp
		cur = ((HHDyn)dynamics).im[i] = (lat_var_left[0]-2*vp+lat_var_right[0])/ra;
		return((cur-gna*t1*var[i][hhm]*var[i][hhh]*(vp-vna)-gk*t8*(vp-vk)-
						gl*(vp-vl))/cm);
	} 

	protected double f_n() {
		double vp = lat_var[0];
		if (Dbl.eq(vp, 10)) return 0.1 - 0.2103121128*var[i][hhn];
		return((1-var[i][hhn])*(0.1-0.1E-1*vp)/(Math.exp(1-vp/10)-1)-var[i][hhn]*
					 Math.exp(-vp/80)/8);
	} 

	protected double f_m() {
		double vp = lat_var[0];
		if (Dbl.eq(vp, 25)) return 1 - 1.997408835*var[i][hhm];
		return(0.1*(1-var[i][hhm])*(25-vp)/(Math.exp(5.0/2.0-vp/10)-1)-4*var[i][hhm]*
					 Math.exp(-vp/18));
	}

	protected double f_h() {
		double vp = lat_var[0];
		return(0.7E-1*(1-var[i][hhh])*Math.exp(-vp/20)-var[i][hhh]/(Math.exp(3-vp/10)+1));
	}


	protected double a_n(double vp) {
		if (Dbl.eq(vp, 10.0)) return 0.1;
		return (0.01*(10.0 - vp) / (Math.exp((10.0 - vp) / 10.0) - 1.0));
	}

	protected double b_n(double vp) {
		return (0.125*Math.exp(- vp / 80.0));
	}

	protected double a_m(double vp) {
		if (Dbl.eq(vp, 25.0)) return 1.0;
		return (0.1*(25.0 - vp) / (Math.exp((25.0 - vp) / 10.0) - 1.0));
	}

	protected double b_m(double vp) {
		return (4*Math.exp(- vp / 18.0));
	}

	protected double a_h(double vp) {
		return (0.07*Math.exp(- vp / 20.0));
	}

	protected double b_h(double vp) {
		return (1.0 / (Math.exp((30.0 - vp) / 10.0) + 1.0));
	}
}
