// -====-
// HHCell
// -====-

// Applet simulating one Hodgkin-Huxley cell.


// Author:			Pollinger Thomas
// Built:				Feb 27th, 1998
// v 1.0:				May 3rd, 1998:	Splitting into output windows


import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import gjt.*;
import graph.*;

import HHAxon;


// **********************************************
// Main class for simulating Hodgkin-Huxley cells
// **********************************************

public class HHCell extends Applet implements ItemListener, Notifiable {

	public class TextAdapter implements TextListener {
		public void textValueChanged(TextEvent event) {}
	}


	// MEMBERS FOR INPUT CONTROL
	// -------------------------

	// Data members for the input of the simulation.
	private HHAxon hhcell;
	private Simulation sim;
	private boolean running = false;

	private	double gna = 120, gk = 36, gl = 0.3, cm = 1;
	private double ra = 0.03, diameter = 0.05;
	private	double nai = 70.96, nao = 460.0, ki = 301.4, ko = 10.0;
	private	double tbegStim = 1.0, tdurStim = 40.0; 
	private double tbeg = 0.0, tend = 60.0, tdt = 0.005;
	private double tres;
	private double inject_cur = 0.5, clamp_volt = 10.0;
	private boolean doing_space_clamp = true;

	private final double length = 0.05;
	private final double comp_length = 0.05;
	private final double hold_pot = -70.0;

	// Length of text fields
	private final int LENGTH = 8;
	private final int TIMELENGTH = 5;

	// Members related to parameter input
	private Label title = new Label("Simulating a Hodgkin-Huxley Cell");

	// Grouping input field into four panels
	// The first represents all parameters charaterizing a HH cell.
	// Next are externally applied values (e.g. current or voltage).
	// Then, the next panel holds all paramters related to a simulation.
	// The last panel controls the output windows.
	private Panel cellInputPanel = new Panel();
	private Panel cellInputPanel1 = new Panel();
	private Panel cellInputPanel2 = new Panel();
	private Panel cellInputPanel3 = new Panel();
	private Panel cellInputPanel4 = new Panel();
	private Panel extInputPanel = new Panel();
	private Panel extInputPanel1 = new Panel();
	private Panel extInputPanel2 = new Panel();
	private Panel simInputPanel = new Panel();
	private Panel simInputPanel1 = new Panel();
	private Panel simInputPanel2 = new Panel();
	private Panel outputPanel = new Panel();
	private ButtonPanel outputPanel1 = new ButtonPanel();
	private ButtonPanel outputPanel2 = new ButtonPanel();

	// Frames
	private gjt.Box cellInputPanelBox = 
	new gjt.Box(cellInputPanel, "Cell Charateristics", Orientation.LEFT);
	private gjt.Box extInputPanelBox = 
	new gjt.Box(extInputPanel, "Applying External Current/Voltage", 
							Orientation.LEFT);
	private gjt.Box simInputPanelBox = 
	new gjt.Box(simInputPanel, "Simulation Parameters", Orientation.LEFT);
	private gjt.Box outputPanelBox =
	new gjt.Box(outputPanel, "Output Control", Orientation.LEFT);

	// Layout Managers
	private GridBagLayout gbl = new GridBagLayout();
	private GridBagConstraints gbc = new GridBagConstraints();

	private Separator sep = new Separator();

	// These are the text fields and labels for the input parameters
	private Label diameterLabel = new Label("Cell diameter: ");
	private Label cmLabel = new Label("C_m: ");
	private Label raLabel = new Label("R_a: ");

	private Label gnaLabel = new Label("G_Na: ");
	private Label gkLabel = new Label("G_K: ");
	private Label glLabel = new Label("G_l: ");
	private Label naiLabel = new Label("Na_i: ");
	private Label naoLabel = new Label("Na_o: ");
	private Label kiLabel = new Label("K_i: ");
	private Label koLabel = new Label("K_o: ");

	private CheckboxGroup extGrp = new CheckboxGroup();
	private Checkbox spaceclmp = new Checkbox("Current clamp", true, extGrp);
	private Checkbox voltclmp = new Checkbox("Voltage clamp", false, extGrp);

	private Label tbegStimLabel = new Label("Begin Stimulation: ");
	private Label tdurStimLabel = new Label("Duration: ");

	private Label tbegLabel = new Label("Begin: ");
	private Label tendLabel = new Label("End: ");
	private Label tdtLabel = new Label("Time Step: ");
	private Label tresLabel = new Label("Time Scale Res: ");

	private TextField diameterField = new TextField(LENGTH);
	private TextField cmField = new TextField(LENGTH);
	private TextField raField = new TextField(LENGTH);

	private TextField gnaField = new TextField(LENGTH);
	private TextField gkField = new TextField(LENGTH);
	private TextField glField = new TextField(LENGTH);
	private TextField naiField = new TextField(LENGTH);
	private TextField naoField = new TextField(LENGTH);
	private TextField kiField = new TextField(LENGTH);
	private TextField koField = new TextField(LENGTH);

	private TextField extField = new TextField(LENGTH);

	private TextField tbegStimField = new TextField(TIMELENGTH);
	private TextField tdurStimField = new TextField(TIMELENGTH);

	private TextField tbegField = new TextField(TIMELENGTH);
	private TextField tendField = new TextField(TIMELENGTH);
	private TextField tdtField = new TextField(TIMELENGTH);
	private TextField tresField = new TextField(TIMELENGTH);

	// Buttons to control simulation / output
	private Button showButton = new Button("Show");
	private Button hideButton = new Button("Hide");
	private Button refreshButton = new Button("Refresh");

	private Button runButton = new Button("Run");
	private Button stopButton = new Button("Stop");
	private Button clearButton = new Button("Clear");

	// Reference to the display window
	private GraphDialog graphWindow = 
	new GraphDialog(new Frame(), "Graphic Output", tbeg, tend);


	// --------------
	// PUBLIC METHODS
	// --------------

	public void init() {

		// SETTING UP INDIVIDUAL PANELS
		// ----------------------------

		// Setting up Cell panels
		cellInputPanel1.add(diameterLabel);
		cellInputPanel1.add(diameterField);
		cellInputPanel1.add(cmLabel);
		cellInputPanel1.add(cmField);
		cellInputPanel1.add(raLabel);
		cellInputPanel1.add(raField);

		cellInputPanel2.add(gnaLabel);
		cellInputPanel2.add(gnaField);
		cellInputPanel2.add(gkLabel);
		cellInputPanel2.add(gkField);
		cellInputPanel2.add(glLabel);
		cellInputPanel2.add(glField);

		cellInputPanel3.add(naiLabel);
		cellInputPanel3.add(naiField);
		cellInputPanel3.add(naoLabel);
		cellInputPanel3.add(naoField);

		cellInputPanel4.add(kiLabel);
		cellInputPanel4.add(kiField);
		cellInputPanel4.add(koLabel);
		cellInputPanel4.add(koField);

		// Setting up Ext panel
		spaceclmp.addItemListener(this);
		voltclmp.addItemListener(this);
		extInputPanel1.add(spaceclmp);
		extInputPanel1.add(voltclmp);
		extInputPanel1.add(extField);

		extInputPanel2.add(tbegStimLabel);
		extInputPanel2.add(tbegStimField);
		extInputPanel2.add(tdurStimLabel);
		extInputPanel2.add(tdurStimField);

		// Setting up Simulation panel
		simInputPanel1.add(tbegLabel);
		simInputPanel1.add(tbegField);
		simInputPanel1.add(tendLabel);
		simInputPanel1.add(tendField);

		simInputPanel2.add(tdtLabel);
		simInputPanel2.add(tdtField);
		simInputPanel2.add(tresLabel);
		simInputPanel2.add(tresField);

		// Setting up Output Control panel
		hideButton.setEnabled(false);
		outputPanel1.add(showButton);
		outputPanel1.add(hideButton);
		outputPanel1.add(refreshButton);

		stopButton.setEnabled(false);
		outputPanel2.add(runButton);
		outputPanel2.add(stopButton);
		outputPanel2.add(clearButton);


		// Get graph display resolution
		tres = graphWindow.getDisplayRes(tbeg, tend);

		// Setting default values
		diameterField.setText(Double.toString(diameter));
		cmField.setText(Double.toString(cm));
		raField.setText(Double.toString(ra));
		gnaField.setText(Double.toString(gna));
		gkField.setText(Double.toString(gk));
		glField.setText(Double.toString(gl));
		naiField.setText(Double.toString(nai));
		naoField.setText(Double.toString(nao));
		kiField.setText(Double.toString(ki));
		koField.setText(Double.toString(ko));

		extField.setText(Double.toString(inject_cur));
		tbegStimField.setText(Double.toString(tbegStim));
		tdurStimField.setText(Double.toString(tdurStim));
		
		tbegField.setText(Double.toString(tbeg));
		tendField.setText(Double.toString(tend));
		tdtField.setText(Double.toString(tdt));
		tresField.setText(Double.toString(tres));


		// LAYING OUT PANELS
		// -----------------

		// Setting up layout manager for applet
		setLayout(gbl);
		
		// Title
		title.setFont(new Font("Times-Roman", Font.BOLD, 14));
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbl.setConstraints(title, gbc);
		add(title);

		// Insert separator
		gbc.fill  = GridBagConstraints.HORIZONTAL;
		gbc.insets = new Insets(0, 0, 10, 0);
		gbl.setConstraints(sep, gbc);
		add(sep);  

		// Cell input panel
		cellInputPanel.setLayout(new GridLayout(4,1));
		cellInputPanel.add(cellInputPanel1);
		cellInputPanel.add(cellInputPanel2);
		cellInputPanel.add(cellInputPanel3);
		cellInputPanel.add(cellInputPanel4);
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbl.setConstraints(cellInputPanelBox, gbc);
		add(cellInputPanelBox);

		// Ext input panel
		extInputPanel.setLayout(new GridLayout(2,1));
		extInputPanel.add(extInputPanel1);
		extInputPanel.add(extInputPanel2);
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbl.setConstraints(extInputPanelBox, gbc);
		add(extInputPanelBox);

		// Sim input panel
		simInputPanel.setLayout(new GridLayout(2,1));
		simInputPanel.add(simInputPanel1);
		simInputPanel.add(simInputPanel2);
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbl.setConstraints(simInputPanelBox, gbc);
		add(simInputPanelBox);

		// Output Control panel
		outputPanel.setLayout(new GridLayout(2,1));
		outputPanel.add(outputPanel1);
		outputPanel.add(outputPanel2);
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbl.setConstraints(outputPanelBox, gbc);
		add(outputPanelBox);

		
		// PLACING ACTION LISTENERS
		// ------------------------

		// If run button pressed, initialize a new simulation and start drawing
		// the results.
		runButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				setAllEnabled(false);
				updateParameters();
				hhcell = new HHAxon(length, comp_length, diameter, tbegStim, tdurStim,
														inject_cur, clamp_volt, hold_pot, cm, ra,
														gna, gk, gl, nai, nao, ki, ko, doing_space_clamp);
				
				sim = new Simulation(hhcell, tend, tdt, tres, HHCell.this);
				sim.setPriority(Thread.MIN_PRIORITY);
				sim.setDisplays(graphWindow.getDisplays(), 
												graphWindow.getDataSetHandles(),
												graphWindow.getVariables(), 
												graphWindow.getCompartments());
				sim.start();
				running = true;
			}
		});
		
		// If the stop button is pressed, halt immediately ongoing simulation.
		stopButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				if (running) {
					sim.halt();
					notifyOnEnd();
				}
				running = false;
			}
		});

		// Clears the graphic window displays.
		clearButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				graphWindow.clearDisplays(tend);
			}
		});

		// Shows the graphic window.
		showButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				graphWindow.show();
				showButton.setEnabled(false);
				refreshButton.setEnabled(true);
				hideButton.setEnabled(true);
			}
		});

		// Hides the graphic window.
		hideButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				graphWindow.setVisible(false);
				showButton.setEnabled(true);
				refreshButton.setEnabled(false);
				hideButton.setEnabled(false);
			}
		});

		// Refreshes the graphic output window.
		refreshButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				graphWindow.repaintDisplays();
			}
		});

		// If the time simulation control panel is modified, the graphic display
		// dimensions have to be adapted to the new range of the time scale.
		tbegField.addTextListener(new TextAdapter() {
			public void textValueChanged(TextEvent event) {
				tbeg = new Double(tbegField.getText()).doubleValue();				
				tres = graphWindow.getDisplayRes(tbeg, tend);
				tresField.setText(Double.toString(tres));
				graphWindow.resizeDisplays(tend);
				graphWindow.repaintDisplays();
			}
		});

		tendField.addTextListener(new TextAdapter() {
			public void textValueChanged(TextEvent event) {
				tend = new Double(tendField.getText()).doubleValue();				
				tres = graphWindow.getDisplayRes(tbeg, tend);
				tresField.setText(Double.toString(tres));
				graphWindow.resizeDisplays(tend);
				graphWindow.repaintDisplays();
			}
		});

		// Add the first data sets for each graph
		graphWindow.addDataSets();
		graphWindow.resizeDisplays(tend);
	}

	public void start() {}

	public void stop() {}

	public void destroy() {
		remove(cellInputPanel);
		remove(title);
	}

	// Notified if simulation ended
	public void notifyOnEnd() {
		setAllEnabled(true);
		graphWindow.cloneAllDataSets();
	}

	// Checkbox listener for space/voltage clamp
	public void itemStateChanged(ItemEvent event) {
		Checkbox cbox = (Checkbox)event.getSource();

		// Test whether current clamp
		if (cbox == spaceclmp) {
			doing_space_clamp = true;
			clamp_volt = new Double(extField.getText()).doubleValue();
			extField.setText(Double.toString(inject_cur));

			// Doing voltage clamp
		} else if (cbox == voltclmp) {
			doing_space_clamp = false;
			inject_cur = new Double(extField.getText()).doubleValue();
			extField.setText(Double.toString(clamp_volt));
		}
	}

	// Update entered parameters
	public void updateParameters() {
		diameter = new Double(diameterField.getText()).doubleValue();
		cm = new Double(cmField.getText()).doubleValue();
		ra = new Double(raField.getText()).doubleValue();
		gna = new Double(gnaField.getText()).doubleValue();
		gk = new Double(gkField.getText()).doubleValue();
		gl = new Double(glField.getText()).doubleValue();
		nai = new Double(naiField.getText()).doubleValue();
		nao = new Double(naoField.getText()).doubleValue();
		ki = new Double(kiField.getText()).doubleValue();
		ko = new Double(koField.getText()).doubleValue();

		if (doing_space_clamp) 
			inject_cur = new Double(extField.getText()).doubleValue();
		else clamp_volt = new Double(extField.getText()).doubleValue();
		tdurStim = new Double(tdurStimField.getText()).doubleValue();
		tbegStim = new Double(tbegStimField.getText()).doubleValue();

		tbeg = new Double(tbegField.getText()).doubleValue();
		tend = new Double(tendField.getText()).doubleValue();
		tdt = new Double(tdtField.getText()).doubleValue();
		tres = new Double(tresField.getText()).doubleValue();
	}

	// Protected members
	// -----------------


	// Sets all buttons and labels enabled or disabled according to the 
	// boolean parameter 'enabled'.
	protected void setAllEnabled(boolean enabled) {
		diameterLabel.setEnabled(enabled);
		cmLabel.setEnabled(enabled);
		raLabel.setEnabled(enabled);
		
		gnaLabel.setEnabled(enabled);
		gkLabel.setEnabled(enabled);
		glLabel.setEnabled(enabled);
		naiLabel.setEnabled(enabled);
		naoLabel.setEnabled(enabled);
		kiLabel.setEnabled(enabled);
		koLabel.setEnabled(enabled);

		spaceclmp.setEnabled(enabled);
		voltclmp.setEnabled(enabled);
	
		tdurStimLabel.setEnabled(enabled);
		tbegStimLabel.setEnabled(enabled);
	
		tbegLabel.setEnabled(enabled);
		tendLabel.setEnabled(enabled);
		tdtLabel.setEnabled(enabled);
		tresLabel.setEnabled(enabled);
 
		diameterField.setEnabled(enabled);
		cmField.setEnabled(enabled);
		raField.setEnabled(enabled);

		gnaField.setEnabled(enabled);
		gkField.setEnabled(enabled);
		glField.setEnabled(enabled);
		naiField.setEnabled(enabled);
		naoField.setEnabled(enabled);
		kiField.setEnabled(enabled);
		koField.setEnabled(enabled);

		extField.setEnabled(enabled);
		tdurStimField.setEnabled(enabled);
		tbegStimField.setEnabled(enabled);

		tbegField.setEnabled(enabled);
		tendField.setEnabled(enabled);
		tdtField.setEnabled(enabled);
		tresField.setEnabled(enabled);

		runButton.setEnabled(enabled);
		stopButton.setEnabled(!enabled);
		clearButton.setEnabled(enabled);

		graphWindow.setButtonsEnabled(enabled);
	}
}


// ********************************************************************
// Class GraphDialog for graphs that will be shown in a separate window
// ********************************************************************

class GraphDialog extends Dialog implements ItemListener {

	// MEMBERS FOR GRAPHICAL COLOR & SHAPE DEFINITION
	// ----------------------------------------------

	// Some constant values
	private final double hold_pot = -70.0;

	// Layout manager
	private GridBagLayout gbl = new GridBagLayout();
	private GridBagConstraints gbc = new GridBagConstraints();

	// Variables that set the initial graph window size (y-axis)
	private final double vmMin = hold_pot - 20, vmMax = 50;
	private final double curMin = -10, curMax = 10;
	private final double nmhMin = 0, nmhMax = 1;

	// Height & width of a particular graph display
	private int displayHeight, displayWidth;

	// Default colors
	private final Color DATABACKGROUND = new Color(180, 180, 180);
	private final Color GRAPHBACKGROUND = new Color(150,150,150);

	private final Color Cvm = new Color(255,255,0);

	private final Color Cim = new Color(255,255,0);
	private final Color Cina = new Color(255,0,0);
	private final Color Cik = new Color(0,255,0);
	private final Color Cil = new Color(0,0,255);
	private final Color Cic = new Color(0,255,255);

	private final Color Cn = new Color(255,255,0);
	private final Color Cm = new Color(255,0,0);
	private final Color Ch = new Color(0,255,0);


	// MEMBERS FOR GRAPHICAL OUTPUT CONTROL
	// ------------------------------------

	// Graph displays
	private Display vmDisplay, nmhDisplay, iDisplay;

	// Panels for graph displays
	private Panel voltagePanel = new Panel();
	private Panel currentPanel = new Panel(); 
	private Panel paramPanel = new Panel();

	// Frames
	private gjt.Box voltagePanelBox = 
	new gjt.Box(voltagePanel, "Membrane Voltage", Orientation.LEFT);
	private gjt.Box currentPanelBox = 
	new gjt.Box(currentPanel, "Membrane Current", Orientation.LEFT);
	private gjt.Box paramPanelBox = 
	new gjt.Box(paramPanel, "Parameters of HH Dynamics", Orientation.LEFT);

	// Panels for checkboxes
	private Panel iPanel = new Panel();
	private Panel nmhPanel = new Panel();

	// Graph control chechboxes
	private boolean imDraw = true, inaDraw = true, ikDraw = true;
	private boolean ilDraw = true, icDraw = true;
	private boolean nDraw = true, mDraw = true, hDraw = true;

 	private Checkbox imBox = new Checkbox("i_m", true);
 	private Checkbox inaBox = new Checkbox("i_na", true);
 	private Checkbox ikBox = new Checkbox("i_k", true);
 	private Checkbox ilBox = new Checkbox("i_l", true);
 	private Checkbox icBox = new Checkbox("i_c", true);

 	private Checkbox nBox = new Checkbox("n", true);
	private Checkbox mBox = new Checkbox("m", true);
	private Checkbox hBox = new Checkbox("h", true);

	// Graph variable references
	private Display[] displays = null;
	private int[] variables = null, datasetHandles = null, compartments = null;


	// --------------
	// PUBLIC METHODS
	// --------------

	GraphDialog(Frame f, String title, double tbeg, double tend) {
		super(f, title);
		//setResizable(false);

		// Setting up panels for graphic output control
		imBox.setForeground(Cim);
		inaBox.setForeground(Cina);
		ikBox.setForeground(Cik);
		ilBox.setForeground(Cil);
		icBox.setForeground(Cic);
		nBox.setForeground(Cn);
		mBox.setForeground(Cm);
		hBox.setForeground(Ch);

		imBox.addItemListener(this);
		inaBox.addItemListener(this);
		ikBox.addItemListener(this);
		ilBox.addItemListener(this);
		icBox.addItemListener(this);
		nBox.addItemListener(this);
		mBox.addItemListener(this);
		hBox.addItemListener(this);

		iPanel.add(imBox);
		iPanel.add(inaBox);
		iPanel.add(ikBox);
		iPanel.add(ilBox);
		iPanel.add(icBox);

		nmhPanel.add(nBox);
		nmhPanel.add(mBox);
		nmhPanel.add(hBox);

		// Setting initial preferred display width and height.
		displayWidth = (int)(iPanel.getPreferredSize().width*16);
		displayHeight = (int)(displayWidth/2.5);
		System.out.println("Display width: " + displayWidth);
		
		// Evaluate default member values
		vmDisplay =	new Display
			(DATABACKGROUND, GRAPHBACKGROUND, "Time in ms", "V_{m} in mV",
			 tbeg, tend, vmMin, vmMax, displayWidth, displayHeight);
		iDisplay =	new Display
			(DATABACKGROUND, GRAPHBACKGROUND, 
			 "Time in ms", "i_{m}, i_{Na}, i_{K}, i_{l}, i_{c} in uA",
			 tbeg, tend, curMin, curMax, displayWidth, displayHeight);
		nmhDisplay =	new Display
			(DATABACKGROUND, GRAPHBACKGROUND, "Time in ms", "n, m, h",
			 tbeg, tend, nmhMin, nmhMax, displayWidth, displayHeight);

		// Set constraints of the grid bag layout manager.
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;

		// Group components of panels
		voltagePanel.add(vmDisplay.getDisplay());

		currentPanel.setLayout(gbl);
		gbl.setConstraints(iDisplay.getDisplay(), gbc);
		currentPanel.add(iDisplay.getDisplay());
		gbl.setConstraints(iPanel, gbc);
		currentPanel.add(iPanel);

		paramPanel.setLayout(gbl);
		gbl.setConstraints(nmhDisplay.getDisplay(), gbc);
		paramPanel.add(nmhDisplay.getDisplay());
		gbl.setConstraints(nmhPanel, gbc);
		paramPanel.add(nmhPanel);

		// Layout graph displays
		setLayout(gbl);

		gbl.setConstraints(voltagePanelBox, gbc);
		add(voltagePanelBox);

		gbl.setConstraints(currentPanelBox, gbc);
		add(currentPanelBox);

		gbl.setConstraints(paramPanelBox, gbc);
		add(paramPanelBox);

		pack();
	}

	// A possible event the graph dialog window reacts is an action issued by
	// one of its graphic output control buttons (show / hide data sets)
	public void itemStateChanged(ItemEvent e) {
		Checkbox src = (Checkbox)e.getSource();

		if (src == imBox || src == inaBox || src == ikBox || 
				src == ilBox || src == icBox) {
			imDraw = imBox.getState();
			inaDraw = inaBox.getState();
			ikDraw = ikBox.getState();
			ilDraw = ilBox.getState();
			icDraw = icBox.getState();
			updateDisplays();
			iDisplay.resize();
			iDisplay.getDisplay().repaint();

		} else {		
			nDraw = nBox.getState();
			mDraw = mBox.getState();
			hDraw = hBox.getState();
			updateDisplays();
			nmhDisplay.resize();
			nmhDisplay.getDisplay().repaint();
		}
	}

	// Get the context of the graph dialog (displays, variables, graph handles
	// and the cell compartment numbers.
	public Display[] getDisplays() {
		return displays;
	}

	public int[] getVariables() {
		return variables;
	}

	public int[] getDataSetHandles() {
		return datasetHandles;
	}

	public int[] getCompartments() {
		return compartments;
	}

	// Adds a new data set to the display 'd'. Indicate the data set handle
	// to be added and the variable and the cell compartment number the graph
	// should be linked to.
	protected void addDataSet(Display d, int setHandle, int variable,
														int comp) {
		int k;
		Display[] tdispls = displays;
		int[] tdata = datasetHandles, tv = variables, tcomps = compartments;

		if (displays == null) k = 1;
		else k = displays.length + 1;

		displays = new Display[k];
		datasetHandles = new int[k];
		variables = new int[k];
		compartments = new int[k];
		k--;

		if (tdispls != null)
			for (k = 0; k < tdispls.length; k++) {
				displays[k] = tdispls[k];
				datasetHandles[k] = tdata[k];
				variables[k] = tv[k];
				compartments[k] = tcomps[k];
			}
		
		displays[k] = d;
		variables[k] = variable;
		datasetHandles[k] = setHandle;
		compartments[k] = comp;
	}

	// Decides whether a particular data set can be shown or should be hided.
	public void updateDisplays() {
		if (imDraw) displays[1].show(datasetHandles[1]); 
		else displays[1].hide(datasetHandles[1]);

		if (inaDraw) displays[2].show(datasetHandles[2]); 
		else displays[2].hide(datasetHandles[2]);

		if (ikDraw) displays[3].show(datasetHandles[3]); 
		else displays[3].hide(datasetHandles[3]);

		if (ilDraw) displays[4].show(datasetHandles[4]); 
		else displays[4].hide(datasetHandles[4]);

		if (icDraw) displays[5].show(datasetHandles[5]); 
		else displays[5].hide(datasetHandles[5]);

		if (nDraw) displays[6].show(datasetHandles[6]); 
		else displays[6].hide(datasetHandles[6]);

		if (mDraw) displays[7].show(datasetHandles[7]); 
		else displays[7].hide(datasetHandles[7]);

		if (hDraw) displays[8].show(datasetHandles[8]); 
		else displays[8].hide(datasetHandles[8]);
	}

	// Clear all data sets from the displays
	public void clearDisplays(double tend) {
		vmDisplay.removeAll();
		iDisplay.removeAll();
		nmhDisplay.removeAll();

		addDataSets();
		resizeDisplays(tend);
	}

	// Resize all displays according to a new time scale on the x-axis.
	public void resizeDisplays(double tend) {
		vmDisplay.newXRange(0, tend);
		iDisplay.newXRange(0, tend);
		nmhDisplay.newXRange(0, tend);

		vmDisplay.resize();
		iDisplay.resize();
		nmhDisplay.resize();
	}

	// Repaint all displays.
	public void repaintDisplays() {
		vmDisplay.getDisplay().repaint();
		iDisplay.getDisplay().repaint();
		nmhDisplay.getDisplay().repaint();
	}

	// Adds the initial data sets for the simulation
	public void addDataSets() {
		displays = null;
		variables = null;
		datasetHandles = null;

		addDataSet(vmDisplay, vmDisplay.addDataSet(Cvm), 0, 0);

		addDataSet(iDisplay, iDisplay.addDataSet(Cim), 1, 0);
		addDataSet(iDisplay, iDisplay.addDataSet(Cina), 2, 0);
		addDataSet(iDisplay, iDisplay.addDataSet(Cik), 3, 0);
		addDataSet(iDisplay, iDisplay.addDataSet(Cil), 4, 0);
		addDataSet(iDisplay, iDisplay.addDataSet(Cic), 5, 0);

		addDataSet(nmhDisplay, nmhDisplay.addDataSet(Cn), 6, 0);
		addDataSet(nmhDisplay, nmhDisplay.addDataSet(Cm), 7, 0);
		addDataSet(nmhDisplay, nmhDisplay.addDataSet(Ch), 8, 0);

		updateDisplays();
		repaintDisplays();
	}

	// Clones all data sets in all displays
	public void cloneAllDataSets() {
		int k;

		for(k = 0; k < displays.length; k++)
			displays[k].cloneDataSet(datasetHandles[k]);
	}

	// Determines and sets the resolution of the display graphs according to
	// the display dimensions.
	public double getDisplayRes(double tbeg, double tend) {
		Dimension appSize = getSize();
		displayWidth = (int)(appSize.width*0.9);
		displayHeight = (int)(appSize.width / 2.5);
		return (Math.round(10000*(tend-tbeg) / (double)displayWidth))/10000.0;
	}

	// Sets all buttons to enabled.
	public void setButtonsEnabled(boolean enabled) {
		imBox.setEnabled(enabled);
		inaBox.setEnabled(enabled);
		ikBox.setEnabled(enabled);
		ilBox.setEnabled(enabled);
		icBox.setEnabled(enabled);

		nBox.setEnabled(enabled);
		mBox.setEnabled(enabled);
		hBox.setEnabled(enabled);
	}
}		
