// -====-
// HHProp
// -====-

// Applet simulating 100 Hodgkin-Huxley cells and simulating the
// propagation speed of a spike.


// Author:			Pollinger Thomas
// Built:				March 6th, 1998


import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import gjt.*;
import graph.*;

import HHAxon;

public class HHProp extends Applet implements ItemListener, Notifiable{

	public class TextAdapter implements TextListener {
		public void textValueChanged(TextEvent event) {}
	}


	// Simulating a Hodgkin-Huxley cell
	private HHAxon hhcell;
	private Simulation sim;
	private boolean running = false;

	private	double gna = 120, gk = 36, gl = 0.3, cm = 1;
	private double ra = 0.03, diameter = 0.05;
	private	double nai = 70.96, nao = 460.0, ki = 301.4, ko = 10.0;
	private	double tbegStim = 0.0, tdurStim = 1.0; 
	private double tbeg = 0, tend = 7.0, tdt = 0.005;
	private double tres;
	private double inject_cur = 2.2, clamp_volt = 40.0;
	private boolean doing_space_clamp = true;
	private double speed = 0.0;

	private final int NBCELLS = 40;
	
	private final double cable_length = 0.1;
	private final double length = 0.1*NBCELLS;
	private final double hold_pot = -70.0;

	// Variables that set the initial graph window size (y-axis)
	private final double vmMin = hold_pot - 20, vmMax = 50;


	// COLOR / SHAPE DEFINITIONS
	// -------------------------

	private int displayHeight, displayWidth;

	private final Color DATABACKGROUND = new Color(180, 180, 180);
	private final Color GRAPHBACKGROUND = new Color(150,150,150);

	private final Color Cvm = new Color(0,50,0);
	private final double greenStep = 20;



	// MEMBERS FOR INPUT CONTROL
	// -------------------------

	// Length of text fields
	private final int LENGTH = 8;
	private final int TIMELENGTH = 5;
	private final int CELLLENGTH = 3;

	// Members related to parameter input
	private Label title = 
	new Label("Simulating Traveling Nerve Signals In a Smoothed Axon");

	// Grouping input field into three panels
	// The first represents all parameters charaterizing a HH cell.
	// Next are externally applied values (e.g. current or voltage).
	// The last panel holds all paramters related to a simulation.
	private Panel cellInputPanel = new Panel();
	private Panel cellInputPanel1 = new Panel();
	private Panel cellInputPanel2 = new Panel();
	private Panel cellInputPanel3 = new Panel();
	private Panel cellInputPanel4 = new Panel();
	private Panel extInputPanel = new Panel();
	private Panel extInputPanel1 = new Panel();
	private Panel extInputPanel2 = new Panel();
	private Panel simInputPanel = new Panel();
	private Panel simInputPanel1 = new Panel();
	private Panel simInputPanel2 = new Panel();

	// Frames
	private gjt.Box cellInputPanelBox = 
	new gjt.Box(cellInputPanel, "Cell Charateristics", Orientation.LEFT);
	private gjt.Box extInputPanelBox = 
	new gjt.Box(extInputPanel, "Applying External Current", 
							Orientation.LEFT);
	private gjt.Box simInputPanelBox = 
	new gjt.Box(simInputPanel, "Simulation Parameters", Orientation.LEFT);

	// Layout Managers
	private GridBagLayout gbl = new GridBagLayout();
	private GridBagConstraints gbc = new GridBagConstraints();

	private Separator sep = new Separator();

	// These are the text fields and labels for the input parameters
	private Label diameterLabel = new Label("Cell diameter: ");
	private Label cmLabel = new Label("C_m: ");
	private Label raLabel = new Label("R_a: ");

	private Label gnaLabel = new Label("G_Na: ");
	private Label gkLabel = new Label("G_K: ");
	private Label glLabel = new Label("G_l: ");
	private Label naiLabel = new Label("Na_i: ");
	private Label naoLabel = new Label("Na_o: ");
	private Label kiLabel = new Label("K_i: ");
	private Label koLabel = new Label("K_o: ");

	private Label currClampLabel = new Label("External current: ");
	private CheckboxGroup extGrp = new CheckboxGroup();
	private Checkbox spaceclmp = new Checkbox("Current clamp", true, extGrp);
	private Checkbox voltclmp = new Checkbox("Voltage clamp", false, extGrp);

	private Label tbegStimLabel = new Label("Begin Stimulation: ");
	private Label tdurStimLabel = new Label("Duration: ");

	private Label tbegLabel = new Label("Begin: ");
	private Label tendLabel = new Label("End: ");
	private Label tdtLabel = new Label("Time Step: ");
	private Label tresLabel = new Label("Time Scale Res: ");

	private TextField diameterField = new TextField(LENGTH);
	private TextField cmField = new TextField(LENGTH);
	private TextField raField = new TextField(LENGTH);

	private TextField gnaField = new TextField(LENGTH);
	private TextField gkField = new TextField(LENGTH);
	private TextField glField = new TextField(LENGTH);
	private TextField naiField = new TextField(LENGTH);
	private TextField naoField = new TextField(LENGTH);
	private TextField kiField = new TextField(LENGTH);
	private TextField koField = new TextField(LENGTH);

	private TextField extField = new TextField(LENGTH);

	private TextField tbegStimField = new TextField(TIMELENGTH);
	private TextField tdurStimField = new TextField(TIMELENGTH);

	private TextField tbegField = new TextField(TIMELENGTH);
	private TextField tendField = new TextField(TIMELENGTH);
	private TextField tdtField = new TextField(TIMELENGTH);
	private TextField tresField = new TextField(TIMELENGTH);

	// Buttons to control simulation
	private Button runButton = new Button("Run");
	private Button stopButton = new Button("Stop");
	private Button clearButton = new Button("Clear");
	private Button refreshButton = new Button("Refresh");

	// MEMBERS FOR GRAPHICAL OUTPUT CONTROL
	// ------------------------------------

	// Graph displays
	private Display vmDisplay;

	// Panels for cell number input query and speed output
	private Panel cellPanel = new Panel();
	private Panel speedPanel = new Panel();

	// Cell control fields
	private Button addCellButton = new Button("Add Cell: ");
	private Button removeCellButton = new Button("Remove Cell: ");
	private TextField cellNbField = new TextField(CELLLENGTH);

	// Speed output value
	private Label speedLabel = new Label("Simulated Spike Speed: ");
	private TextField speedField = new TextField(LENGTH);

	// Graph variable references
	private int cellNb = 0;
	private Display[] displays = new Display[NBCELLS];
	private int[] variables = new int[NBCELLS], 
		datasetHandles = new int[NBCELLS], compartments = new int[NBCELLS];
	

	// Public methods
	// --------------

	public void init() {

		// Initialize graph window references
		for (int k = 0; k < NBCELLS; k++) {
			displays[k] = null;
			datasetHandles[k] = 0;
		}

		// Evaluate default member values
		setDefDisplayParams(getPreferredSize());
		vmDisplay =	new Display
			(DATABACKGROUND, GRAPHBACKGROUND, "Time in ms", "V_{m} in mV",
			 tbeg, tend, vmMin, vmMax, displayWidth, displayHeight);

		// Setting up cell panels
		cellInputPanel1.add(diameterLabel);
		cellInputPanel1.add(diameterField);
		cellInputPanel1.add(cmLabel);
		cellInputPanel1.add(cmField);
		cellInputPanel1.add(raLabel);
		cellInputPanel1.add(raField);

		cellInputPanel2.add(gnaLabel);
		cellInputPanel2.add(gnaField);
		cellInputPanel2.add(gkLabel);
		cellInputPanel2.add(gkField);
		cellInputPanel2.add(glLabel);
		cellInputPanel2.add(glField);

		cellInputPanel3.add(naiLabel);
		cellInputPanel3.add(naiField);
		cellInputPanel3.add(naoLabel);
		cellInputPanel3.add(naoField);

		cellInputPanel4.add(kiLabel);
		cellInputPanel4.add(kiField);
		cellInputPanel4.add(koLabel);
		cellInputPanel4.add(koField);

		// Setting up ext panel
		spaceclmp.addItemListener(this);
		voltclmp.addItemListener(this);
		extInputPanel1.add(currClampLabel);
		extInputPanel1.add(extField);

		extInputPanel2.add(tbegStimLabel);
		extInputPanel2.add(tbegStimField);
		extInputPanel2.add(tdurStimLabel);
		extInputPanel2.add(tdurStimField);
		

		// Setting up sim panel
		simInputPanel1.add(tbegLabel);
		simInputPanel1.add(tbegField);
		simInputPanel1.add(tendLabel);
		simInputPanel1.add(tendField);
		simInputPanel2.add(tdtLabel);
		simInputPanel2.add(tdtField);
		simInputPanel2.add(tresLabel);
		simInputPanel2.add(tresField);

		// Setting default values
		diameterField.setText(Double.toString(diameter));
		cmField.setText(Double.toString(cm));
		raField.setText(Double.toString(ra));
		gnaField.setText(Double.toString(gna));
		gkField.setText(Double.toString(gk));
		glField.setText(Double.toString(gl));
		naiField.setText(Double.toString(nai));
		naoField.setText(Double.toString(nao));
		kiField.setText(Double.toString(ki));
		koField.setText(Double.toString(ko));

		extField.setText(Double.toString(inject_cur));
		tbegStimField.setText(Double.toString(tbegStim));
		tdurStimField.setText(Double.toString(tdurStim));
		
		tbegField.setText(Double.toString(tbeg));
		tendField.setText(Double.toString(tend));
		tdtField.setText(Double.toString(tdt));
		tresField.setText(Double.toString(tres));

		// Setting up panels for graphic output control
		cellNbField.setText(Integer.toString(cellNb));
		speedField.setText(Double.toString(speed));
		

		// Setting up layout manager for applet
		setLayout(gbl);
		
		// Title
		title.setFont(new Font("Times-Roman", Font.BOLD, 14));
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbl.setConstraints(title, gbc);
		add(title);

		// Insert separator
		gbc.fill  = GridBagConstraints.HORIZONTAL;
		gbc.insets = new Insets(0, 0, 10, 0);
		gbl.setConstraints(sep, gbc);
		add(sep);  

		// Cell input panel
		cellInputPanel.setLayout(new GridLayout(4,1));
		cellInputPanel.add(cellInputPanel1);
		cellInputPanel.add(cellInputPanel2);
		cellInputPanel.add(cellInputPanel3);
		cellInputPanel.add(cellInputPanel4);
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbl.setConstraints(cellInputPanelBox, gbc);
		add(cellInputPanelBox);

		// Ext input panel
		extInputPanel.setLayout(new GridLayout(2,1));
		extInputPanel.add(extInputPanel1);
		extInputPanel.add(extInputPanel2);
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbl.setConstraints(extInputPanelBox, gbc);
		add(extInputPanelBox);

		// Sim input panel
		simInputPanel.setLayout(new GridLayout(2,1));
		simInputPanel.add(simInputPanel1);
		simInputPanel.add(simInputPanel2);
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbl.setConstraints(simInputPanelBox, gbc);
		add(simInputPanelBox);

		// Setting up and adding buttons
		ButtonPanel butPanel = new ButtonPanel();
		stopButton.setEnabled(false);
		butPanel.add(runButton);
		butPanel.add(stopButton);
		butPanel.add(clearButton);
		butPanel.add(refreshButton);
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbl.setConstraints(butPanel, gbc);
		add(butPanel);

		// Setting up graphical displays
		DynGraph2D graph = vmDisplay.getDisplay();
		gbc.anchor = GridBagConstraints.CENTER;
		gbl.setConstraints(graph, gbc);
		add(graph);

		cellPanel.setLayout(new GridLayout(1,4));
		cellPanel.add(addCellButton);
		cellPanel.add(removeCellButton);
		cellPanel.add(cellNbField);
		gbc.anchor = GridBagConstraints.NORTH;
		gbl.setConstraints(cellPanel, gbc);
		// add(cellPanel); Not functional yet.

		speedPanel.setLayout(new GridLayout(1,2));
		speedField.setEditable(false);
		speedPanel.add(speedLabel);
		speedPanel.add(speedField);
		gbl.setConstraints(speedPanel, gbc);
		// add(speedPanel); Not functional yet.

		// Defining predifined shown compartments
		Color darkGreen = new Color(0, 100, 0);
		addCell(1, darkGreen);
		addCell(4, Color.green);
		addCell(36, Color.cyan);
		addCell(40, Color.blue);
		
		
		// Placing action listeners
		runButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				setAllEnabled(false);
				updateParameters();
				hhcell = new HHAxon(length, cable_length, diameter, tbegStim, tdurStim,
														inject_cur, clamp_volt, hold_pot, cm, ra,
														gna, gk, gl, nai, nao, ki, ko, doing_space_clamp);
				
				sim = new Simulation(hhcell, tend, tdt, tres, HHProp.this);
				sim.setPriority(Thread.MIN_PRIORITY);
				sim.setDisplays(displays, datasetHandles, variables, compartments);
				sim.start();
				running = true;
			}
		});
		
		stopButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				if (running) {
					sim.halt();
					notifyOnEnd();
				}
				running = false;
			}
		});

		clearButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				clearDisplays();
			}
		});

		refreshButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				repaintDisplays();
			}
		});
			
		addCellButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				try {
					cellNb = new Integer(cellNbField.getText()).intValue();
					addCell(cellNb, Color.black);
				} catch (Exception e) {}
			}
		});

		removeCellButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent event) {
				try {
					cellNb = new Integer(cellNbField.getText()).intValue();
					removeCell(cellNb);
				} catch (Exception e) {}
			}
		});


		tbegField.addTextListener(new TextAdapter() {
			public void textValueChanged(TextEvent event) {
				tbeg = new Double(tbegField.getText()).doubleValue();				
				setDefDisplayParams(getPreferredSize());
				resizeDisplays();
				repaintDisplays();
			}
		});

		tendField.addTextListener(new TextAdapter() {
			public void textValueChanged(TextEvent event) {
				tend = new Double(tendField.getText()).doubleValue();				
				setDefDisplayParams(getPreferredSize());
				resizeDisplays();
				repaintDisplays();
			}
		});
	}

	public void start() {}

	public void stop() {}

	public void destroy() {
		remove(cellInputPanel);
		remove(title);
	}

	// Notified if simulation ended
	public void notifyOnEnd() {
		setAllEnabled(true);
		cloneAllDataSets();
	}

	// Checkbox listener for space/voltage clamp
	public void itemStateChanged(ItemEvent event) {
		Checkbox cbox = (Checkbox)event.getSource();

		// Test whether current clamp
		if (cbox == spaceclmp) {
			doing_space_clamp = true;
			clamp_volt = new Double(extField.getText()).doubleValue();
			extField.setText(Double.toString(inject_cur));

			// Doing voltage clamp
		} else if (cbox == voltclmp) {
			doing_space_clamp = false;
			inject_cur = new Double(extField.getText()).doubleValue();
			extField.setText(Double.toString(clamp_volt));
		}
	}

	// Update entered parameters
	public void updateParameters() {
		diameter = new Double(diameterField.getText()).doubleValue();
		cm = new Double(cmField.getText()).doubleValue();
		ra = new Double(raField.getText()).doubleValue();
		gna = new Double(gnaField.getText()).doubleValue();
		gk = new Double(gkField.getText()).doubleValue();
		gl = new Double(glField.getText()).doubleValue();
		nai = new Double(naiField.getText()).doubleValue();
		nao = new Double(naoField.getText()).doubleValue();
		ki = new Double(kiField.getText()).doubleValue();
		ko = new Double(koField.getText()).doubleValue();

		if (doing_space_clamp) 
			inject_cur = new Double(extField.getText()).doubleValue();
		else clamp_volt = new Double(extField.getText()).doubleValue();
		tdurStim = new Double(tdurStimField.getText()).doubleValue();
		tbegStim = new Double(tbegStimField.getText()).doubleValue();

		tbeg = new Double(tbegField.getText()).doubleValue();
		tend = new Double(tendField.getText()).doubleValue();
		tdt = new Double(tdtField.getText()).doubleValue();
		tres = new Double(tresField.getText()).doubleValue();
	}

	// Protected members
	// -----------------

	// Clones all data sets in all displays
	protected void cloneAllDataSets() {
		int k;

		for(k = 0; k < displays.length; k++)
			if (displays[k] != null)
				displays[k].cloneDataSet(datasetHandles[k]);
	}

	// Add data set with display d, set ID setHandle, the variable and the
	// compartment number. The data set will only be prepared but will not be
	// shown yet.
	protected void addDataSet(Display d, int setHandle, int variable, int comp) {
		int k;

		displays[comp] = d;
		datasetHandles[comp] = setHandle;
		variables[comp] = variable;
		compartments[comp] = comp;
	}

	// Add a new compartment to be shown on the graph.
	protected void addCell(int cell, Color clr) {
		cell--;
		if (cell >= NBCELLS) return;
		if (displays[cell] != null) return;
		addDataSet(vmDisplay, vmDisplay.addDataSet(clr), 0, cell);
		resizeDisplays();
		repaintDisplays();
	}

	// Removes an added compartment from the graph. 
	protected void removeCell(int cell) {
		if (cell >= NBCELLS) return;
		if (displays[cell] == null) return;
		vmDisplay.removeDataSet(datasetHandles[cell]);
		displays[cell] = null;
		datasetHandles[cell] = 0;
		resizeDisplays();
		repaintDisplays();
	}

	protected void setDefDisplayParams(Dimension size) {
		Dimension appSize = getSize();
		displayWidth = (int)(appSize.width*0.9);
		displayHeight = (int)(appSize.width / 2.5);
		tres = Math.round(10000*(tend-tbeg) / (double)displayWidth)/10000.0;
		tresField.setText(Double.toString(tres));
	}

	protected void clearDisplays() {
		vmDisplay.clearAll();
		resizeDisplays();
		repaintDisplays();
	}

	protected void resizeDisplays() {
		vmDisplay.newXRange(tbeg, tend);
		vmDisplay.resize();
	}

	protected void repaintDisplays() {
		vmDisplay.getDisplay().repaint();
	}

	protected void setAllEnabled(boolean enabled) {
		diameterLabel.setEnabled(enabled);
		cmLabel.setEnabled(enabled);
		raLabel.setEnabled(enabled);
		
		gnaLabel.setEnabled(enabled);
		gkLabel.setEnabled(enabled);
		glLabel.setEnabled(enabled);
		naiLabel.setEnabled(enabled);
		naoLabel.setEnabled(enabled);
		kiLabel.setEnabled(enabled);
		koLabel.setEnabled(enabled);

		spaceclmp.setEnabled(enabled);
		voltclmp.setEnabled(enabled);

		tdurStimLabel.setEnabled(enabled);
		tbegStimLabel.setEnabled(enabled);
	
		tbegLabel.setEnabled(enabled);
		tendLabel.setEnabled(enabled);
		tdtLabel.setEnabled(enabled);
		tresLabel.setEnabled(enabled);
 
		diameterField.setEnabled(enabled);
		cmField.setEnabled(enabled);
		raField.setEnabled(enabled);

		gnaField.setEnabled(enabled);
		gkField.setEnabled(enabled);
		glField.setEnabled(enabled);
		naiField.setEnabled(enabled);
		naoField.setEnabled(enabled);
		kiField.setEnabled(enabled);
		koField.setEnabled(enabled);

		extField.setEnabled(enabled);
		tdurStimField.setEnabled(enabled);
		tbegStimField.setEnabled(enabled);

		tbegField.setEnabled(enabled);
		tendField.setEnabled(enabled);
		tdtField.setEnabled(enabled);
		tresField.setEnabled(enabled);

		runButton.setEnabled(enabled);
		stopButton.setEnabled(!enabled);
		clearButton.setEnabled(enabled);

		addCellButton.setEnabled(enabled);
		removeCellButton.setEnabled(enabled);
	}
}
