// -========-
// RungeKutta
// -========-

// This is the Runge-Kutta algorithm of 4th order to 
// solve ordinary differential equations like dx/dt = f(t, x).

// The algorithme scheme is as follows (for the i-th time step):
// k1 = dt * f(x_i)
// k2 = dt * f(x_i + k1/2)
// k3 = dt * f(x_i + k2/2)
// k4 = dt * f(x_i + k3)

// x_{i+1} = x_i + 1/6*(k1 + 2*k2 + 2*k3 + k4) + O(dt^5)


// Written by Pollinger Thomas
// Built:				Feb 26th, 1998


public class RungeKutta extends BaseOde {

	protected OdeFct[] k1, k2;

	public RungeKutta() {
		k1 = null;
		k2 = null;
	}
	
	public RungeKutta(final RungeKutta rk) {
		super(rk);

		k1 = ArrayDup.dup(rk.k1);
		k2 = ArrayDup.dup(rk.k2);
	}

	public void init(OdeFct x) {
		super.init(x);
		k1 = ArrayDup.dup(xv);
		k2 = ArrayDup.dup(xv);
	}

	public void init(OdeFct[] x) {
		super.init(x);
		k1 = ArrayDup.dup(xv);
		k2 = ArrayDup.dup(xv);
	}

	public void eval(double dt) {
		int k;

		swap(xv, orig_x);
		
		for (k = 0; k < xv.length; k++)
			eval_k1(dt, xv[k], orig_x[k], k1[k]);
		
		for (k = 0; k < xv.length; k++)
			eval_k2(dt, xv[k], orig_x[k], k1[k], k2[k]);
		
		for (k = 0; k < xv.length; k++)
			eval_k3(dt, xv[k], orig_x[k], k2[k], k1[k]);
		
		for (k = 0; k < xv.length; k++)
			eval_k4(dt, xv[k], orig_x[k], k1[k]);
		
		for (k = 0; k < xv.length; k++)
			xv[k].pass_values(orig_x[k]);
	}

	public void eval(OdeFct x, double dt) {
		init(x);
		eval(dt);
	}

	public void eval(OdeFct[] x, double dt) {
		init(x);
		eval(dt);
	}

	public Object clone() {
		return new RungeKutta(this);
	}

	// Private members
	// ---------------
	
	private final void eval_k1(double dt, OdeFct oxv, OdeFct oox, OdeFct ok1) {
		double w;
		
		oox.update_ode_state(oxv);
		ok1.update_ode_state(oxv);
		
		oox.init_for_f();
		oox.init_for_ong();
		oxv.init_for_set();
		ok1.init_for_set();
		
		oxv.first(); oox.first(); ok1.first();
		do {
			w = dt*oox.f();
			oxv.set(oox.ong() + w/6);
			ok1.set(oox.ong() + w/2);
		} while (oox.next() & oxv.next() & ok1.next());
		
		oox.end_f();
		oox.end_ong();
		oxv.end_set();
		ok1.end_set();
	}
	
	private final void eval_k2
	(double dt, OdeFct oxv, OdeFct oox, OdeFct ok1, OdeFct ok2) {
		double w;
		
		ok2.update_ode_state(oxv);
		
		oox.init_for_ong();
		oxv.init_for_ong();
		oxv.init_for_set();
		ok1.init_for_f();
		ok2.init_for_set();
		
		oox.first(); oxv.first(); ok1.first(); ok2.first();
		do {
			w = dt*ok1.f();
			oxv.set(oxv.ong() + w/3);
			ok2.set(oox.ong() + w/2);
		} while (oox.next() & oxv.next() & ok1.next() & ok2.next());
		
		oox.end_ong();
		oxv.end_ong();
		oxv.end_set();
		ok1.end_f();
		ok2.end_set();
	}
	
	private final void eval_k3
	(double dt, OdeFct oxv, OdeFct oox, OdeFct ok2, OdeFct ok3) {
		double w;
		
		oox.init_for_ong();
		oxv.init_for_ong();
		oxv.init_for_set();
		ok2.init_for_f();
		ok3.init_for_set();
		
		oox.first(); oxv.first(); ok2.first(); ok3.first();
		do {
			w = dt*ok2.f();
			oxv.set(oxv.ong() + w/3);
			ok3.set(oox.ong() + w);
		} while (oox.next() & oxv.next() & ok2.next() & ok3.next());
		
		oox.end_ong();
		oxv.end_ong();
		oxv.end_set();
		ok2.end_f();
		ok3.end_set();
	}

	private final void eval_k4(double dt, OdeFct oxv, OdeFct oox, OdeFct ok3) {
		ok3.init_for_f();
		oxv.init_for_ong();
		oxv.init_for_set();
		
		ok3.first(); oxv.first();
		do {
			oxv.set(oxv.ong() + dt*ok3.f()/6);
		} while (oxv.next() & ok3.next());
		
		ok3.end_f();
		oxv.end_ong();
		oxv.end_set();
	}
}
