/**
 * This canvas is used to display a linear separability graph for a
 * 2 input perceptron.
 *
 * @author  Fred Corbett
 * @version February 5, 1997
 */

import java.awt.*;

class PatternSpaceCanvas extends Canvas {


  /*******************/
  /* Class Variables */
  /*******************/

  /**
   * Equality epsilon; used for floating-point equality testing.
   */

  public final static float EE = 1e-10f;


  /**
   * X offset of clip rectangle.
   */

  private static final int crXOff = 20;

  /**
   * Y offset of clip rectangle.
   */

  private static final int crYOff = 20;

  /**
   * X offset of points from CR.
   */

  private static final int xOff = 30; //50 - crXOff;

  /**
   * Y offset of points from CR.
   */

  private static final int yOff = 30; //50 - crYOff;

  /**
   * Radius of points.
   */

  private static final int r = 5;

  /**
   * Diameter of points.
   */

  private static final int d = r * 2;


  /**********************/
  /* Instance Variables */
  /**********************/

  /**
   * Reference to simple perceptron applet.
   */

  private PerceptronApplet pa;

  /**
   * Coordinates of the graph line.
   */

  private int x1, x2, y1, y2;

  /**
   * Width of the canvas in pixels.
   */

  private int width = 200;

  /**
   * Height of the canvas in pixels.
   */

  private int height = 200;

  /**
   * Bounding coordinates of the 1 x 1 rectangle (0,0),(0,1),(1,1),(0,1).
   */

  private int xMin, xMax, yMin, yMax;

  /**
   * Length and width of 1 x 1 rectangle.
   */

  private int xRange, yRange;

  /**
   * Bounding coordinates of the clip rectangle.
   */

  private int crXMin, crXMax, crYMin, crYMax;


  /**
   * Length and width of clip rectangle.
   */

  private int crXRange, crYRange;

  /**
   * Offscreen image for buffered animation.
   */

  private Image offImage;

  /**
   * Graphics context of offscreen image.
   */

  private Graphics offGraphics;


  /***********************/
  /* Constructor Methods */
  /***********************/

  /**
   * Construct a new pattern space canvas and initialize it.
   * @param PerceptronApplet pa - reference to parent applet.
   * @param int w - width of the canvas.
   * @param int h - height of the canvas.
   */

  public PatternSpaceCanvas(PerceptronApplet pa, int w, int h) {
    super();
    this.pa = pa;
    width = w;
    height = h;
    init();
  }


  /******************/
  /* Public Methods */
  /******************/

  /**
   * Overidden to allow for double-buffering.
   * @param Graphics g - the canvas graphics context.
   */

  public void paint(Graphics g) {
    update(g);
  }

  /**
   * Update the canvas graph using double-buffering.
   * @param Graphics g - the canvas graphics context.
   */

  public void update(Graphics g) {
    if (offGraphics == null) {
      offImage = createImage(width,height);
      offGraphics = offImage.getGraphics();
    }

    // Erase the previous image and reset the drawing colour
    offGraphics.setColor(getBackground());
    offGraphics.fillRect(0, 0, width, height);
    offGraphics.setColor(Color.black);

    // Add the clip rectangle
    offGraphics.clipRect(crXMin,crYMin,crXRange,crYRange);

    // Draw vertical axis and label
    pa.drawArrow(offGraphics,xMin,crYMax-5,xMin,crYMin+5);
    offGraphics.drawString("x2", xMin-20, crYMin+20);

    // Draw horizontal axis and label
    pa.drawArrow(offGraphics,crXMin+5,yMax,crXMax-5,yMax);
    offGraphics.drawString("x1", crXMax-20, yMax+20);

    // Draw the 4 points (green if on, red if off)
    if (pa.outputs[0][0] > 0) {
      offGraphics.setColor(Color.green);
      offGraphics.fillOval(xMin-r,yMax-r,d,d);
    } else {
      offGraphics.setColor(Color.red);
      offGraphics.fillOval(xMin-r,yMax-r,d,d);
    }
    if (pa.outputs[1][0] > 0) {
      offGraphics.setColor(Color.green);
      offGraphics.fillOval(xMax-r,yMax-r,d,d);
    } else {
      offGraphics.setColor(Color.red);
      offGraphics.fillOval(xMax-r,yMax-r,d,d);
    }
    if (pa.outputs[2][0] > 0) {
      offGraphics.setColor(Color.green);
      offGraphics.fillOval(xMin-r,yMin-r,d,d);
    } else {
      offGraphics.setColor(Color.red);
      offGraphics.fillOval(xMin-r,yMin-r,d,d);
    }
    if (pa.outputs[3][0] > 0) {
      offGraphics.setColor(Color.green);
      offGraphics.fillOval(xMax-r,yMin-r,d,d);
    } else {
      offGraphics.setColor(Color.red);
      offGraphics.fillOval(xMax-r,yMin-r,d,d);
    }

    // Draw the new line(s)
    drawLine(offGraphics);
    g.drawImage(offImage,0,0,this);
  }

  /**
   * This method allows a class to update the graph without having a
   * reference to the canvas graphics context.
   * @param float w0 - threshold weight
   * @param float w1 - weight 1
   * @param float w2 - weight 2
   * @param float x0 - threshold input
   */

  public void updateLine(float w0, float w1, float w2, float x0) {
    calculateLineCoords(w0,w1,w2,x0);
    update(getGraphics());
  }


  /*******************/
  /* Private Methods */
  /*******************/

  /**
   * Initialize the drawing/canvas parameters.
   */

  private void init() {
    crXMin = crXOff;
    crXMax = width - crXOff;
    crYMin = crYOff;
    crYMax = height - crYOff;
    crXRange = crXMax - crXMin;
    crYRange = crYMax - crYMin;

    xMin = crXMin + xOff;
    xMax = crXMax - xOff;
    yMin = crYMin + yOff;
    yMax = crYMax - yOff;
    xRange = xMax - xMin;
    yRange = yMax - yMin;

    setSize(width,height);
  }

  /**
   * Draw the line separating the two classes in the graph.
   * @param Graphics g - the canvas graphics context.
   */

  private void drawLine(Graphics g) {
    g.setColor(Color.blue);
    g.drawLine(x1,y1,x2,y2);
  }

  /**
   * Calculate the new line coodinates using the current state of the
   * Perceptron held by the SimplePerceptron applet.
   * @param float w0 - threshold weight
   * @param float w1 - weight 1
   * @param float w2 - weight 2
   * @param float x0 - threshold input
   */

  private void calculateLineCoords(float w0, float w1, float w2, float x0) {
    // Check for a vertical line
    if (Math.abs(w2) < EE) {
      try {
        x1 = convertX((w0 * x0) / w1);
      }
      catch(ArithmeticException e) {}
      x2 = x1;
      y1 = 0; y2 = height;
    } else {
      // Calulate y (x2) value at x1 points on canvas boundary
      x1 = convertX(-1f);
      x2 = convertX(2f);
      y1 = convertY((-(w0 * x0 + w1 * -1f)) / w2);
      y2 = convertY((-(w0 * x0 + w1 * 2f)) / w2);
    }
  }

  // Currently these only work with the specific dimensions used
  // and would probably be better inline

  /**
   * Convert a floating point x coordinate in the Cartesian plane to an
   * integer value specific to the canvas size.
   * @param float n - the Cartesian x coordinate.
   * @return int - the integer equivalent.
   */

  private int convertX(float n) {
    return (int) ((n * xRange) + crXOff + xOff);
  }

  /**
   * Convert a floating point y coordinate in the Cartesian plane to an
   * integer value specific to the canvas size.
   * @param float n - the Cartesian y coordinate.
   * @return int - the integer equivalent.
   */

  private int convertY(float n) {
   return (int) ((n * -yRange) + height - (crYOff + yOff));
  }

} // End of PatternSpaceCanvas Class
