/**
 * This canvas is used to display a 3D separability graph for a 3 input
 * perceptron. Input 3 is the logical AND of inputs 1 and 2. Many of the
 * routines are modified versions of those used in the Sun Microsystems
 * sample applets: Model3D and XYZApplet.
 *
 * @author  Fred Corbett
 * @version February 4, 1996
 */

import java.applet.*;
import java.awt.*;
import java.awt.event.*;

class PatternSpaceCanvas3D extends Canvas implements Runnable, MouseListener, MouseMotionListener {


    /**********************/
    /* Instance Variables */
    /**********************/

    /**
     * Reference to simple perceptron applet.
     */

    private PerceptronApplet pa;

    /**
     * Width of the canvas in pixels.
     */

    private int width = 200;

    /**
     * Height of the canvas in pixels.
     */

    private int height = 200;
    /**
     * Offscreen image for buffered animation.
     */

    private Image offImage;

    /**
     * Graphics context of offscreen image.
     */

    private Graphics offGraphics;

    /**
     * Can the models be rotated with the mouse.
     */

    private boolean rotationEnabled = true;


    // Variables for 3D Models

    /**
     * 3D model for points (spheres) in coordinate system.
     */

    SphereModel pointsModel;

    /**
     * Flag for points painted to screen.
     */

    boolean paintedPoints = true;

    /**
     *
     */

    float xfacPoints;

    /**
     *
     */

    Matrix3D amatPoints = new Matrix3D();

    /**
     * 3D model for coordinate axes.
     */

    WireFrameModel axesModel;

    /**
     *
     */

    boolean paintedAxes = true;

    /**
     *
     */

    float xfacAxes;

    /**
     *
     */

    Matrix3D amatAxes = new Matrix3D();

    /**
     * 3D model for hyperplane.
     */

    WireFrameModel planeModel;

    /**
     *
     */

    boolean paintedPlane = true;

    /**
     *
     */

    float xfacPlane;

    /**
     *
     */

    Matrix3D amatPlane = new Matrix3D();


    // Common parameters to all models

    /**
     * Matrix for 3D transformations. Shared by all models.
     */

    private Matrix3D tmat = new Matrix3D();

    /**
     * Previous x-coordinate of mouse pointer.
     */

    private int prevx;

    /**
     * previous y-coordinate of mouse pointer.
     */

    private int prevy;

    /**
     * Scale fudge factor.
     */

    private float scalefudge = 1;


    /**
     * The number of milliseconds that should bound a double click.
     */

    private static long mouseDoubleClickThreshold	= 300L;

    /**
     * The previous mouseUp event's time.
     */

    private long mouseUpClickTime = 0;

    /**
     *
     */

    private float[][] planeCoords =
    {{0f,-0.5f,-0.5f},{0f,-0.5f,1.5f},{0f,1.5f,-0.5f},{0f,1.5f,1.5f}};

    /**
     *
     */

    private static final float[][] TEST_COORDS =
    {{-0.5f,-0.5f},{-0.5f,1.5f},{1.5f,-0.5f},{1.5f,1.5f}};

    /**
     *
     */

    private static final int NUM_PLANE_COORDS = 4;

    /**
     *
     */

    private static final int NUM_TEST_COORDS = 4;


    /***********************/
    /* Constructor Methods */
    /***********************/

    /**
     * Construct a new pattern space canvas and initialize it.
     * @param PerceptronApplet pa - reference to parent applet.
     * @param int w - width of the canvas.
     * @param int h - height of the canvas.
     */

    public PatternSpaceCanvas3D(PerceptronApplet pa, int w, int h) {
	super();
	this.pa = pa;
	this.width = w;
	this.height = h;
	this.init();
    }


    /****************************/
    /* Accessor/Mutator Methods */
    /****************************/


    /**
     *
     * @return boolean -
     */


    public boolean getRotationEnabled() {
	return rotationEnabled;
    }

    /**
     *
     * @param boolean b -
     */

    public void setRotationEnabled(boolean b) {
	rotationEnabled = b;
    }


    /******************/
    /* Public Methods */
    /******************/

    /**
     * Initialize the 3D pattern space canvas.
     */

    public void init() {
	this.addMouseListener(this);
	this.addMouseMotionListener(this);
	setSize(width,height);
    }

    /**
     * Run the execution thread for the canvas.
     */

    public void run() {
	Thread.currentThread().setPriority(Thread.MIN_PRIORITY);

	// Wireframe model for the 3D coordinates axes
	WireFrameModel axes = new WireFrameModel(Color.gray);

	axes.addVert(0,0,0);
	axes.addVert(1.5f,0,0);
	axes.addVert(0,1.5f,0);
	axes.addVert(0,0,1.5f);
	axes.addVert(-0.5f,0,0);
	axes.addVert(0,-0.5f,0);
	axes.addVert(0,0,-0.5f);

	axes.addLine(0,1);
	axes.addLine(0,2);
	axes.addLine(0,3);
	axes.addLine(0,4);
	axes.addLine(0,5);
	axes.addLine(0,6);

	axes.addVert(-0.5f,-0.5f,-0.5f);
	axes.addVert(-0.5f,-0.5f,1.5f);
	axes.addVert(-0.5f,1.5f,-0.5f);
	axes.addVert(-0.5f,1.5f,1.5f);
	axes.addVert(1.5f,-0.5f,-0.5f);
	axes.addVert(1.5f,-0.5f,1.5f);
	axes.addVert(1.5f,1.5f,-0.5f);
	axes.addVert(1.5f,1.5f,1.5f);

	axes.addLine(7,8);
	axes.addLine(7,9);
	axes.addLine(7,11);
	axes.addLine(9,10);
	axes.addLine(8,10);
	axes.addLine(10,14);
	axes.addLine(8,12);
	axes.addLine(11,12);
	axes.addLine(12,14);
	axes.addLine(9,13);
	axes.addLine(11,13);
	axes.addLine(13,14);

	axesModel = axes;

	axes.findBB();
	axes.compress();
	float xw = axes.xmax - axes.xmin;
	float yw = axes.ymax - axes.ymin;
	float zw = axes.zmax - axes.zmin;
	if (yw > xw)
	    xw = yw;
	if (zw > xw)
	    xw = zw;
	float f1 = getSize().width / xw;
	float f2 = getSize().height / xw;
	xfacAxes = 0.7f * (f1 < f2 ? f1 : f2) * scalefudge;

	// Wireframe model for the 3D hyperplane
	WireFrameModel plane = new WireFrameModel(Color.blue);

	buildPlane(plane);
	planeModel = plane;
	plane.findBB();
	plane.compress();

	/*
	  xw = plane.xmax - plane.xmin;
	  yw = plane.ymax - plane.ymin;
	  zw = plane.zmax - plane.zmin;
	  if (yw > xw)
	  xw = yw;
	  if (zw > xw)
	  xw = zw;
	  f1 = size().width / xw;
	  f2 = size().height / xw;
	*/
	xfacPlane = 0.7f * (f1 < f2 ? f1 : f2) * scalefudge;

	// Build the Sphere model
	SphereModel points = new SphereModel();
	if (pa.outputs[0][0] > 0)
	    points.addVert("green",0,0,0);
	else
	    points.addVert("red",0,0,0);
	if (pa.outputs[1][0] > 0)
	    points.addVert("green",1,0,0);
	else
	    points.addVert("red",1,0,0);
	if (pa.outputs[2][0] > 0)
	    points.addVert("green",0,1,0);
	else
	    points.addVert("red",0,1,0);
	if (pa.outputs[3][0] > 0)
	    points.addVert("green",1,1,1);
	else
	    points.addVert("red",1,1,1);
	Sphere.setApplet(((Applet) pa));
	points.findBB();
	pointsModel = points;
	xfacPoints = 0.7f * (f1 < f2 ? f1 : f2) * scalefudge;

	repaint();
    }

    /**
     * Start a new thread to initialize models and to allow for rotation
     * of the 3D model.
     */

    public void start() {
	if ((pointsModel == null) || (axesModel == null) || (planeModel == null))
	    new Thread(this).start();
    }

    /**
     * Stop the thread.
     */
    public void stop() {}

    public void mouseDragged(MouseEvent e) {
	if (rotationEnabled) {
	    tmat.unit();
	    float xtheta = Math.round((prevy - e.getY()) * 360.0f / getSize().width);
	    float ytheta = Math.round((e.getX() - prevx) * 360.0f / getSize().height);
	    tmat.xrot(xtheta);
	    tmat.yrot(ytheta);
	    
	    // Transform the models and repaint.
	    amatPoints.mult(tmat);
	    amatAxes.mult(tmat);
	    amatPlane.mult(tmat);
	    
	    if (paintedPoints || paintedAxes || paintedPlane) {
		paintedPoints = false;
		paintedAxes = false;
		paintedPlane = false;
		repaint();
	    }
	    prevx = e.getX();
	    prevy = e.getY();
	}
    
    }
    public void mouseMoved(MouseEvent e) {}
    public void mouseClicked(MouseEvent e) {
	prevx = e.getX();
	prevy = e.getY();
    }
    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void mousePressed(MouseEvent e) {}               
    public void mouseReleased(MouseEvent e) {	
	long eventTime = e.getWhen();
	long timeDiff = eventTime - mouseUpClickTime;
	if (timeDiff < 0)
	    timeDiff = -timeDiff;
	
	// Did this click happen with the given time bound?
	if (mouseDoubleClickThreshold > timeDiff) {
	    mouseUpClickTime = 0;
	    resetRotation();
	} else {
	    mouseUpClickTime = eventTime;
	}
    }

    /**
     * Reset the 3D graph to the original state; set each model matrix to
     * the unit matrix and repaint the canvas.
     */

    public void resetRotation() {
	amatPoints.unit();
	amatAxes.unit();
	amatPlane.unit();

	if (paintedPoints || paintedAxes || paintedPlane) {
	    paintedPoints = false;
	    paintedAxes = false;
	    paintedPlane = false;
	    repaint();
	}
    }


    /**
     * Overidden to allow for double-buffering.
     * @param Graphics g - the canvas graphics context.
     */

    public void paint(Graphics g) {
	update(g);
    }


    /**
     * Update the canvas graph using double-buffering.
     * @param Graphics g - the canvas graphics context.
     */

    public void update(Graphics g) {
	if (offGraphics == null) {
	    offImage = createImage(width,height);
	    offGraphics = offImage.getGraphics();
	}

	// Erase the previous image and reset the drawing colour
	offGraphics.setColor(getBackground());
	offGraphics.fillRect(0, 0, width, height);

	// Wireframe paint
	if (axesModel != null) {
	    axesModel.mat.unit();
	    axesModel.mat.translate(-(axesModel.xmin + axesModel.xmax) / 2,
				    -(axesModel.ymin + axesModel.ymax) / 2,
				    -(axesModel.zmin + axesModel.zmax) / 2);
	    axesModel.mat.mult(amatAxes);
	    axesModel.mat.scale(xfacAxes, -xfacAxes, 16 * xfacAxes / getSize().width);
	    axesModel.mat.translate(getSize().width / 2, getSize().height / 2, 8);
	    axesModel.transformed = false;
	    axesModel.paint(offGraphics);
	}

	// Sphere paint
	if (pointsModel != null) {
	    pointsModel.mat.unit();
	    pointsModel.mat.translate(-(pointsModel.xmin + pointsModel.xmax) / 2,
				      -(pointsModel.ymin + pointsModel.ymax) / 2,
				      -(pointsModel.zmin + pointsModel.zmax) / 2);
	    pointsModel.mat.mult(amatPoints);
	    pointsModel.mat.scale(xfacPoints, -xfacPoints, 16 * xfacPoints / getSize().width);
	    pointsModel.mat.translate(getSize().width / 2, getSize().height / 2, 8);
	    pointsModel.transformed = false;
	    pointsModel.paint(offGraphics);
	}
	// Plane paint
	if (planeModel != null) {
	    planeModel.mat.unit();
	    planeModel.mat.translate(-(planeModel.xmin + planeModel.xmax) / 2,
				     -(planeModel.ymin + planeModel.ymax) / 2,
				     -(planeModel.zmin + planeModel.zmax) / 2);
	    planeModel.mat.mult(amatPlane);
	    planeModel.mat.scale(xfacPlane, -xfacPlane, 16 * xfacPlane / getSize().width);
	    planeModel.mat.translate(getSize().width / 2, getSize().height / 2, 8);
	    planeModel.transformed = false;
	    planeModel.paint(offGraphics);
	}

	setPainted();
	g.drawImage(offImage,0,0,this);
    }


    /**
     * Change the colour of the sphere with a specfied index.
     * @param index - index of sphere in sphere array.
     * @param colour - string representing new colour.
     */

    public void changeSphereColour(int index, String colour) {
	if (pointsModel == null)
	    new Thread(this).start();
	else {
	    Sphere a = (Sphere) pointsModel.sphereTable.get(colour.toLowerCase());
	    if (a == null) a = pointsModel.defaultSphere;
	    pointsModel.spheres[index] = a;
	}
	paint(getGraphics());
    }

    /**
     * This method allows a class to update the graph without having a
     * reference to the canvas graphics context.
     * @param float w0 - threshold weight
     * @param float w1 - weight 1
     * @param float w2 - weight 2
     * @param float w3 - weight 3
     * @param float x0 - threshold input
     */

    public void updatePlane(float w0, float w1, float w2, float w3, float x0) {
	calculatePlaneCoords(w0,w1,w2,w3,x0);
	buildPlane(planeModel);
	update(getGraphics());
    }

    /*******************/
    /* Private Methods */
    /*******************/


    /**
     * Set both models painted flags to true.
     */

    private synchronized void setPainted() {
	paintedPoints = true;
	paintedAxes = true;
	paintedPlane = true;
	notifyAll();
    }

    /**
     * Wait unit both models have been painted.
     */

    private synchronized void waitPainted() {
	while (!(paintedPoints && paintedAxes && paintedPlane)) {
	    try {
		wait();
	    }
	    catch (InterruptedException e) {}
	}
	paintedPoints = false;
	paintedAxes = false;
	paintedPlane = false;
    }


    /**
     * Calculate the new plane coodinates using the current state of the
     * Perceptron held by the SimplePerceptron applet.
     * NOTE: NOT WORKING YET!
     * @param float w0 - threshold weight
     * @param float w1 - weight 1
     * @param float w2 - weight 2
     * @param float w3 - weight 3
     * @param float x0 - threshold input
     */

    private void calculatePlaneCoords(float w0, float w1, float w2, float w3, float x0) {
	int i;
	int c = 0;
	float x1 = 0f;
	float x2 = 0f;
	float x3 = 0f;

	// Check x1 coordinate
	for (i=0; i < NUM_TEST_COORDS; i++) {
	    x1 = ( -(w2 * TEST_COORDS[i][0]) - (w3 * TEST_COORDS[i][1] ) + x0 ) / w1;
	    if ( (x1 <= 1.5f) && (x1 >= -0.5f) ) {
		planeCoords[c][0] = x1;
		planeCoords[c][1] = TEST_COORDS[i][0];
		planeCoords[c][2] = TEST_COORDS[i][1];
		c++;
		if (c > 3) return;
	    }
	}

	// Check x2 coordinate
	for (i=0; i < NUM_TEST_COORDS; i++) {
	    x2 = ( -(w1 * TEST_COORDS[i][0]) - (w3 * TEST_COORDS[i][1] ) + x0 ) / w2;
	    if ( (x2 <= 1.5f) && (x2 >= -0.5f) ) {
		planeCoords[c][0] = TEST_COORDS[i][0];
		planeCoords[c][1] = x2;
		planeCoords[c][2] = TEST_COORDS[i][1];
		c++;
		if (c > 3) return;
	    }
	}


	// Check x3 coordinate
	for (i=0; i < NUM_TEST_COORDS; i++) {
	    x3 = ( -(w1 * TEST_COORDS[i][0]) - (w2 * TEST_COORDS[i][1] ) + x0 ) / w3;
	    if ( (x1 <= 1.5f) && (x1 >= -0.5f) ) {
		planeCoords[c][0] = TEST_COORDS[i][0];
		planeCoords[c][1] = TEST_COORDS[i][1];
		planeCoords[c][2] = x3;
		c++;
		if (c > 3) return;
	    }
	}

    }


    private void buildPlane(WireFrameModel pm) {
	pm.removeAll();

	for (int i=0; i < NUM_PLANE_COORDS; i++) {
	    pm.addVert(planeCoords[i][0], planeCoords[i][1], planeCoords[i][2]);
	    //System.out.println(planeCoords[i][0] + " " + planeCoords[i][1] + " " + planeCoords[i][2]);
	}

	// Connect the plane vertices
	pm.addLine(0,1);
	pm.addLine(0,2);
	pm.addLine(1,3);
	pm.addLine(2,3);
    }


} // End of PatternSpaceCanvas3D
