/**
 * This applet allows a user to experiment with a single perceptron...
 *
 * @author  Fred Corbett
 * @version March 9, 1997
 */

import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import anns.*;

public class PerceptronApplet extends SimpleNeuronApplet {


    /*******************/
    /* Class Variables */
    /*******************/

    /**
     * Normal input vectors.
     */

    public static final float[][] REG_INPUTS =
    {{0f,0f,0f}, {1f,0f,0f}, {0f,1f,0f}, {1f,1f,0f}};

    /**
     * AND input vectors (input3 = input2 AND input 1).
     */

    public static final float[][] AND_INPUTS =
    {{0f,0f,0f}, {1f,0f,0f}, {0f,1f,0f}, {1f,1f,1f}};

    /**
     * Number of test cases (number of input and output vectors).
     */

    public final static int NUM_TEST_CASES = 4;


    /**********************/
    /* Instance Variables */
    /**********************/

    /**
     * The single-neuron, single-layer perceptron network data model.
     */

    protected SingleLayerPerceptron slp;

    /**
     * Reference to the single neuron in the network.
     */

    protected ArtificialNeuron p;

    /**
     * Output vectors.
     */

    protected float[][] outputs = {{0f},{0f},{0f},{0f}};

    /**
     * Thread for training the perceptron network.
     */

    protected TrainingThread tt;

    /**
     * Flag to indicate training thread has been suspended.
     */

    protected boolean suspended = false;

    /**
     * Flag to indicate training should be stopped (Stop button pressed).
     */

    protected boolean stopThread = false;

    /**
     * Flag to indicate perceptron is being tested.
     */

    private boolean testing = false;

    /**
     * Current test case being performed.
     */

    private int testCase = 0;

    /**
     * Pointer to truth table panel.
     */

    protected TruthTablePanel ttPanel;

    /**
     * Pointer to 2D pattern space canvas.
     */

    protected PatternSpaceCanvas psCanvas;

    /**
     * Pointer to 3D pattern space canvas.
     */

    protected PatternSpaceCanvas3D psCanvas3D;

    /**
     * Pointer to training panel canvas.
     */

    protected TrainingPanel tPanel;

    /**
     * True if AND gate input is shown/being used in applet.
     */

    protected boolean showANDInput = false;


    /***************************/
    /* AWT Component Variables */
    /***************************/

    /**
     * TextField for the sum-squared error over all training vectors.
     */

    protected TextField sumSquaredError;

    /**
     * TextField for the error of the current training vector.
     */

    protected TextField error;

    /**
     * TextField for the AND gate input (x3 = x2 AND x1).
     */

    protected TextField weight3;

    /**
     * Checkbox for displaying the AND gate / solving XOR problem.
     */

    private Checkbox solveXOR;


    /****************************************/
    /* Component/Drawing Location Variables */
    /****************************************/

    /**
     * Diameter of the AND gate semi-circle.
     */

    private static final int andDia = 20;

    /**
     * Radius of the AND gate semi-circle.
     */

    private static final int andRad = andDia / 2;

    /**
     * X and Y coordinates of the AND weight text-field.
     */

    private int w3X, w3Y;

    /**
     *
     */

    private int cX, cY, rY;

    /**
     *
     */

    private int aX,aY1,aY2;

    /**
     * Coordinates for AND input wires (lines).
     */

    private int l1X,l1Y1,l1Y2;
    private int l2X,l2Y1,l2Y2;
    private int l3X,l3Y1,l3Y2;
    private int l4X1,l4Y,l4X2;
    private int l5X1,l5X2,l5Y2;


    /******************/
    /* Public Methods */
    /******************/

    /**
     * Return applet information string.
     * @return String - Author, version information string.
     */

    public String getAppletInfo() {
	return "PerceptronApplet (March 9, 1997), by Fred Corbett";
    }

    /**
     * Initialize this applet.
     */

    public void init() {
	int width = getSize().width;
	int height = getSize().height;

	super.init(350,350,0,0);

	resize(width,height);

	calculateDrawingCoordinates();

	// Add the panels and canvas
	ttPanel = new TruthTablePanel(this,200,200);
	add(ttPanel);
	ttPanel.setLocation(350,0);

	psCanvas = new PatternSpaceCanvas(this,200,200);
	add(psCanvas);
	psCanvas.setLocation(550,0);

	psCanvas3D = new PatternSpaceCanvas3D(this,200,200);
	add(psCanvas3D);
	psCanvas3D.setLocation(550,0);
	psCanvas3D.setVisible(false);

	tPanel = new TrainingPanel(this,400,150);
	add(tPanel);
	tPanel.setLocation(350,200);

	validate();
	setNeuronFieldsEditable(false);
	backgroundColor = getBackground();

	// Initialize, add, size, and move the components
	int w = 0;
	Label errorLabel = new Label("Current Error:");
	add(errorLabel);
	errorLabel.setSize(errorLabel.getPreferredSize());
	errorLabel.setLocation(w,height-50);
	w += errorLabel.getPreferredSize().width;

	error = new TextField("0", 4);
	add(error);
	error.setSize(error.getPreferredSize());
	error.setLocation(w,height-50);
	error.setEditable(false);
	w += error.getPreferredSize().width;

	Label sumSquaredErrorLabel = new Label("Squared Error:");
	add(sumSquaredErrorLabel);
	sumSquaredErrorLabel.setSize(sumSquaredErrorLabel.getPreferredSize());
	sumSquaredErrorLabel.setLocation(w,height-50);
	w += sumSquaredErrorLabel.getPreferredSize().width;

	sumSquaredError = new TextField("0", 4);
	add(sumSquaredError);
	sumSquaredError.setSize(sumSquaredError.getPreferredSize());
	sumSquaredError.setLocation(w,height-50);
	sumSquaredError.setEditable(false);
	w += sumSquaredError.getPreferredSize().width + 15;

	solveXOR = new Checkbox("Solve XOR");
	ItemListener solveXORListener = new ItemListener() {
	    public void itemStateChanged(ItemEvent e) {
		clickedSolveXOR();
	    }
	};
	solveXOR.addItemListener(solveXORListener);
	add(solveXOR);
	solveXOR.setSize(solveXOR.getPreferredSize());
	solveXOR.setLocation(w,height-50);

	weight3 = new TextField("0", 5);
	add(weight3);
	weight3.setSize(super.tfD);
	weight3.setLocation(w3X,w3Y);
	weight3.setEditable(false);
	weight3.setVisible(false);

	// Create a perceptron to manipulate
	slp = new SingleLayerPerceptron(3,1);
	p = slp.neuronAt(0,0);
	p.f(af);
    }

    /**
     * Paint the graphical elememts of this applet.
     * @param Graphics g - the applet graphics context.
     */

    public void paint(Graphics g) {
	super.paint(g);
	paintANDGate(g);
    }

    /**
     * Paint a 2 input AND gate connecting the two neuron inputs.
     * @param Graphics g - the applet graphics context.
     */

    public void paintANDGate(Graphics g) {
	if (showANDInput)
	    g.setColor(Color.blue);
	else
	    g.setColor(backgroundColor);
	// Draw (or erase) the AND gate and connections
	drawArrow(g,aX,aY1,aX,aY2);
	g.drawLine(l1X,l1Y1,l1X,l1Y2);
	g.fillOval(cX,cY,andDia,andDia);
	g.fillRect(cX,rY,andDia+1,andDia);
	g.drawLine(l2X,l2Y1,l2X,l2Y2);
	g.drawLine(l3X,l3Y1,l3X,l3Y2);
	g.drawLine(l4X1,l4Y,l4X2,l4Y);
	g.drawLine(l5X1,l4Y,l5X2,l4Y);
    }


    /**
     * Update the screen graphics. Over-ridden so background is not erased
     * each time an update occurs -> reduce flickering.
     * @param Graphics g - the screen graphics context.
     */

    public void update(Graphics g) {
	paint(g);
    }

    /**
     * Called when the applet is first loaded or is restarted. It will resume
     * the training thread if it has been suspended.
     */

    public void start() {
    }

    /**
     * Called whenever the applet must be stopped or suspended. This method
     * will stop the training thread if it is running
     */
    public void stop() {
	stopThread = true;
	this.afDialog.dispose();
    }

    public void destroy() {
	this.afDialog.dispose();
    }

    /**
     * Start a new training session.
     */

    public void train() {
	showStatus("Training...");
	stopThread = false;
	initializeTraining();
	tt = new TrainingThread(this);
	tt.start();
    }

    /**
     * Flag to indicate step-training is in progress.
     */

    private boolean stepTraining = false;

    /**
 * Current training step for step-training :(
 */

    private int trainingStep;

    /**
 * Number of step training steps.
 */

    private int numSteps;

    /**
 * Number of iterations for step training.
 */

    private int t;

    /**
 * Current count for step training.
 */

    private int count;

    /**
 * Total error for a step training iteration.
 */

    private float totalError;

    /**
 * Responds to Step button press; start a step-training session or
 * perform next step.
 */

    public void step() {
	if (!stepTraining) {
	    // First time step button has been pressed, initialize step training
	    showStatus("Training...click 'Step' to continue");
	    initializeTraining();
	    t = slp.getIterations();
	    stepTraining = true;
	    totalError = 0f;
	    count = 0;
	    trainingStep = 0;
	    numSteps = t * 4;
	    tPanel.stepButton().setEnabled(true);
	} else if (count < t) {
	    // Perform the next training step
	    showStatus("Training...iteration " + (count+1) + " of " + t);
	    float e = slp.nextTrainingStep();

	    if (showANDInput)
		psCanvas3D.updatePlane(p.getThreshold(),p.getWeight(0),p.getWeight(1),
				       p.getWeight(2),p.THRESHOLD_INPUT);
	    else
		psCanvas.updateLine(p.getThreshold(),p.getWeight(0),p.getWeight(1),
				    p.THRESHOLD_INPUT);

	    if (!((float) Math.abs(e) < slp.getErrorThreshold()))
		totalError += Math.abs(e);

	    // Update the text-fields
	    setFieldToFloat(error,e);
	    setNeuronFields(p);
	    setInputFields(slp.getInputs());
	    if (showANDInput)
		setFieldToFloat(weight3, p.getWeight(2));
	    trainingStep++;

	    if (trainingStep == NUM_TEST_CASES) {
		trainingStep = 0;
		setFieldToFloat(sumSquaredError, slp.getSumSquaredError());
		tPanel.progressBar().setValue(++count);
		tPanel.progressBar().updateDisplay();

		if (Math.abs(totalError) < slp.EE) {
		    stepTraining = false;
		    enableComponents();
		    showStatus("Training complete...required only " + count + " iterations.");
		} else
		    totalError = 0f;
	    }
	} else {
	    stepTraining = false;
	    enableComponents();
	    showStatus("Training complete...required all " + count + " iterations.");
	}
    }

    /**
 * Initialize the applet for a training session.
 */

    public void initializeTraining() {
	disableComponents();
	psCanvas3D.setRotationEnabled(false);

	// Reset any previous testing
	testing = false;
	neuronOutput.setBackground(backgroundColor);

	// Retrieve and set the training parameters
	// Correct any illegal values in the training panel text fields
	if (!slp.setErrorThreshold(tPanel.getErrorThreshold()))
	    tPanel.setErrorThreshold(slp.getErrorThreshold());
	if (!slp.setIterations(tPanel.getIterations()))
	    tPanel.setIterations(slp.getIterations());
	if (!slp.setLearningRate(tPanel.getLearningRate()))
	    tPanel.setLearningRate(slp.getLearningRate());

	// Initialize the neural network and applet fields
	if (showANDInput) {
	    slp.initializeTraining(AND_INPUTS, outputs);
	    setFieldToFloat(weight3, p.getWeight(2));
	    psCanvas3D.updatePlane(p.getThreshold(),p.getWeight(0),p.getWeight(1),
				   p.getWeight(2),p.THRESHOLD_INPUT);
	} else {
	    slp.initializeTraining(REG_INPUTS, outputs);
	    psCanvas.updateLine(p.getThreshold(),p.getWeight(0),p.getWeight(1),
				p.THRESHOLD_INPUT);
	}

	// Set the weights and clear all other fields
	setNeuronFields(p);
	setFieldToFloat(neuronOutput,0f);
	setFieldToFloat(weightedSum,0f);
	setFieldToFloat(input1, 0f);
	setFieldToFloat(input2, 0f);
	setFieldToFloat(error, 0f);
	setFieldToFloat(sumSquaredError, 0f);

	// Initialize the progress bar
	tPanel.progressBar().setValue("completeValue",slp.getIterations());
	tPanel.progressBar().setValue(0);
	tPanel.progressBar().updateDisplay();
    }


    /**
     * Stop the training (step or thread) in progress.
     */

    public void stopTraining() {
	if (!stopThread)
	    stopThread = true;
	if (stepTraining)
	    stepTraining = false;
	if (testing)
	    testing = false;
	enableComponents();
	psCanvas3D.setRotationEnabled(true);
	showStatus("Training stopped...");
    }

    /**
     * Coordinate the testing of results from a neuron training session.
     */

    public void test() {
	// Not currently testing, reset testing process.
	if (!testing) {
	    testing = true;
	    disableComponents();
	    tPanel.testButton().setEnabled(true);
	    testCase = 0;
	    setFieldToFloat(error,0f);
	    setFieldToFloat(sumSquaredError,0f);
	    if (showANDInput)
		slp.initializeTesting(AND_INPUTS, outputs);
	    else
		slp.initializeTesting(REG_INPUTS, outputs);
	}
	showStatus("Testing...case " + (testCase + 1) + " of " + NUM_TEST_CASES);
	float e = slp.nextTestingStep();
	setFieldToFloat(error,e);
	if (Math.abs(e) < tPanel.getErrorThreshold())
	    neuronOutput.setBackground(Color.green);
	else
	    neuronOutput.setBackground(Color.red);
	setNeuronFields(p);
	setFieldToFloat(weight3, p.getWeight(2));
	setInputFields(slp.getInputs());
	testCase++;
	if (testCase >= NUM_TEST_CASES) {
	    testCase = 0;
	    setFieldToFloat(sumSquaredError, slp.getSumSquaredError());
	    enableComponents();
	}
    }

    /**
     * Disable the applet's AWT components.
     */

    public void disableComponents() {
	solveXOR.setEnabled(false);
	ttPanel.disableComponents();
	tPanel.disableComponents();
	afEnabled(false);
    }

    /**
     * Enable the applet's AWT components.
     */

    public void enableComponents() {
	solveXOR.setEnabled(true);
	ttPanel.enableComponents();
	tPanel.enableComponents();
	afEnabled(true);
    }

    /**
     * Respond to a change in the truth table panel's choice components.
     * @param int index - the index of the choice component.
     * @param float value - the new value selected in the choice box.
     */

    public void buttonHasChanged(int index, float value) {
	outputs[index][0] = value;
	if (showANDInput) {
	    psCanvas3D.updatePlane(p.getThreshold(),p.getWeight(0),p.getWeight(1),
				   p.getWeight(2),p.THRESHOLD_INPUT);
	} else
	    psCanvas.updateLine(p.getThreshold(),p.getWeight(0),p.getWeight(1),
				p.THRESHOLD_INPUT);

	if (value > 0)
	    psCanvas3D.changeSphereColour(index, "green");
	else
	    psCanvas3D.changeSphereColour(index, "red");
    }


    /*******************/
    /* Private Methods */
    /*******************/

    /**
     * Calculate the drawing coordinates for the screen objects.
     * DON'T TRY TO READ THIS!
     */

    private void calculateDrawingCoordinates() {
	w3X = super.wsX;  w3Y = super.w1Y;
	aX = (super.width / 2);
	aY1 = w3Y;  aY2 = super.cY + super.cDia;
	int vGap = (super.height - (super.tfHH+super.tfH+w3Y+andRad+andDia)) / 2;
	l1X = aX;
	l1Y1 = (w3Y + super.tfH);  l1Y2 = l1Y1 + vGap;
	cX = aX - andRad;
	cY = l1Y2;
	rY = cY + andRad;
	l2X = cX + 5;
	l2Y1 = rY + andDia;
	l2Y2 = super.i2Y + super.tfHH;
	l3X = cX + andDia - 5;
	l3Y1 = l2Y1;
	l3Y2 = l2Y2;
	l4X1 = super.i2X + super.tfW;
	l4Y = l2Y2;
	l4X2 = width / 2 - 5;
	l5X1 = super.i1X;
	l5X2 = width / 2 + 5;
    }

    /**
     * Respond to user clicking the solveXOR checkbox.
     */

    private void clickedSolveXOR() {
	if (solveXOR.getState()) {
	    showANDInput = true;
	    paintANDGate(getGraphics());
	    weight3.setVisible(true);
	    psCanvas.setVisible(false);
	    psCanvas3D.setVisible(true);
	    psCanvas3D.start();
	}
	else {
	    showANDInput = false;
	    weight3.setVisible(false);
	    psCanvas3D.setVisible(false);
	    psCanvas.setVisible(true);
	    psCanvas3D.stop();
	}
	paintANDGate(getGraphics());
    }

} // End of PerceptronApplet Class


/**
 * Provides threaded execution for neuron training. This allows the user to
 * 1) start a training session and work on other tasks and 2) interrupt the
 * training in progress.
 *
 * @author  Fred Corbett
 * @version December 31, 1996
 */

class TrainingThread extends Thread {


    /**********************/
    /* Instance Variables */
    /**********************/

    /**
     * Parent applet.
     */

    private PerceptronApplet pa;


    /***********************/
    /* Constructor Methods */
    /***********************/

    /**
     * Instantiate and initialize a new TrainingThread.
     * @param PerceptronApplet - reference to parent applet.
     */

    public TrainingThread(PerceptronApplet pa) {
	super();
	this.pa = pa;
    }


    /******************/
    /* Public Methods */
    /******************/

    /**
     * Run the training algorithm thread.
     */

    public void run() {
	int i = 0;
	float e = 0f;
	float te = 0f;

	// Initialize
	boolean zeroError = false;

	int t = pa.slp.getIterations();
	float et = pa.slp.getErrorThreshold();

	while ((i < t) && (!zeroError) && (!pa.stopThread)) {
	    pa.showStatus("Training...iteration " + (i+1) + " of " + t);
	    te = 0f;
	    for (int j = 0; j < 4; j++) {
		e = pa.slp.nextTrainingStep();
		if (pa.showANDInput)
		    pa.psCanvas3D.updatePlane(pa.p.getThreshold(),pa.p.getWeight(0),
					      pa.p.getWeight(1), pa.p.getWeight(2), pa.p.THRESHOLD_INPUT);
		else
		    pa.psCanvas.updateLine(pa.p.getThreshold(),pa.p.getWeight(0),
					   pa.p.getWeight(1), pa.p.THRESHOLD_INPUT);
	    
		if (!((float) Math.abs(e) < et)) {
		    te += Math.abs(e);
		}
		pa.setFieldToFloat(pa.error,e);
		pa.setNeuronFields(pa.p);
		pa.setInputFields(pa.slp.getInputs());
		if (pa.showANDInput)
		    pa.setFieldToFloat(pa.weight3, pa.p.getWeight(2));
	    }
	    pa.setFieldToFloat(pa.sumSquaredError,pa.slp.getSumSquaredError());
	    pa.tPanel.progressBar().setValue(++i);
	    pa.tPanel.progressBar().updateDisplay();
	
	    if (Math.abs(te) < pa.slp.EE) {
		zeroError = true;
		pa.showStatus("Training complete...required only " + i + " iterations.");
		pa.stopThread = true;
	    }
	}
	if (!pa.stopThread) {
	    pa.showStatus("Training complete...required all " + t + " iterations.");
	    pa.tPanel.progressBar().setValue(t);
	    pa.tPanel.progressBar().updateDisplay();
	}
	pa.enableComponents();
	pa.tt = null;
	
	pa.psCanvas3D.setRotationEnabled(true);
    }
    
    
}  // End of TrainingThread Class
