/**
 * This applet displays and manages a graphical representation of a simple
 * 2 input artificial neuron. It includes text fields that allow the
 * display and user entry of neuron parameters (such as inputs and weights).
 *
 * @author  Fred Corbett
 * @version March 9, 1997
 * modified Feb. 13, 1998 (Alix Herrmann): changed number of activation
 *          functions from the hard-coded value 4 to af.NUM_FUNCTIONS.
 *          If adding a function, note that the list of image names
 *          is still hard-coded and has to be changed by hand.
 *                                         .
 */

import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import java.util.Vector;
import anns.ActivationFunction;
import anns.ArtificialNeuron;

public class SimpleNeuronApplet extends Applet  {


  /*******************/
  /* Class Variables */
  /*******************/

    /**
     * Diameter of neuron body (circle).
     */
    
    public static int cDia = 50;

  /**
   * Radius of neuron body (circle).
   */

  public static int cRad = cDia / 2;

  /**
   * Width/height of activation function .gif images files.
   */

  public static int gifSize = 50;

  /**
   * Offset from edge of activation function rectangle for drawing function images.
   */

  public static int gifOffset = (cDia - gifSize) / 2;

  /**
   * Constant: Array containing names of function images. These are 50 x 50
   * transparent .GIF files that show a graphical rendition of the
   * particular function type.
   */

  public final static String functionImageNames[] =
    {"unitStep.gif","sigmoid.gif", "piecewiseLinear.gif", "Gaussian.gif", 
    "identity.gif"};


  /**********************/
  /* Instance variables */
  /**********************/

  /**
   * Activation function image index.
   */

  private byte imageIndex = 0;

  /**
   * Array of activation function images.
   */

  private static Image functionImages[];

  /**
   * This rectangle defines the display area of the activation function
   * rectangle and image.
   */

  private Rectangle afRect;

  /**
   * If true, the activation function can be changed by clicking the mouse
   * in its display area in the applet.
   */

  private boolean afEnabled = true;

  /**
   * Frame for activation function parameter dialog.
   */

  protected ActivationFunctionDialog afDialog;

  /**
   * Activation function for the neuron.
   */

  protected ActivationFunction af; 

  /**
   * The width of this applet in pixels.
   */

  protected int width = 400;

  /**
   * The height of this applet in pixels.
   */

  protected int height = 400;

  /**
   * Distance in pixels from left edge of applet where neuron paints start.
   */

  protected int xOffset = 0;

  /**
   * Distance in pixels from top edge of applet where neuron paints start.
   */

  protected int yOffset = 0;

  /**
   * The background color of the applet.
   */

  protected Color backgroundColor;


  /***************************/
  /* AWT Component Variables */
  /***************************/

  /**
   * TextField for total output of the neuron.
   */

  protected TextField neuronOutput;

  /**
   * Textfield for weighted sum value of the neuron. This is the input to
   * the neuron's activation function.
   */

  protected TextField weightedSum;

  /**
   * TextField for threshold of the neuron.
   */

  protected TextField threshold;

  /**
   * TextField for weight 1 of the neuron.
   */

  protected TextField weight1;

  /**
   * TextField for weight 2 of the neuron.
   */

  protected TextField weight2;

  /**
   * TextField for input 1 of the neuron.
   */

  protected TextField input1;

  /**
   * TextField for input 2 of the neuron.
   */

  protected TextField input2;


  /****************************************/
  /* Component/Drawing Location Variables */
  /****************************************/

  /**
   * Textfield dimensions.
   */

  protected Dimension tfD;

  /**
   * Textfield height, width, half-height and half-width parameters.
   */

  protected int  tfH, tfW, tfHH, tfHW;

  /**
   * Neuron rectangle (activation function) and circle (body) drawing
   * coordinates.
   */

  protected int rX,rY,cX,cY;

  /**
   * Height of neuron arrows.
   */

  protected int arrowH;

  /**
   * Textfield placement coordinates.
   */

  protected int noX,noY,wsX,wsY,thX,thY,w1X,w1Y,w2X,w2Y,i1X,i1Y,i2X,i2Y;

  /**
   * Endpoints of the neuron arrows.
   */

  private int a1X1,a1Y1,a1X2,a1Y2;
  private int a2X1,a2Y1,a2X2,a2Y2;
  private int a3X1,a3Y1,a3X2,a3Y2;
  private int a4X1,a4Y1,a4X2,a4Y2;
  private int a5X1,a5Y1,a5X2,a5Y2;

  /**
   * Label drawing coordinates.
   */

  private int l1X,l1Y,l2X,l2Y,l3X,l3Y,l4X,l4Y,l5X,l5Y,l6X,l6Y,l7X,l7Y;


  /************************/
  /* Constructors Methods */
  /************************/


  /******************************/
  /* Accessor / Mutator Methods */
  /******************************/

  /**
   * Accessor for afEnabled.
   * @return boolean - true activation function is enabled, false otherwise.
   */

  public boolean afEnabled() {
    return afEnabled;
  }

  /**
   * Mutator for afEnabled.
   * @param boolean b - the new value for afEnabled.
   */

  public void afEnabled(boolean b) {
    afEnabled = b;
  }


  /******************/
  /* Public Methods */
  /******************/

  /**
   * Return applet information string.
   * @return String - Author, version information string.
   */

  public String getAppletInfo() {
    return "SimpleNeuronApplet (March 4, 1996), by Fred Corbett";
  }


  /**
   * This is a hack to allow more control over the neuron diagram size...
   */

  public void init(int w, int h, int xOff, int yOff) {
    width = w;
    height = h;
    xOffset = xOff;
    yOffset = yOff;

    cDia = height / 5;
    if (cDia > 65)
      cDia = 65;
    else if (cDia < 50)
      cDia = 50;
    cRad = cDia / 2;
    gifSize = cDia;
    gifOffset = (cDia - gifSize) / 2;
    init1();
  }

  /**
   * Initializes the applet, user interface and compnents.
   */
  public void init1() {
    // Initialize the activations function, images, and dialog
    af = new ActivationFunction(af.UNIT_STEP);
    loadAFImages();
    afDialog = new ActivationFunctionDialog(this);

    // Translate the graphics context if required
    getGraphics().translate(xOffset,yOffset);

    // Initialize tab chain, components (drawing locations)...
    initComponents();
    afRect = new Rectangle(rX, rY, cDia, cDia);
    backgroundColor = this.getBackground();
    
    // We enable the events for the applet.
    this.enableEvents(AWTEvent.MOUSE_EVENT_MASK);
  }

  /**
   * Responsible for drawing graphical elements of the neuron.
   * @param Graphics g - the graphics context of the applet.
   */

  public void paint(Graphics g) {

    // Draw the neuron graphical elements
    g.setColor(Color.blue);
    drawArrow(g,a1X1,a1Y1,a1X2,a1Y2);
    g.setColor(Color.white);
    g.fillRect(rX,rY,cDia,cDia);
    g.drawImage(functionImages[af.functionType()],rX+gifOffset,rY+gifOffset,gifSize,gifSize,this);
    g.setColor(Color.blue);
    drawArrow(g,a2X1,a2Y1,a2X2,a2Y2);
    g.fillOval(cX,cY,cDia,cDia);
    drawArrow(g,a3X1,a3Y1,a3X2,a3Y2);
    drawArrow(g,a4X1,a4Y1,a4X2,a4Y2);
    drawArrow(g,a5X1,a5Y1,a5X2,a5Y2);

    // Draw the neuron labels
    g.setColor(Color.black);
    g.drawString("y",l1X,l1Y);
    g.drawString("-1",l2X,l2Y);
    g.drawString("u",l3X,l3Y);
    g.drawString("w2",l4X,l4Y);
    g.drawString("w1",l5X,l5Y);
    g.drawString("x2",l6X,l6Y);
    g.drawString("x1",l7X,l7Y);
  }

  public void update(Graphics g) {
    paint(g);
  }

  /**
   * Respond to a mouse click over the applet. If the click occurs over the
   * activation function image, it is changed. If the click is from the second
   * mouse button the function parameter dialog is displayed.
   * @param MouseEvent e - the mouse event data object.
   */
    protected void processMouseEvent(MouseEvent e) {
	if (afRect.contains(e.getX(),e.getY()) && afEnabled && (e.getID() == MouseEvent.MOUSE_CLICKED)) {
	    if (e.getModifiers() == MouseEvent.BUTTON1_MASK) {
		changeActivationFunction();
	    }
	    else if (e.getModifiers() == MouseEvent.BUTTON1_MASK + MouseEvent.SHIFT_MASK) {
		afDialog.show();
	    }
	}	    
    }
  
  /**
   * Change the activation function and its image to the next function.
   */

  public void changeActivationFunction() {
    imageIndex++;
    imageIndex %= af.NUM_FUNCTIONS;
    changeActivationFunctionTo(imageIndex);
  }

  /**
   * Change the activation function and image to the function specified by
   * afCode.
   * @param byte afCode - the code for the activation function.
   */

  public void changeActivationFunctionTo(byte afCode) {
    Graphics g;  // The applet graphics context

    // Don't redraw the image if current function type is specified
    if (afCode != af.functionType()) {
      af.changeFunctionTypeTo(afCode);
      g = getGraphics();
      g.setColor(Color.white);
      g.fillRect(rX,rY,cDia,cDia);
      g.drawImage(functionImages[af.functionType()],rX+gifOffset,rY+gifOffset,gifSize,gifSize,this);
    }
  }

  /**
   * Draws an arrow from (x1,y1) to (x2,y2), with (x2,y2) as the tip of the arrow head.
   * NOTE: Not much error checking is done here, use cautiously.
   * @param Graphics g - graphics context of applet
   * @param int x1, y1 - the start endpoint of the arrow
   * @param int y2, x2 - the endpoint with the arrow head.
   */

  public void drawArrow(Graphics g, int x1, int y1, int x2, int y2) {
    Polygon p;  // Polygon used for arrow head

    p = new Polygon();
    p.addPoint(x2,y2);

    // Vertical arrow
    if (x2 == x1) {
      if (y2 > y1) {
        p.addPoint(x2-5,y2-10);
        p.addPoint(x2+5,y2-10);
      } else {
        p.addPoint(x2-5,y2+10);
        p.addPoint(x2+5,y2+10);
      }
    }
    // Horizontal arrow
    else if (y2 == y1) {
      if (x2 > x1) {
        p.addPoint(x2-10,y2-5);
        p.addPoint(x2-10,y2+5);
      } else {
        p.addPoint(x2+10,y2-5);
        p.addPoint(x2+10,y2+5);
      }
    }
    // Now we need to rotate the arrow head about the origin
    else {
      // Calculate the angle of rotation and adjust for the quadrant
      double t1 = Math.abs(new Integer(y2 - y1).doubleValue());
      double t2 = Math.abs(new Integer(x2 - x1).doubleValue());
      double theta = Math.atan(t1 / t2);
      if (x2 < x1) {
        if (y2 < y1)
          theta = Math.PI + theta;
        else
          theta = - (Math.PI + theta);
      } else if (x2 > x1 && y2 < y1)
        theta =  2*Math.PI - theta;
      double cosTheta = Math.cos(theta);
      double sinTheta = Math.sin(theta);

      // Create the other points and translate the arrow to the origin
      Point p2 = new Point(-10,-5);
      Point p3 = new Point(-10,+5);

      // Rotate the points (without using matrices!)
      int x = new Long(Math.round((cosTheta * p2.x) - (sinTheta * p2.y))).intValue();
      p2.y = new Long(Math.round((sinTheta * p2.x) + (cosTheta * p2.y))).intValue();
      p2.x = x;
      x = new Long(Math.round((cosTheta * p3.x) - (sinTheta * p3.y))).intValue();
      p3.y = new Long(Math.round((sinTheta * p3.x) + (cosTheta * p3.y))).intValue();
      p3.x = x;

      // Translate back to desired location and add to polygon
      p2.translate(x2,y2);
      p3.translate(x2,y2);
      p.addPoint(p2.x,p2.y);
      p.addPoint(p3.x,p3.y);
    }
    g.fillPolygon(p);
    g.drawLine(x1,y1,x2,y2);
  }

  /**
   * Return the contents of a textfield as an int.
   * @param TextField t - the textfield component.
   * @return int.
   * @exception NumberFormatException.
   */

  public int getFieldAsInt(TextField t) {
    try {
      return (Integer.valueOf(t.getText())).intValue();
    }
    catch (NumberFormatException e) {
      setFieldToInt(t, 0);
      return 0;
    }
  }

  /**
   * Return the contents of a textfield as a float.
   * @param TextField t - the textfield component.
   * @return float.
   * @exception NumberFormatException.
   */

  public float getFieldAsFloat(TextField t) {
    try {
      return (Float.valueOf(t.getText())).floatValue();
    }
    catch (NumberFormatException e) {
      setFieldToFloat(t, 0f);
      return 0f;
    }
  }

  /**
   * Converts the contents of the applet text fields to floats and stores them in
   * n, the aritificial neuron.
   * @param ArtificialNeuron n - the neuron data model.
   */

  public void getNeuronFields(ArtificialNeuron n) {
    n.setThreshold(getFieldAsFloat(threshold));
    n.setWeight(0, getFieldAsFloat(weight1));
    n.setWeight(1,getFieldAsFloat(weight2));
  }

  /**
   * Return the current input field values in an array.
   * @return the input field values in an 2 element array.
   */

  public float[] getInputFields() {
    float[] temp;

    temp = new float[2];
    temp[0] = getFieldAsFloat(input1);
    temp[1] = getFieldAsFloat(input2);
    return temp;
  }

  /**
   * Sets the value of a text field to an int.
   * @param TextField t - the textfield component.
   * @param int i - the number to display.
   * @exception NumberFormatException.
   */

  public void setFieldToInt(TextField t, int i) {
    String s;

    try {
      s = String.valueOf((int) i);
    }
    catch (NumberFormatException e) {
      s = "0";
    }
    if (s != t.getText())
      t.setText(s);
    t.select(0,0);
  }

  /**
   * Sets the value of a text field to a float.
   * @param TextField t - the textfield component.
   * @param float f - the number to display.
   * @exception NumberFormatException.
   */

  public void setFieldToFloat(TextField t, float f) {
    String s;

    try {
      s = String.valueOf((float) f);
    }
    catch (NumberFormatException e) {
      s = "0";
    }
    if (s != t.getText())
      t.setText(s);
    t.select(0,0);
  }

  /**
   * Sets the applet text fields to the current neuron values.
   * @param ArtificialNeuron n - the data model for the neuron.
   */

  public void setNeuronFields(ArtificialNeuron n) {
    setFieldToFloat(neuronOutput, n.getOutput());
    setFieldToFloat(weightedSum, n.getWeightedSum());
    setFieldToFloat(threshold, n.getThreshold());
    setFieldToFloat(weight1, n.getWeight(0));
    setFieldToFloat(weight2, n.getWeight(1));
  }

  /**
   * Set the input field to the values specified in an 2 element array.
   * @param float[] inps - the input values.
   */

  public void setInputFields(float[] inps) {
    setFieldToFloat(input1, inps[0]);
    setFieldToFloat(input2, inps[1]);
  }

  /**
   * Disables the neuron text fields.
   */

  public void disableNeuronFields() {
    neuronOutput.setEnabled(false);
    weightedSum.setEnabled(false);
    threshold.setEnabled(false);
    weight1.setEnabled(false);
    weight2.setEnabled(false);
    input1.setEnabled(false);
    input2.setEnabled(false);
  }

  /**
   * Enables the neuron text fields.
   */

  public void enableNeuronFields() {
    neuronOutput.setEnabled(true);
    weightedSum.setEnabled(true);
    threshold.setEnabled(true);
    weight1.setEnabled(true);
    weight2.setEnabled(true);
    input1.setEnabled(true);
    input2.setEnabled(true);
  }

  /**
   * Set the neuron fields to editable if t is true. This technique has an
   * advantage over completely disabling the fields. If a field is set to
   * uneditable, the field can still take focus. Also, the user can scoll
   * the contents of the field even if the value can't be changed.
   * @param boolean t - if true, fields are set to editable otherwise the
   * user cannot change the field contents.
   */

  public void setNeuronFieldsEditable(boolean t) {
    neuronOutput.setEditable(t);
    weightedSum.setEditable(t);
    threshold.setEditable(t);
    weight1.setEditable(t);
    weight2.setEditable(t);
    input1.setEditable(t);
    input2.setEditable(t);
  }


  /*******************/
  /* Private Methods */
  /*******************/

  /**
   * Loads the AF images and tracks the progress using a Mediatracker.
   * Progress is displayed in the status bar.
   */

  private void loadAFImages() {
    MediaTracker mt;   // The media tracker
    int i;             // Loop counter
    String imageName;  // Name of image being loaded

    functionImages = new Image[af.NUM_FUNCTIONS];
    mt = new MediaTracker(this);
    for (i = 0; i < af.NUM_FUNCTIONS; i++) {
      imageName = "images/" + functionImageNames[i];
      try {	 
	  functionImages[i] = getImage(getCodeBase(), imageName);	  
	  mt.addImage(functionImages[i],i);
      } catch(Exception e) {
        System.err.println(e);
      }
      try {
        mt.checkID(i, true);
        mt.waitForID(i);
        showStatus("Loading image: " + imageName);
      } catch(Exception e) {
        System.err.println(e);
      }
     }
  }

  /**
   * Add, resize, and move the applet components. Components are also added
   * to the applet.
   */

  public void initComponents() {
    // Initialize a null layout manager for absolute positioning
    setLayout(null);

    // Setup the neuronOutput and activationValue text fields
    neuronOutput = new TextField("0",5);
    add(neuronOutput);
    neuronOutput.setSize(cDia - 10, neuronOutput.getPreferredSize().height);
    tfD = neuronOutput.getSize();

    calculateDrawingCoordinates();
    neuronOutput.setLocation(noX,noY);

    weightedSum = new TextField("0", 5);
    add(weightedSum);
    weightedSum.setSize(tfD);
    weightedSum.setLocation(wsX,wsY);

    // Setup the threshold text field
    threshold = new TextField("0", 5);
    add(threshold);
    threshold.setSize(tfD);
    threshold.setLocation(thX,thY);

    // Setup the input and weight text fields
    weight2 = new TextField("0", 5);
    add(weight2);
    weight2.setSize(tfD);
    weight2.setLocation(w2X,w2Y);

    weight1 = new TextField("0", 5);
    add(weight1);
    weight1.setSize(tfD);
    weight1.setLocation(w1X,w1Y);

    input2 = new TextField("0", 5);
    add(input2);
    input2.setSize(tfD);
    input2.setLocation(i2X,i2Y);

    input1 = new TextField("0", 5);
    add(input1);
    input1.setSize(tfD);
    input1.setLocation(i1X,i1Y);

    validate();
  }

  /**
   * Calculate the drawing and component placement coordinates for this
   * applet.
   * WARNING don't waste your time reading this...this is pure garbage, but
   * it works, speeds up the paints and allows some flexiblilty in sizing
   * the applet.
   */

  private void calculateDrawingCoordinates() {
    int xPos, yPos;
    int dx1,dy1,dx2,dy2;
    int  hGap;

    // Initialize
    tfH = tfD.height;
    tfW = tfD.width;
    tfHH = Math.round(tfH / 2);
    tfHW = Math.round(tfW / 2);
    arrowH = (height - ((cDia * 2) + (3 * tfH)) ) / 4;
    xPos = (width / 2);
    yPos = 0;
    dx1 = (int) (cRad * Math.cos(Math.PI/4));
    dy1 = (int) (cRad * Math.sin(Math.PI/4));
    dx2 = dy2 = (arrowH +  arrowH + tfH + cRad - dy1);
    noX = xPos - tfHW;                noY = yPos;
    yPos += (tfH + arrowH + arrowH + cDia + cRad - tfHH);
    wsX = xPos - tfHW;                wsY = yPos;
    hGap = (xPos - (cRad + 20 + tfW)) / 2;
    thX = hGap + 20;                  thY = yPos;
    w2X = xPos - arrowH - tfHW - dx1 - dx1;
    w2Y = yPos + tfHH + cRad + arrowH;
    w1X = xPos + arrowH - tfHW + dx1 + dx1;
    w1Y = w2Y;
    xPos = width / 2;                 yPos = tfH;
    a1X1 = xPos;                      a1Y1 = yPos + arrowH;
    a1X2 = xPos;                      a1Y2 = yPos;
    yPos += arrowH;
    rX = xPos - cRad;                 rY = yPos;
    yPos += cDia;
    a2X1 = xPos;                      a2Y1 = yPos+arrowH;
    a2X2 = xPos;                      a2Y2 = yPos;
    yPos += arrowH;
    cX = xPos - cRad;                 cY = yPos;
    xPos = 20;                        yPos += cRad;
    a3X1 = xPos;                      a3Y1 = yPos;
    a3X2 = rX;                        a3Y2 = yPos;
    xPos = width / 2;                 yPos += dy1;
    a4X1 = xPos - dx1 - dx2;          a4Y1 = yPos + dy2;
    a4X2 = xPos - dx1;                a4Y2 = yPos;
    a5X1 = xPos + dx1 + dx2;          a5Y1 = yPos + dy2;
    a5X2 = xPos + dx1;                a5Y2 = yPos;
    i2X = a4X1 - tfHW;                i2Y = a4Y1;
    i1X = width - i2X - tfW;          i1Y = i2Y;

    // Set the label drawing coordinates
    l1X = noX -10;                    l1Y = noY + tfHH + 5;
    l2X = 5;                          l2Y = thY + tfHH + 5;
    l3X = thX + tfW + 5;              l3Y = thY + 5;
    l4X = w2X + - 20;                 l4Y = w2Y + 5;
    l5X = w1X + tfW + 5;              l5Y = w1Y + 5;
    l6X = i2X + tfW + 5;              l6Y = i2Y + 5;
    l7X = i1X - 15;                   l7Y = i1Y + 5;
  }

}  // End of SimpleNeuronApplet Class


/**
 * This subclass of frame implements a dialog box that allows activation
 * function parameters to be modified by the user. Note: This is not a true
 * dialog as Java does not allow applets to instantiate and use dialogs.
 *
 * NOTE: Need to change layout to deal with platform specific sizing of
 * compoments and the frame.
 *
 * @author  Fred Corbett
 * @version January 2, 1997
 */

class ActivationFunctionDialog extends Frame implements WindowListener {


  /**********************/
  /* Instance variables */
  /**********************/

  /**
   * The parent applet object.
   */

  private SimpleNeuronApplet sn;


  /***************************/
  /* AWT Component Variables */
  /***************************/

  /**
   * When pressed, textfields are checked for changes and activation function
   * parameters are change accordingly.
   */

  private Button okButton;

  /**
   * When pressed, the dialog is hidden and changes to textfields are ignored.
   */

  private Button cancelButton;

  /**
   * TextField for sigmoid beta parameter.
   */

  private TextField sigmoidBeta;

  /**
   * TextField for piecewise linear slope parameter.
   */

  private TextField pwlSlope;

  /**
   * TextField for Gaussian mu parameter.
   */

  private TextField gaussianMu;


  /***********************/
  /* Constructor Methods */
  /***********************/

  /**
   * Default constructor: user must specify size of frame.
   * @param master - the master or instantiating object.
   * @param w - the width of the dialog
   * @param h - the height of the dialog
   */

  public ActivationFunctionDialog(SimpleNeuronApplet master) {
    super("Function Parameters:");
    sn = master;
    initComponents();
  }


  /******************/
  /* Public Methods */
  /******************/

  /**
   * Set the textfields to their current values, reset the tab chain and
   * make this frame visible.
   */

  public void show() {
    sn.setFieldToFloat(sigmoidBeta, sn.af.beta());
    sn.setFieldToFloat(pwlSlope, sn.af.slope());
    sn.setFieldToFloat(gaussianMu, sn.af.mu());
    super.show();
  }

  /*******************/
  /* Private Methods */
  /*******************/

  /**
   * Add, resize and move the components in this frame.
   */

  private void initComponents() {
    GridBagLayout gridbag = new GridBagLayout();
    GridBagConstraints c = new GridBagConstraints();
    setLayout(gridbag);

    // Add the parameter labels and textfields
    Label l1 = new Label("Sigmoid beta:");
    c.gridwidth = 1;
    c.weightx = 0.0;
    c.weighty = 1.0;
    c.insets = new Insets(5,5,0,5);
    c.anchor = GridBagConstraints.WEST;
    gridbag.setConstraints(l1, c);
    add(l1);

    sigmoidBeta = new TextField(10);
    c.gridwidth = GridBagConstraints.REMAINDER;
    c.weightx = 1.0;
    c.anchor = GridBagConstraints.CENTER;
    c.fill = GridBagConstraints.HORIZONTAL;
    gridbag.setConstraints(sigmoidBeta, c);
    add(sigmoidBeta);

    Label l2 = new Label("Piecewise linear slope:");
    c.gridwidth = 1;
    c.weightx = 0.0;
    c.anchor = GridBagConstraints.WEST;
    c.fill = GridBagConstraints.NONE;
    gridbag.setConstraints(l2, c);
    add(l2);

    pwlSlope = new TextField(10);
    c.gridwidth = GridBagConstraints.REMAINDER;
    c.weightx = 1.0;
    c.anchor = GridBagConstraints.CENTER;
    c.fill = GridBagConstraints.HORIZONTAL;
    gridbag.setConstraints(pwlSlope, c);
    add(pwlSlope);

    Label l3 = new Label("Gaussian mu:");
    c.gridwidth = 1;
    c.weightx = 0.0;
    c.anchor = GridBagConstraints.WEST;
    c.fill = GridBagConstraints.NONE;
    gridbag.setConstraints(l3, c);
    add(l3);

    gaussianMu = new TextField(10);
    c.gridwidth = GridBagConstraints.REMAINDER;
    c.weightx = 1.0;
    c.anchor = GridBagConstraints.CENTER;
    c.fill = GridBagConstraints.HORIZONTAL;
    gridbag.setConstraints(gaussianMu, c);
    add(gaussianMu);

    okButton = new Button("OK");
    // We add a mouselistener on the button.
    ActionListener okListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		clickedOKButton();		
	    }
	};    
    okButton.addActionListener(okListener);
    
    c.gridwidth = 1;
    c.insets = new Insets(5,5,5,5);
    c.fill = GridBagConstraints.NONE;
    gridbag.setConstraints(okButton, c);
    add(okButton);

    cancelButton = new Button("Cancel");
    ActionListener cancelListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		clickedCancelButton();		
	    }
	};    
    cancelButton.addActionListener(cancelListener);
    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints(cancelButton, c);
    add(cancelButton);

    this.addWindowListener(this);

    pack();
  }

  /**
   * The user clicked the OK button. Set the activation function parameters
   * hide the frame, and give the focus back to the master applet.
   */
  private void clickedOKButton() {
    sn.af.beta(sn.getFieldAsFloat(sigmoidBeta));
    sn.af.slope(sn.getFieldAsFloat(pwlSlope));
    sn.af.mu(sn.getFieldAsFloat(gaussianMu));
    setVisible(false);
    sn.requestFocus();
  }

  /**
   * The clicked the Cancel button (or the window close button). Hide the
   * frame, and give the focus back to the master applet.
   */
  private void clickedCancelButton() {
      setVisible(false);
      sn.requestFocus();
  } 
    
    public void windowActivated(WindowEvent e) {}
    public void windowClosed(WindowEvent e) {
	clickedCancelButton();
    }
    public void windowClosing(WindowEvent e) {}
    public void windowDeactivated(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowIconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}

}  // End of ActivationFunctionDialog Class
