import java.applet.*;
import java.awt.*;
import java.awt.image.*;

/**
 * This class provides a graphical model for a 3D sphere.
 */

class Sphere {

  /**********************/
  /* Instance Variables */
  /**********************/

  public static Applet applet;

  private static byte[] data;

  private final static int R = 30;

  private final static int hx = 15;

  private final static int hy = 15;

  private final static int bgGrey = 192;

  private final static int nBalls = 16;

  private static int maxr;

  private int Rl;

  private int Gl;

  private int Bl;

  private Image balls[];

  static {
    data = new byte[R * 2 * R * 2];
    int mr = 0;
    for (int Y = 2 * R; --Y >= 0;) {
      int x0 = (int) (Math.sqrt(R * R - (Y - R) * (Y - R)) + 0.5);
      int p = Y * (R * 2) + R - x0;
      for (int X = -x0; X < x0; X++) {
        int x = X + hx;
        int y = Y - R + hy;
        int r = (int) (Math.sqrt(x * x + y * y) + 0.5);
        if (r > mr)
          mr = r;
        data[p++] = r <= 0 ? 1 : (byte) r;
      }
    }
    maxr = mr;
  }


  /***********************/
  /* Constructor Methods */
  /***********************/

  /**
   * Construct a new sphere object.
   * @param int Rl - red colour level.
   * @param int Gl - green colour level.
   * @param int Bl - blue colour level.
   */

  public Sphere(int Rl, int Gl, int Bl) {
    this.Rl = Rl;
    this.Gl = Gl;
    this.Bl = Bl;
  }


  /****************************/
  /* Accessor/Mutator Methods */
  /****************************/

  /**
   * Accessor for the red colour level of the sphere.
   * @return int Rl - the red colour level from 0 to 255.
   */

  public int getRedLevel() {
    return Rl;
  }

  /**
   * Mutator for the red colour level of the sphere.
   * @param int l - the new colour level from 0 to 255
   */

  public void setRedLevel(int l) {
    if ((l >=0) && (l < 255))
      this.Rl = l;
    Setup();
  }

  /**
   * Accessor for the green colour level of the sphere.
   * @return int Gl - the green colour level from 0 to 255.
   */

  public int getGreenLevel() {
    return Gl;
  }

  /**
   * Mutator for the green colour level of the sphere.
   * @param int l - the new colour level from 0 to 255
   */

  public void setGreenLevel(int l) {
    if ((l >=0) && (l < 255))
      this.Gl = l;
    Setup();
  }

  /**
   * Accessor for the blue colour level of the sphere.
   * @return int Bl - the blue colour level from 0 to 255.
   */

  public int getBlueLevel() {
    return Bl;
  }

  /**
   * Mutator for the blue colour level of the sphere.
   * @param int l - the new colour level from 0 to 255
   */

  public void setBlueLevel(int l) {
    if ((l >=0) && (l < 255))
      this.Bl = l;
    Setup();
  }


  /******************/
  /* Public Methods */
  /******************/

  /**
   * Set the applet for the sphere objects.
   * @param Applet app - the applet.
   */

  public static void setApplet(Applet app) {
    applet = app;
  }

  /**
   * Paint the spheres or balls.
   * @param Graphics gc - the graphics context for the paint.
   * @param int x - the x coordinate of the sphere centre.
   * @param int y - the y coordinate of the sphere centre.
   * @param int r - the radius of the sphere
   */

  public void paint(Graphics gc, int x, int y, int r) {
    Image ba[] = balls;
    if (ba == null) {
      Setup();
      ba = balls;
    }
    Image i = ba[r];
    int size = 10 + r;
    gc.drawImage(i, x - (size >> 1), y - (size >> 1), size, size, applet);
  }


  /*******************/
  /* Private Methods */
  /*******************/

  /**
   *
   */

  private void Setup() {
    balls = new Image[nBalls];
    byte red[] = new byte[256];
    red[0] = (byte) bgGrey;
    byte green[] = new byte[256];
    green[0] = (byte) bgGrey;
    byte blue[] = new byte[256];
    blue[0] = (byte) bgGrey;
    for (int r = 0; r < nBalls; r++) {
      float b = (float) (r+1) / nBalls;
      for (int i = maxr; i >= 1; --i) {
        float d = (float) i / maxr;
        red[i] = (byte) blend(blend(Rl, 255, d), bgGrey, b);
        green[i] = (byte) blend(blend(Gl, 255, d), bgGrey, b);
        blue[i] = (byte) blend(blend(Bl, 255, d), bgGrey, b);
      }
      IndexColorModel model = new IndexColorModel(8, maxr + 1,
      red, green, blue, 0);
      balls[r] = applet.createImage(
        new MemoryImageSource(R*2, R*2, model, data, 0, R*2));
    }
  }

  /**
   * Blend the sphere surfaces.
   */

  private final int blend(int fg, int bg, float fgfactor) {
    return (int) (bg + (fg - bg) * fgfactor);
  }


}  // End of Sphere Class
