/*
 * @(#)SphereModel.java
 * Portions Copyright (c) 1994-1996 Sun Microsystems, Inc. All Rights Reserved.
 */

import java.applet.*;
import java.awt.*;
import java.awt.image.*;
import java.util.Hashtable;


/**
 * The representation of a spherical .xyz model
 */

class SphereModel extends Model3D {


  /**********************/
  /* Instance Variables */
  /**********************/

  /**
   * Array of spheres in the model.
   */

  Sphere spheres[];

  /**
   * Array for z-buffer visibility algorithm.
   */

  private int ZsortMap[];

  /**
   * Hashtable for sphere types (different colours).
   */

  public static Hashtable sphereTable = new Hashtable();

  /**
   * Default sphere (black).
   */

  public static Sphere defaultSphere;

  static {
    sphereTable.put("green", new Sphere(0, 255, 0));
    sphereTable.put("red", new Sphere(255, 0, 0));
    sphereTable.put("lightGrey", new Sphere(210, 210, 210));
    defaultSphere = new Sphere(0, 0, 0);
  }


  /***********************/
  /* Constructor Methods */
  /***********************/

  /**
   * Construct a new sphere model.
   */

  public SphereModel() {
    mat = new Matrix3D();
  }

  /**
   * Construct a new sphere model with a specified initial rotation.
   * @param double xTheta - degrees to rotate model about x-axis.
   * @param double yTheta - degrees to rotate model about y-axis.
   * @param double zTheta - degrees to rotate model about z-axis.
   */

  public SphereModel(double xTheta, double yTheta, double zTheta) {
    mat = new Matrix3D();
    mat.xrot(xTheta);
    mat.yrot(yTheta);
    mat.zrot(zTheta);
  }

  /******************/
  /* Public Methods */
  /******************/

  /**
   * Add a vertex (sphere) to this model.
   * @param String name - colour of sphere.
   * @param float x - x coordinate of sphere centre.
   * @param float y - y coordinate of sphere centre.
   * @param float z - z coordinate of sphere centre.
   */

  public int addVert(String name, float x, float y, float z) {
    int i = nvert;
    if (i >= maxvert)
      if (vert == null) {
        maxvert = 100;
        vert = new float[maxvert * 3];
        spheres = new Sphere[maxvert];
      } else {
        maxvert *= 2;
        float nv[] = new float[maxvert * 3];
        System.arraycopy(vert, 0, nv, 0, vert.length);
        vert = nv;
        Sphere na[] = new Sphere[maxvert];
        System.arraycopy(spheres, 0, na, 0, spheres.length);
        spheres = na;
      }
    Sphere a = (Sphere) sphereTable.get(name.toLowerCase());
    if (a == null) a = defaultSphere;
    spheres[i] = a;
    i *= 3;
    vert[i] = x;
    vert[i + 1] = y;
    vert[i + 2] = z;
    return nvert++;
  }

  /**
   * Paint this model to a graphics context. It uses the matrix associated
   * with this model to map from model space to screen space.
   * @param Graphics g - the painting graphics context.
   */

  public void paint(Graphics g) {
    if (vert == null || nvert <= 0)
      return;
    transform();
    int v[] = tvert;
    int zs[] = ZsortMap;
    if (zs == null) {
      ZsortMap = zs = new int[nvert];
      for (int i = nvert; --i >= 0;)
      zs[i] = i * 3;
    }

    for (int i = nvert - 1; --i >= 0;) {
      boolean flipped = false;
      for (int j = 0; j <= i; j++) {
        int a = zs[j];
        int b = zs[j + 1];
        if (v[a + 2] > v[b + 2]) {
          zs[j + 1] = a;
          zs[j] = b;
          flipped = true;
        }
      }
      if (!flipped)
        break;
    }

    int lg = 0;
    int lim = nvert;
    Sphere ls[] = spheres;
    if (lim <= 0 || nvert <= 0)
      return;
    for (int i = 0; i < lim; i++) {
      int j = zs[i];
      int grey = v[j + 2];
      if (grey < 0)
        grey = 0;
      if (grey > 15)
        grey = 15;
      spheres[j/3].paint(g, v[j], v[j + 1], grey);
    }
  }

}// End of SphereModel Class

