/**
 * This class implements a training panel that consists of some input
 * textfields, buttons, and a progress bar component. In order to use this
 * class, the parent object must implement the following methods:
 * train(), step(), test() and stopTraining().
 *
 * @author  Fred Corbett
 * @version January 2, 1997
 */

import java.awt.*;
import java.awt.event.*;
import java.applet.*;

// Import the MicroLine classes (used for the progress bar component)

import mlsoft.mct.*;

class TrainingPanel extends Panel {


  /**********************/
  /* Instance Variables */
  /**********************/

  /**
   * The parent applet.
   */

  private PerceptronApplet pa;

  /**
   * The width of the training panel in pixels.
   */

  private int width;

  /**
   * The height of the training panel in pixels.
   */

  private int height;


  /***************************/
  /* AWT Component Variables */
  /***************************/

  /**
   * Textfield for the perceptron learning rate. Usually between 0 and 1.
   */

  private TextField learningRate;

  /**
   * Textfield for the number of training iterations. Each iteration cycles
   * through all 4 input vectors.
   */

  private TextField iterations;

  /**
   * Textfield for the error threshold. The neuron output must be within
   * this value for output to be considered correct.
   */

  private TextField errorThreshold;

  /**
   * Button to start perceptron training. Sends train() message to parent
   * class.
   */

  private Button trainButton;

  /**
   * Button for step training. Sends step() to parent class.
   */

  private Button stepButton;

  /**
   * Button to stop training. Sends stopTraining() to parent class.
   */

  private Button stopButton;

  /**
   * Button to test traning results. Sends test() to parent class.
   */

  private Button testButton;

  /**
   * The Microline toolkit progress bar component.
   */

  private MlProgress progressBar;


  /***********************/
  /* Constructor Methods */
  /***********************/

  /**
   * Instantiate a new training panel.
   * @param PerceptronApplet pa - the parent class.
   * @param int w - the width of this panel.
   * @param int h - the height of this panel.
   */

  public TrainingPanel(PerceptronApplet pa, int w, int h) {
    super();
    this.pa = pa;
    width = w;
    height = h;
    initComponents();
  }


  /****************************/
  /* Accessor/Mutator Methods */
  /****************************/

  /**
   * Accessor for the panel width.
   * @return int - the width of this panel in pixels.
   */

  public int getWidth() {
    return width;
  }

  /**
   * Accessor for the panel height.
   * @return int - the height of this panel in pixels.
   */

  public int getHeight() {
    return height;
  }

  /**
   * Accessor for contents of learning rate textfield.
   * @return float - the learning rate.
   */

  public float getLearningRate() {
    return pa.getFieldAsFloat(learningRate);
  }

  /**
   * Mutator for the contents of the learning rate textfield.
   * @param float lr - the new learning rate.
   */

  public void setLearningRate(float lr) {
    pa.setFieldToFloat(learningRate,lr);
  }

  /**
   * Accessor for contents of iterations textfield.
   * @return int - the number of iterations.
   */

  public int getIterations() {
    return pa.getFieldAsInt(iterations);
  }

  /**
   * Mutator for the contents of the iterations textfield.
   * @param int i - the new number of iterations.
   */

  public void setIterations(int i) {
    pa.setFieldToInt(iterations,i);
  }

  /**
   * Accessor for contents of error threshold textfield.
   * @return float - the error threshold.
   */

  public float getErrorThreshold() {
    return pa.getFieldAsFloat(errorThreshold);
  }

  /**
   * Mutator for the contents of the error threshold textfield.
   * @param float et - the new error threshold.
   */

  public void setErrorThreshold(float et) {
    pa.setFieldToFloat(errorThreshold,et);
  }

  /**
   * Accessor for the train button
   * @return Button - the train button.
   */

  public Button trainButton() {
    return trainButton;
  }

  /**
   * Accessor for the step button.
   * @return Button - the step button.
   */

  public Button stepButton() {
    return stepButton;
  }

  /**
   * Accessor for the stop button.
   * @return Button - the stop button.
   */

  public Button stopButton() {
    return stopButton;
  }

  /**
   * Accessor for the test button.
   * @return Button - the test button.
   */

  public Button testButton() {
    return testButton;
  }

  /**
   * Accessor for the progress bar component.
   * @return MlProgress - the progress bar component.
   */

  public MlProgress progressBar() {
    return progressBar;
  }

  /******************/
  /* Public Methods */
  /******************/

  /**
   * Respond to user interface events that occur over this panel.
   * @param Event evt - the event data object.
   * @return True if the event is handled here.
   */

    /* public boolean handleEvent(Event evt) {
    switch (evt.id) {
      case Event.ACTION_EVENT:
        if (evt.target == trainButton) {
          clickedTrainButton();
          return true;
        } else
        if (evt.target == stepButton) {
          clickedStepButton();
          return true;
        } else
        if (evt.target == stopButton) {
          clickedStopButton();
          return true;
        } else
        if (evt.target == testButton) {
          clickedTestButton();
          return true;
        }
        break;
    }
    return super.handleEvent(evt);
    }*/

  /**
   * Disable the panel's AWT components (except stop button).
   */

  public void disableComponents() {
    learningRate.setEnabled(false);
    iterations.setEnabled(false);
    errorThreshold.setEnabled(false);
    trainButton.setEnabled(false);
    stepButton.setEnabled(false);
    stopButton.setEnabled(true);
    testButton.setEnabled(false);
  }

  /**
   * Enable the panel's AWT components (except stop button).
   */

  public void enableComponents() {
    learningRate.setEnabled(true);
    iterations.setEnabled(true);
    errorThreshold.setEnabled(true);
    trainButton.setEnabled(true);
    stepButton.setEnabled(true);
    stopButton.setEnabled(false);
    testButton.setEnabled(true);
  }


  /*******************/
  /* Private Methods */
  /*******************/

  /**
   * Initialize the layout manager and components.
   */

  private void initComponents() {
    // Initialize the layout manager and constraints object
    GridBagLayout gridbag = new GridBagLayout();
    GridBagConstraints c = new GridBagConstraints();
    setLayout(gridbag);

    // Add the first row of components
    Label l1 = new Label("Learning rate:");
    c.anchor = GridBagConstraints.WEST;
    c.gridwidth = 3;
    c.weightx = 1.0;
    c.weighty = 1.0;
    gridbag.setConstraints(l1, c);
    add(l1);

    learningRate = new TextField("0.05", 6);
    c.anchor = GridBagConstraints.CENTER;
    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints(learningRate, c);
    add(learningRate);

    // Add the second row of components
    Label l2 = new Label("Number of iterations:");
    c.anchor = GridBagConstraints.WEST;
    c.gridwidth = 3;
    gridbag.setConstraints(l2, c);
    add(l2);

    iterations = new TextField("20", 6);
    c.anchor = GridBagConstraints.CENTER;
    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints(iterations, c);
    add(iterations);

    // Add the third row of components
    Label l3 = new Label("Error threshold:");
    c.anchor = GridBagConstraints.WEST;
    c.gridwidth = 3;
    gridbag.setConstraints(l3, c);
    add(l3);

    errorThreshold = new TextField("0.1", 6);
    c.anchor = GridBagConstraints.CENTER;
    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints(errorThreshold, c);
    add(errorThreshold);

    // Add the training/testing buttons
    trainButton = new Button("Train");
    ActionListener trainListener = new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	    clickedTrainButton();
	}
    };
    trainButton.addActionListener(trainListener);
    c.gridwidth = 1;
    gridbag.setConstraints(trainButton,c);
    add(trainButton);

    stepButton = new Button("Step");
    ActionListener stepListener = new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	    clickedStepButton();
	}
    };
    stepButton.addActionListener(stepListener);
    gridbag.setConstraints(stepButton,c);
    add(stepButton);

    stopButton = new Button("Stop");
    ActionListener stopListener = new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	    clickedStopButton();
	}
    };
    stopButton.addActionListener(stopListener);
    gridbag.setConstraints(stopButton,c);
    add(stopButton);
    stopButton.setEnabled(false);

    testButton = new Button("Test");
    ActionListener testListener = new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	    clickedTestButton();
	}
    };
    testButton.addActionListener(testListener);
    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints(testButton,c);
    add(testButton);
    testButton.setEnabled(false);

    // Add the progress bar component
    progressBar = new MlProgress();
    progressBar.setSize(350,30);
    c.gridwidth = 4;
    c.fill = GridBagConstraints.BOTH;
    gridbag.setConstraints(progressBar,c);
    add(progressBar);
    progressBar.setBackground(Color.white);
    setSize(width,height);
  }

  /**
   * The user clicked the trainButton, send a message to the parent class.
   */

  private void clickedTrainButton() {
    pa.train();
  }

  /**
   * Respond to user clicking the stepButton.
   */

  private void clickedStepButton() {
    pa.step();
  }

  /**
   * Respond to user clicking the stop button. stop() is not used as a
   * method name in the parent class since it would override a standard
   * Applet method.
   */

  private void clickedStopButton() {
    pa.stopTraining();
  }

  /**
   * Respond to clicking the test button.
   */

  private void clickedTestButton() {
    pa.test();
  }

}  // End of TrainingPanel Class
