/**
 * This panel displays and manages a 2 input truth table. This table allows
 * a user to set one of 16 possible Boolean functions through its AWT buttons.
 *
 * NOTE: To use this class, the instantiating class must provide the
 * method buttonHasChanged(int, float) to allow changes in the truth table
 * button boxes to be passed to the parent object.
 *
 * @author  Fred Corbett
 * @version February 5, 1997
 */

import java.awt.*;
import java.awt.event.*;
import java.util.Vector;

class TruthTablePanel extends Panel {


  /**********************/
  /* Instance Variables */
  /**********************/

  /**
   * Reference to parent applet.
   */

  private PerceptronApplet pa;

  /**
   * The width of this canvas in pixels.
   */

  private int width;

  /**
   * The height of this canvas in pixels.
   */

  private int height;

  /**
   * Array containing float version of current button box selections.
   */

  private float[] buttonValues = {0f,0f,0f,0f};


  /***************************/
  /* AWT Component Variables */
  /***************************/

  /**
   * Button for X2 = 0, X1 = 0.
   */

  Button output00;

  /**
   * Button for X2 = 0, X1 = 1.
   */

  Button output01;

  /**
   * Button for X2 = 1, X1 = 0.
   */

  Button output10;

  /**
   * Button for X2 = 1, X1 = 1.
   */

  Button output11;


  /***********************/
  /* Constructor Methods */
  /***********************/

  /**
   * Default constructor for the panel.
   * @param PerceptronApplet sp - reference to parent applet.
   * @param int w - the width of the canvas.
   * @param int h - the height of the canvas.
   */

  public TruthTablePanel(PerceptronApplet pa, int w, int h) {
    super();
    this.pa = pa;
    width = w;
    height = h;
    initComponents();
  }


  /****************************/
  /* Accessor/Mutator Methods */
  /****************************/

  /**
   * Accessor for buttonValues.
   * @return float[] - the array of button box values converted to floats.
   */

  public float[] buttonValues() {
    return buttonValues;
  }


  /******************/
  /* Public Methods */
  /******************/

  /**
   * Paint this canvas.
   * @param Graphics g - the canvas graphics context.
   */

  public void paint(Graphics g) {
    int gridX,gridY;

    gridX = 2 * width / 3;
    gridY = height / 5;

    // Draw the truth table axes
    g.setColor(Color.black);
    g.drawLine(5,gridY,width-5,gridY);
    g.drawLine(gridX,5,gridX,height-5);
  }

  /**
   * Update this canvas. Overide so canvas is not erased before a paint.
   * @param Graphics g - the canvas graphics context.
   */

  public void update(Graphics g) {
    paint(g);
  } 

  /**
   * Disable the button box components.
   */

  public void disableComponents() {
    output00.setEnabled(false);
    output01.setEnabled(false);
    output10.setEnabled(false);
    output11.setEnabled(false);
  }

  /**
   * Enable the button box components.
   */

  public void enableComponents() {
    output00.setEnabled(true);
    output01.setEnabled(true);
    output10.setEnabled(true);
    output11.setEnabled(true);
  }


  /*******************/
  /* Private Methods */
  /*******************/

  /**
   * Initialize the layout manager, add the components, and update the tab
   * chain.
   */

  private void initComponents() {
    // Initialize the layout manager and constraint object
    GridBagLayout gridbag = new GridBagLayout();
    GridBagConstraints c = new GridBagConstraints();
    setLayout(gridbag);

    // Add the input/output truth table row by row
    // Add the first row
    Label l1 = new Label("x2");
    c.anchor = GridBagConstraints.CENTER;
    c.gridwidth = 1;
    c.insets = new Insets(5,5,5,5);
    c.weightx = 1.0;
    c.weighty = 1.0;
    gridbag.setConstraints(l1, c);
    add(l1);

    Label l2 = new Label("x1");
    gridbag.setConstraints(l2, c);
    add(l2);

    Label l3 = new Label("y");
    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints(l3, c);
    add(l3);

    // Add the second row
    Label l4 = new Label("0");
    c.gridwidth = 1;
    gridbag.setConstraints(l4, c);
    add(l4);

    Label l5 = new Label("0");
    gridbag.setConstraints(l5, c);
    add(l5);

    output00 = new Button(" 0 ");
    ActionListener output00Listener = new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	    updateButtonValues(0,(Button)e.getSource());
	}
    };
    output00.addActionListener(output00Listener);
    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints(output00, c);
    add(output00);

    // Add the third row
    Label l6 = new Label("0");
    c.gridwidth = 1;
    gridbag.setConstraints(l6, c);
    add(l6);

    Label l7 = new Label("1");
    gridbag.setConstraints(l7, c);
    add(l7);

    output01 = new Button(" 0 ");
    ActionListener output01Listener = new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	    updateButtonValues(1,(Button)e.getSource());
	}
    };
    output01.addActionListener(output01Listener);
    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints(output01, c);
    add(output01);

    // Add the fourth row
    Label l8 = new Label("1");
    c.gridwidth = 1;
    gridbag.setConstraints(l8, c);
    add(l8);

    Label l9 = new Label("0");
    gridbag.setConstraints(l9, c);
    add(l9);

    output10 = new Button(" 0 ");
    ActionListener output10Listener = new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	    updateButtonValues(2,(Button)e.getSource());
	}
    };
    output10.addActionListener(output10Listener);
    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints(output10, c);
    add(output10);

    // Add the fifth and final row
    Label l10 = new Label("1");
    c.gridwidth = 1;
    gridbag.setConstraints(l10, c);
    add(l10);

    Label l11 = new Label("1");
    gridbag.setConstraints(l11, c);
    add(l11);

    output11 = new Button(" 0 ");
    ActionListener output11Listener = new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	    updateButtonValues(3,(Button)e.getSource());
	}
    };
    output11.addActionListener(output11Listener);
    c.gridwidth = GridBagConstraints.REMAINDER;
    gridbag.setConstraints(output11, c);
    add(output11);

    validate();
    setSize(width,height);
  }

  /**
   * Update the contents of the buttonValues array.
   */   
    private void updateButtonValues(int nbButton, Button button) {
	if (button.getLabel().equals(" 0 ")) {
	    button.setLabel(" 1 ");
	    buttonValues[nbButton] = 1f;
	} else {
	    button.setLabel(" 0 ");
	    buttonValues[nbButton] = 0f;
	}
	
	// Send message to parent applet that a button has changed.
	pa.buttonHasChanged(nbButton, buttonValues[nbButton]);
    }
    
} // End of TruthTablePanel Class
