/*
 * @(#)WireFrameModel.java
 *
 * This is a modified version of the Model3D sample applet distributed with the JDK.
 * Portions Copyright (c) 1994-1996 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * @author  Fred Corbett
 * @version February 4, 1997
 */

import java.applet.Applet;
import java.awt.*;

class WireFrameModel extends Model3D {

  /**********************/
  /* Instance Variables */
  /**********************/

  /**
   * Array of connection data.
   */

  private int con[];

  /**
   * Number of connections.
   */

  private int ncon;

  /**
   * Maximum number of connections.
   */

  private int maxcon;

  /**
   * Array of scaled point colours.
   */

  private Color colours[];
  

  /***********************/
  /* Constructor Methods */
  /***********************/

  /**
   * Construct a new wire frame model using default colour and rotations.
   */

  public WireFrameModel() {
    this(Color.gray,0,0,0);
  }

  /** Construct a new wire frame model with a specified colour.
   * @param Color c - colour of the wire frame model points.
   */

  public WireFrameModel(Color c) {
    this(c, 0, 0, 0);
  }

  /**
   * Construct a new wire frame model with a specified initial rotation.
   * @param double xTheta - degrees to rotate model about x-axis.
   * @param double yTheta - degrees to rotate model about y-axis.
   * @param double zTheta - degrees to rotate model about z-axis.
   */

  public WireFrameModel(double xTheta, double yTheta, double zTheta) {
    this(Color.gray,xTheta,yTheta,zTheta);
  }

  /**
   * Construct a new wire frame model with a specified colour and initial rotation.
   * @param Color c - colour of the wire frame model points.
   * @param double xTheta - degrees to rotate model about x-axis.
   * @param double yTheta - degrees to rotate model about y-axis.
   * @param double zTheta - degrees to rotate model about z-axis.
   */

  public WireFrameModel(Color c, double xTheta, double yTheta, double zTheta) {
    mat = new Matrix3D();
    mat.xrot(xTheta);
    mat.yrot(yTheta);
    mat.zrot(zTheta);
    colours = new Color[16];
    setColor(c);
  }    


  /******************/
  /* Public Methods */
  /******************/

  /**
   * Add a vertex to this model.
   * @param float x - x coordinate of vertex in real space.
   * @param float y - y coordinate of vertex in real space.
   * @param float z - z coordinate of vertex in real space.
   */

  public int addVert(float x, float y, float z) {
    int i = nvert;
    if (i >= maxvert)
      if (vert == null) {
        maxvert = 100;
        vert = new float[maxvert * 3];
      } else {
        maxvert *= 2;
        float nv[] = new float[maxvert * 3];
        System.arraycopy(vert, 0, nv, 0, vert.length);
        vert = nv;
      }
    i *= 3;
    vert[i] = x;
    vert[i + 1] = y;
    vert[i + 2] = z;
    return nvert++;
  }

  /**
   * Add a line from vertex p1 to vertex p2.
   * @param int p1 - start vertex of line.
   * @param int p2 - end vertex of line.
   */

  public void addLine(int p1, int p2) {
    int i = ncon;
    if (p1 >= nvert || p2 >= nvert)
      return;
    if (i >= maxcon)
      if (con == null) {
        maxcon = 100;
        con = new int[maxcon];
      } else {
        maxcon *= 2;
        int nv[] = new int[maxcon];
        System.arraycopy(con, 0, nv, 0, con.length);
        con = nv;
      }
    if (p1 > p2) {
      int t = p1;
      p1 = p2;
      p2 = t;
    }
    con[i] = (p1 << 16) | p2;
    ncon = i + 1;
  }

  /**
   *
   */

  public void removeAll() {
    nvert = 0;
    ncon = 0;
  }

  /**
   * Eliminate duplicate lines.
   */

  public void compress() {
    int limit = ncon;
    int c[] = con;
    quickSort(con, 0, ncon - 1);
    int d = 0;
    int pp1 = -1;
    for (int i = 0; i < limit; i++) {
      int p1 = c[i];
      if (pp1 != p1) {
        c[d] = p1;
        d++;
      }
      pp1 = p1;
    }
    ncon = d;
  }



  /**
   * Paint this model to a graphics context.  It uses the matrix associated
   * with this model to map from model space to screen space.
   * @param Graphics g - the graphics context to be painted.
   */

  public void paint(Graphics g) {
    if (vert == null || nvert <= 0)
      return;
    transform();
    

    int lg = 0;
    int lim = ncon;
    int c[] = con;
    int v[] = tvert;
    if (lim <= 0 || nvert <= 0)
      return;
    for (int i = 0; i < lim; i++) {
      int T = c[i];
      int p1 = ((T >> 16) & 0xFFFF) * 3;
      int p2 = (T & 0xFFFF) * 3;
      int grey = v[p1 + 2] + v[p2 + 2];
      if (grey < 0)
        grey = 0;
      if (grey > 15)
        grey = 15;
      if (grey != lg) {
        lg = grey;
        g.setColor(colours[grey]);
      }
      g.drawLine(v[p1], v[p1 + 1],v[p2], v[p2 + 1]);
    }
  }

  /*******************/
  /* Private Methods */
  /*******************/


  /**
   *
   * @param Color c
   */

  private void setColor(Color c) {  
    if (c == Color.blue) {
      for (int i = 0; i < 16; i++) {
        //int blue = (int) (170*(1-Math.pow(i/15.0, 2.3)));
        //colours[i] = new Color(blue, blue, blue);
        colours[i] = Color.blue; 
      }
    } else if (c == Color.black) {
      for (int i = 0; i < 16; i++) {
        //int blue = (int) (170*(1-Math.pow(i/15.0, 2.3)));
        //colours[i] = new Color(blue, blue, blue);
        colours[i] = Color.black;
      }
    } else {
      for (int i = 0; i < 16; i++) {
        int grey = (int) (170*(1-Math.pow(i/15.0, 2.3)));
        colours[i] = new Color(grey, grey, grey);
      }
    }
  }


  /**
   * Quick Sort implementation
   * @param int a[] - 
   * int left - 
   * int right - 
   */

  private void quickSort(int a[], int left, int right) {
    int leftIndex = left;
    int rightIndex = right;
    int partionElement;
    if (right > left) {
      partionElement = a[(left + right) / 2];
      while(leftIndex <= rightIndex) {
        while((leftIndex < right) && (a[leftIndex] < partionElement))
          ++leftIndex;
        while((rightIndex > left) && (a[rightIndex] > partionElement))
          --rightIndex;
        if(leftIndex <= rightIndex) {
          swap(a, leftIndex, rightIndex);
          ++leftIndex;
          --rightIndex;
        }
      }
      if(left < rightIndex)
        quickSort(a, left, rightIndex);
      if(leftIndex < right)
        quickSort( a, leftIndex, right );
     }
   }


   /**
    * Swap elements i and j from the array a[].
    * @param int a[] - array of values being sorted.
    * @param int i - index of first element.
    * @param int j - index of second element.
    */

   private void swap(int a[], int i, int j) {
     int temp = a[i];
     a[i] = a[j];
     a[j] = temp;
   }

} // End of WireFrameModel Class