/**
 * This class is used to implement the activation function of a neuron.
 * Currently, this class implements 5 typical functions used in neural
 * networks: unit step, sigmoid, piecewise linear, and Gaussian. There are
 * variables/methods that allow some function parameters to be modified.
 *
 * @author  Fred Corbett
 * @version January 2, 1997
 * modified Feb. 13, 1998 (Alix Herrmann) to add the identity function.
 */

package anns;

public class ActivationFunction {


  /*******************/
  /* Class Variables */
  /*******************/

  /**
   * Equality epsilon; used for floating-point equality testing.
   */

  public final static float EE = 1e-10f;

  /**
   * Constant: number of function types.
   */

  public final static byte NUM_FUNCTIONS = 5;

  /**
   * Constant: code for unit step function.
   */

  public final static byte UNIT_STEP = 0;

  /**
   * Constant: code for sigmoid function.
   */

  public final static byte SIGMOID = 1;

  /**
   * Constant: code for piecewise linear function.
   */

  public final static byte PIECEWISE_LINEAR = 2;

  /**
   * Constant: code for Gaussian function.
   */

  public final static byte GAUSSIAN = 3;

  /**
   * Constant: code for Identity function.
   */

  public final static byte IDENTITY = 4;


  /**********************/
  /* Instance Variables */
  /**********************/

  /**
   * Current function type. Eg. if functionType = UNIT_STEP, getOutput()
   * will return the output of a unit step function given an input.
   */

  private byte functionType;

  /**
   * Sigmoid beta or slope parameter.
   */

  private float beta;

  /**
   * Piecewise linear slope parameter.
   */

  private float slope;

  /**
   * The value of x for the piecewise linear function where y = 0.
   */

  private float xLow;

  /**
   * The value of x for the piecewise linear function where y = 1.
   */

  private float xHigh;

  /**
   * Gaussian mu or mean parameter.
   */

  private float mu;

  /**
   * Gaussian sigma or standard deviation parameter.
   */

  private float sigma;

  /**
   * Gaussian constant term. It is calculated using the current setting
   * for sigma and mu in order to save calculation.
   */

  private float k1;

  /**
   * Gaussian constant term.
   */

  private float k2;


  /***********************/
  /* Constructor Methods */
  /***********************/

  /**
   * The function type is unit step by default.
   */

  public ActivationFunction() {
    init(UNIT_STEP);
  }

  /**
   * The function type can be specified.
   * @param fcnType - integer code designating type of function.
   */

  public ActivationFunction(byte fcnType) {
    init(fcnType);
  }


  /******************************/
  /* Accessor / Mutator Methods */
  /******************************/

  /**
   * Accessor for function type code.
   * @return short - the code for the function type.
   */

  public short functionType() {
    return functionType;
  }


  /**
   * Accessor for sigmoid beta parameter.
   * @return float - the value of beta.
   */

  public float beta() {
    return beta;
  }

  /**
   * Mutator for sigmoid beta parameter.
   * @param float b - the new value for beta.
   */

  public void beta(float b) {
    beta = b;
  }

  /**
   * Accessor for piecewise linear slope parameter.
   * @return float - the value of beta.
   */

  public float slope() {
    return slope;
  }

  /**
   * Mutator for piecewise linear slop parameter.
   * @param float s - the new value for slope.
   */

  public void slope(float s) {
    // Take care of negative and zero slopes
    slope = Math.abs(s);
    if (slope < EE)
      slope = 1f;
    // Calculate the xLow and xHigh parameters
    xLow = -0.5f / slope;
    xHigh = Math.abs(xLow);
  }

  /**
   * Accessor for Gaussian mu (mean) parameter.
   * @return float - the value of mu.
   */

  public float mu() {
    return mu;
  }

  /**
   * Mutator for Gaussian mu parameter.
   * @param float m - the new value for mu.
   */

  public void mu(float m) {
    mu = m;
  }

  /**
   * Accessor for Gaussian sigma (std. dev.) parameter.
   * @return float - the value of sigma.
   */

  public float sigma() {
    return sigma;
  }

  /**
   * Mutator for Gaussian sigma parameter.
   * @param float s - the new value for sigma.
   * @exception ArithmeticException - for division by zero.
   */

  public void sigma(float s) {
    // sigma has changed, so update the constant factors k1 and k2
    try {
      k1 = (float) (1.0 / Math.sqrt(2 * Math.PI * s));
      k2 = (float) 2.0 * s * s;
      sigma = s;
    }
    catch (ArithmeticException e) {

    }
  }


  /******************/
  /* Public Methods */
  /******************/

  /**
   * Initializes the ActivationFunction.
   * @param fcnType - integer code designating type of function.
   */

  public void init(byte fcnType) {
    // Set default function parameters
    beta = 1f;
    slope = 1f;
    xLow = -0.5f;
    xHigh = 0.5f;
    mu = 0f;
    sigma(0.159155f);
    changeFunctionTypeTo(fcnType);
  }

  /**
   * Changes the activation function to the specified function type. If an
   * illegal type is specified, UnitStep is used.
   * @param byte fcnType - integer code for the function type.
   */

  public void changeFunctionTypeTo(byte fcnType) {
    if (fcnType > -1 && fcnType < NUM_FUNCTIONS) {
      functionType = fcnType; 
    } else {
      functionType = UNIT_STEP; 
    }
  }

  /**
   * Returns the output of the activation function for a specified input
   * @param x - the input value.
   * @return float - the corresponding value of y.
   */

  public float getOutput(float x) {
    float temp;

    switch (functionType) {
      case UNIT_STEP:
        temp = unitStep(x);
        break;
      case SIGMOID:
        temp = sigmoid(x);
        break;
      case PIECEWISE_LINEAR:
        temp = piecewiseLinear(x);
        break;
      case GAUSSIAN:
        temp = gaussian(x);
        break;
      case IDENTITY:
        temp = identity(x);
        break;
      default:
        temp = unitStep(x);
        break;
    }
    return temp;
  }


  /*******************/
  /* Private Methods */
  /*******************/

 /**
   * Returns the input value.  Useful for simulating a linear associator with 
   * no nonlinearity.
   * @param x - the input value.
   * @return float - the corresponding value of y.
   */

  private float identity (float x) {
      return x;
  }

  /**
   * Returns the output value of a unit step function for a given input.
   * @param x - the input value.
   * @return float - the corresponding value of y.
   */

  private float unitStep(float x) {
    if (x < 0)
      return 0f;
    else
      return 1f;
  }

  /**
   * Returns the output value of a sigmoid function for a given input.
   * The function is given by the following equation:
   * <PRE>
   *                  1
   *  f(x) = ------------------
   *         (1 + e^(-beta * x))
   * </PRE>
   * @param x - the input value.
   * @return float - the corresponding value of y.
   * @exception ArithmeticException - for division by zero
   */

  private float sigmoid(float x) {
    float den;

    den = (float) (1 + Math.exp(-beta * x));
    try {
      return (1/den);
    }
    catch (ArithmeticException e) {
      return 1f;
    }
  }

  /**
   * Returns the output value of a piecewise linear function for a given
   * input.
   * @param x - the input value.
   * @return float - the corresponding value of y.
   */

  private float piecewiseLinear(float x) {
    if (x < xLow)
      return 0f;
    else if (x > xHigh)
      return 1f;
    else
      return (float) (slope * x + 0.5);
  }

  /**
   * Returns the output value of a unit-aread Gaussian function for a given
   * input. The Gaussian or normal probability function is given by:
   * <PRE>
   *                 1                -(x-mu)^2 / (2 * sigma^2)
   *  f(x) =  ------------------- *  e
   *          (2 * pi * sigma)^0.5
   *
   * </PRE>
   * The variable k1 holds the value of the first term, and k2 holds the
   * value of the (2 * simga^2) term for the current value of sigma.
   * @param x - the input value.
   * @return float - the corresponding value of y.
   * @exception ArithmeticException - for division by zero.
   */

  private float gaussian(float x) {
    double num;

    num = -1 * Math.pow((x - mu),2);
    try {
      return (float) (k1 * Math.exp(num / k2));
    }
    catch (ArithmeticException e) {
      return 1f;
    }
  }

} // End of ActivationFunction Class







