/**
  * This class provides the basic data model and function for an artificial
  * neuron.
  *
  * @author  Fred Corbett
  * @version January 2, 1997
  */

package anns;

public class ArtificialNeuron extends Object {


  /*******************/
  /* Class Variables */
  /*******************/

  /**
   * Default degree of a neuron. The degree is the number of inputs to the
   * neuron excluding the threshold.
   */

  public final static int DEFAULT_DEGREE = 2;

  /**
   * Minimum degree of a neuron. Allow for at least one input.
   */

  public final static int MIN_DEGREE = 1;

  /**
   * Maximum degree of a neuron. Seems like a nice number!
   */

  public final static int MAX_DEGREE = 100;

  /**
   * Multiplier for threshold/bias value.
   */

  public final static float THRESHOLD_INPUT = -1;


  /**********************/
  /* Instance Variables */
  /**********************/

   /**
    * Weight values w[0],w[1],...,w[n-1].
    */

   protected float w[];

   /**
    * Theshold/bias value of the neuron. This value is added to the weighted
    * sum of the inputs.
    */

   protected float th;

   /**
    * Weighted sum of the neuron, ie. the sum of all the weighted inputs
    * that is the input to the activation function.
    */

   protected float ws;

   /**
    * The total output of this neuron.
    */

   protected float y;

   /**
    * Degree of this neuron.
    */

   protected int n;

   /**
    * Activation function of the neuron.
    */

   protected ActivationFunction f;


   /***********************/
   /* Constructor Methods */
   /***********************/

   /**
    * Instantiate a new neuron of default degree using a unit step
    * activation function.
    */

   public ArtificialNeuron() {
     this(DEFAULT_DEGREE,(new ActivationFunction(ActivationFunction.SIGMOID)));
   }

  /**
   * Degree must be specified, activation function is the unit
   * step function by default.
   * @param degree - the number of input/weight pairs of the neuron.
   */

  public ArtificialNeuron(int degree) {
    this(degree,(new ActivationFunction(ActivationFunction.SIGMOID)));
  }

  /**
   * ActivationFunction can be specified, degree is set to the
   * default value.
   * @param af - the activation function of the neuron.
   */

  public ArtificialNeuron(ActivationFunction af) {
    this(DEFAULT_DEGREE,af);
  }

  /**
   * Degree and activation function can be specified.
   * @param degree - the number of input/weight pairs.
   * @param af - the activation function of the neuron.
   */

  public ArtificialNeuron(int degree, ActivationFunction af) {
    setDegree(degree);
    f = af;
    init();
  }


  /******************************/
  /* Accessor / Mutator Methods */
  /******************************/

  /**
   * Accessor for neuron weight values.
   * @param int index - index of weight (0 to n - 1)
   * @return float - the specified weight or MAX_VALUE if index out of range.
   */

  public float getWeight(int index) {
    try {
      return w[index];
    }
    catch (ArrayIndexOutOfBoundsException e) {
      System.err.println("Error in anns.ArtificialNeuron.getWeight()");
      System.err.println("\tarray index out of bounds");
      System.err.println("\tvalues must be between 0 and " + (n-1));
      System.err.println(e);
      return Float.MAX_VALUE;
    }
  }

  /**
   * Mutator for neuron weight values.
   * @param int index - index of weight (0 to n-1).
   * @param float f - the new weight value.
   */

  public boolean setWeight(int index, float f) {
    try {
      w[index] = f;
      return true;
    }
    catch (ArrayIndexOutOfBoundsException e) {
      System.err.println("Error in anns.ArtificialNeuron.setWeight()");
      System.err.println("\tarray index out of bounds");
      System.err.println("\tvalues must be between 0 and " + (n-1));
      System.err.println(e);
      return false;
    }
  }

  /**
   * Accessor for the neuron threshold.
   * @return float - the neuron threshold.
   */

  public float getThreshold() {
    return th;
  }

  /**
   * Mutator for the neuron threshold.
   * @param float f - the new threshold value.
   */

  public void setThreshold(float f) {
    th = f;
  }


  /**
   * Accessor for the weighted sum of neuron.
   * @return float - the weighted sum of all inputs and the threshold.
   */

  public float getWeightedSum() {
    return ws;
  }

  /**
   * Accessor for total neuron output.
   * @return float - the total neuron output.
   */

  public float getOutput() {
    return y;
  }

  /**
   * Accessor for degree of neuron.
   * @return int - the degree of the neuron (number of weights).
   */

  public int getDegree() {
    return n;
  }

  /**
   * Accessor for the activation function of this neuron.
   * @return ActivationFunction - the activation function of this neuron.
   */

  public ActivationFunction f() {
    return f;
  }

  /**
   * Mutator for the activation function of this neuron.
   * @param ActivationFunction af - the new ActivationFunction.
   */

  public void f(ActivationFunction af) {
    f = af;
  }

  /******************/
  /* Public Methods */
  /******************/

  /**
   * Clear the inputs, weights, threshold, weighted sum, and neuron output.
   */

  public void clear() {
    for (int i=0; i < n; i++)
      w[i] = 0f;
    th = 0f;
    ws = 0f;
    y = 0f;
  }

  /**
   * Initialize weights and threshold to pseudo-random numbers between
   * 0.0f and 1.0f.
   */

  public void initRandom() {
   for (int i = 0; i < n; i++)
     w[i] = (float) Math.random();
   th = (float) Math.random();
   }

  /**
   * Calculates the weighted sum, ws, and the total output, y, for this
   * neuron.
   */

  public void calculate(float[] x) {
    ws = 0f;
    for (int i=0; i < n; i++)
      ws += (x[i] * w[i]);
    ws += th * THRESHOLD_INPUT;
    y = f.getOutput(ws);
  }


  /*******************/
  /* Private Methods */
  /*******************/

  /**
   * Initialize the neuron by allocating memory for its weight array. Only
   * called by the constructor methods.
   */

  private void init() {
    w = new float[n];
    clear();
  }

  /**
   * Mutator for degree of neuron. Specified value must between MIN_DEGREE
   * and MAX_DEGREE.
   * @param int degree - the new degree value.
   * @return boolean - true if argument was legal, false otherwise.
   */

  private boolean setDegree(int degree) {
    if (degree < MIN_DEGREE) {
      System.err.println("Error in anns.ArtificialNeuron.setDegree():");
      System.err.println("\tillegal neuron degree: " + degree);
      System.err.println("\tmimimum neuron degree is: " + MIN_DEGREE);
      n = MIN_DEGREE;
      return false;
    }
    if (degree > MAX_DEGREE) {
      System.err.println("Error in anns.ArtificialNeuron.setDegree():");
      System.err.println("\tillegal neuron degree: " + degree);
      System.err.println("\tmaximum neuron degree is: " + MAX_DEGREE);
      n = MAX_DEGREE;
      return false;
    }
    n = degree;
    return true;
  }

} // End of ArtificialNeuron Class
