/**
 * This class serves as a data model for a single layer perceptron network.
 * Much of its structure is derived from the abstract class NeuralNetwork.
 *
 * @author  Fred Corbett
 * @version January 2, 1997
 */

package anns;

public class SingleLayerPerceptron extends NeuralNetwork {


  /*******************/
  /* Class Variables */
  /*******************/

  /**
   * The maximum number of layers for a single-layer perceptron network.
   */

  public final static int MAX_LAYERS = 1;


  /**********************/
  /* Instance Variables */
  /**********************/

  /**
   * Matrix of input training data.
   */

  protected float[][] inputMatrix;

  /**
   * Matrix of output training data.
   */

  protected float[][] outputMatrix;

  /**
   * The current testing case or index of the current test vector.
   */

  protected int trainingCase = 0;

  /**
   * The total number of training vectors.
   */

  protected int numTrainingCases = 0;

  /**
   * Flag to indicate a training session in progress.
   */

  protected boolean training = false;

  /**
   * The current test case or index into inputMatrix.
   */

  protected int testCase = 0;

  /**
   * The total number of test cases or size of the inputMatrix.
   */

  protected int numTestCases = 0;

  /**
   * Flag to indicate testing in progress.
   */

  protected boolean testing = false;

  /**
   * The sum square error over all input vectors. See Jain.
   * <PRE>
   *      1     p       (i)      (i)  2
   *  E = - * Sigma || y    -  d    ||
   *      2    i=1
   * </PRE>
   */

  protected float sumSquaredError = 0f;

  /**
   * The test error over for the current input/output vector.
   */

  protected float error = 0f;


  /***********************/
  /* Constructor Methods */
  /***********************/

  /**
   * Construct and initialize a single layer perceptron network.
   * @param ni - number of inputs to the network (degree of each neuron)
   * @param no - the number of network outputs (number of neurons)
   */

  public SingleLayerPerceptron(int ni, int no) {
    int[] layerSizes = new int[1];
    layerSizes[0] = no;
    initNeuralNetwork(ni, layerSizes);
  }


  /****************************/
  /* Accessor/Mutator Methods */
  /****************************/

  /**
   * Accessor for sum-squared error. May not be accurate depending on
   * current train/test state.
   * @return float - the sum-squared error.
   */

  public float getSumSquaredError() {
    return (sumSquaredError / 2);
  }

  /******************/
  /* Public Methods */
  /******************/

  /**
   * Train the neural network.
   * @param float[][] intMat - the array of row input vectors.
   * @param float[][] outMat - the array of row output vectors.
   * @return float - the sum-squared error of the last iteration or
   * MAX_VALUE if an error occured during training.
   */

  public float train(float[][] inpMat, float[][] outMat) {
    if (initializeTraining(inpMat, outMat)) {
      for (int i = 0; i < (t-1); t++) {
        for (int j = 0; j < inpMat.length; j++) {
          nextTrainingStep();
        }
      }

      // Do the last iteration
      for (int j = 0; j < inpMat.length; j++) {
        error = nextTrainingStep();
        sumSquaredError += error;
      }
      return (getSumSquaredError());
    } else
      return Float.MAX_VALUE;
  }

  /**
   * Initialize the training session.
   * @param float[][] intMat - the array of row input vectors.
   * @param float[][] outMat - the array of row output vectors.
   * @return boolean - false if error detected, true otherwise.
   */

  public boolean initializeTraining(float[][] inpMat, float[][] outMat) {
    if (inpMat.length == outMat.length) {
      clear();
      initRandom();
      inputMatrix = inpMat;
      outputMatrix = outMat;
      numTrainingCases = inpMat.length;
      resetTraining();
      return true;
    }
    System.err.println("Error in anns.SingleLayerPerceptron.initializeTraining()");
    System.err.println("\tmatrices do not have the same number of rows.");
    return false;
  }

  /**
   * Reset the training process.
   */

  public void resetTraining() {
    trainingCase = 0;
    error = 0f;
    sumSquaredError = 0f;
  }

  /**
   * Perform the next training step.
   * @param int i - index of input/output vectors.
   * @return float - error for this input/output vector.
   */

  public float nextTrainingStep() {
    if (trainingCase >= numTrainingCases)
      resetTraining();
    inputs = inputMatrix[trainingCase];
    calculate();
    adaptWeights(trainingCase);
    error = calculateTestError(trainingCase);
    sumSquaredError += error;
    trainingCase++;
    return error;
  }

  /**
   * Test all cases specified in the input matrix and output matrix.
   * @param float[][] intMat - the array of row input vectors.
   * @param float[][] outMat - the array of row output vectors.
   * @return float - measure of the total sumSquaredError, or MAX_VALUE if
   * an error occured during testing.
   */

  public float test(float[][] inpMat, float[][] outMat) {
    if (initializeTesting(inpMat,outMat)) {
      for (int i = 0; i < numTestCases; i++) {
        error = nextTestingStep();
        sumSquaredError += error;
      }
      return (getSumSquaredError());
    }
    return Float.MAX_VALUE;
  }

  /**
   * Initialize the test input and output data.
   * @param float[][] intMat - the array of row input vectors.
   * @param float[][] outMat - the array of row output vectors.
   * @return boolean - false if error detected, true otherwise.
   */

  public boolean initializeTesting(float[][] inpMat, float[][] outMat) {
    if (inpMat.length == outMat.length) {
      inputMatrix = inpMat;
      outputMatrix = outMat;
      numTestCases = inpMat.length;
      resetTesting();
      return true;
    }
    System.err.println("Error in anns.SingleLayerPerceptron.initializeTesting()");
    System.err.println("\tmatrices do not have the same number of rows.");
    return false;
  }

  /**
   * Reset the testing variables.
   */

  public void resetTesting() {
    testCase = 0;
    error = 0f;
    sumSquaredError = 0f;
  }

  /**
   * Perform the next testing step.
   */

  public float nextTestingStep() {
    if (testCase >= numTestCases)
      resetTesting();
    inputs = inputMatrix[testCase];
    calculate();
    error = calculateTestError(testCase);
    sumSquaredError += error;
    testCase++;
    return error;
  }


  /*******************/
  /* Private Methods */
  /*******************/

  /**
   * Implements the Perceptron Learning Algorithm.
   * @param int i - the current test case or index of input/output vectors.
   */

  private void adaptWeights(int i) {
    ArtificialNeuron n;  // Temporary pointer to current neuron
    float e;             // Measure of neuron error

    for (int j = 0; j < layerSizes[numLayers-1]; j++) {
      n = neuronAt(0,j);
      e = outputMatrix[i][j] - n.getOutput();
      n.setThreshold(n.getThreshold() + eta * -1 * e);
      for (int k = 0; k < n.getDegree(); k++) {
        n.setWeight(k, n.getWeight(k) + eta * inputs[k] * e);
      }
    }
  }

  /**
   * Caclulates the sum-squared error for the specified iteration.
   * @param int index - the iteration or test case index.
   */

  private float calculateTestError(int index) {
    ArtificialNeuron n;
    float e = 0f;

    int outputLayerIndex = numLayers - 1;
    for (int i = 0; i < layerSizes[outputLayerIndex]; i++) {
      n = neuronAt(outputLayerIndex,i);
      e += Math.pow((Math.abs(n.getOutput() - outputMatrix[index][i])),2);
    }
    return (float) Math.sqrt(e);
  }

} // End of SingleLayerPerceptron Class

