/**
 * This class implements a small panel to allow the user to modify the activation functions and
 * their associated parameters for each of the 3 neurons in the multi-layer perceptron network.
 *
 * @author  Fred Corbett
 * @version April 15, 1997
 */

import java.awt.*;
import anns.*;

public class ActivationFunctionPanel extends Panel {

  /*******************/
  /* Class Constants */
  /*******************/

  /**
   *
   */

  private final static String unitStepString = "Unit-Step";

  /**
   *
   */

  private final static String sigmoidString = "Sigmoid";

  /**
   *
   */

  private final static String piecewiseLinearString = "Piecewise";

  /**
   *
   */

  private final static String gaussianString = "Gaussian";

  /**
   *
   */

  private final static int NUM_NEURONS = 3;

  /**
   *
   */

  private final static int NUM_FUNCTIONS = 4;


  /**********************/
  /* Instance Variables */
  /**********************/

  /**
   * Reference to parent applet.
   */

  private MultiLayerPerceptronApplet mlpa;

  /**
   * Reference to neural network of parent applet.
   */

  private MultiLayerPerceptron mlp;

  /**
   *
   */

  private ArtificialNeuron[] an;

  /**
   *
   */

  private ActivationFunction[][] af;


  /***************************/
  /* AWT Component Variables */
  /***************************/

  /**
   * Neuron selection panel.
   */

  private Panel neuronPanel;

  /**
   * Checkbox group for neuron check boxes
   */

  private CheckboxGroup neuronGroup;

  /**
   * Checkbox for selecting neuron1.
   */

  private Checkbox neuron1;

  /**
   * Checkbox for selecting neuron2.
   */

  private Checkbox neuron2;

  /**
   * Checkbox for selecting neuron3.
   */

  private Checkbox neuron3;

  /**
   * Activation function selection panel.
   */

  private Panel functionPanel;

  /**
   * Checkbox group for function check boxes.
   */

  private CheckboxGroup functionGroup;

  /**
   * Checkbox for selecting unitStep
   */

  private Checkbox unitStep;

  /**
   * Checkbox for selecting sigmoid
   */

  private Checkbox sigmoid;

  /**
   * Checkbox for selecting piecewiseLinear
   */

  private Checkbox piecewiseLinear;

  /**
   * Checkbox for selecting gaussian
   */

  private Checkbox gaussian;

  /**
   * Textfield for sigmoid beta parameter.
   */

  private TextField beta;

  /**
   * Textfield for piecewise linear slope parameter.
   */

  private TextField slope;

  /**
   * Textfield for gaussian mu (mean) parameter.
   */

  private TextField mu;

  /**
   * Textfield for gaussain sigma (standard deviation) parameter.
   */

  private TextField sigma;

  /**
   * Panel for displaying the parameters textfields.
   */

  private Panel parametersPanel;

  /**
   * Layout for parameters panel.
   */

  private CardLayout parametersLayout;


  /**
   * Panel for unit step parameters. (NONE)
   */

  private Panel unitStepPanel;

  /**
   * Panel for sigmoid parameters.
   */

  private Panel sigmoidPanel;

  /**
   * Panel for piecewise linear parameters.
   */

  private Panel piecewiseLinearPanel;

  /**
   * Panel for gaussian parameters.
   */

  private Panel gaussianPanel;

  /**
   * Apply changes button.
   */

  private Button applyButton;

  /**
   * Set to default values button.
   */

  private Button defaultsButton;


  /***********************/
  /* Constructor Methods */
  /***********************/

  /**
   * Default constructor.
   * MultiLayerPerceptronApplet mlpa - reference to parent applet.
   */

  public ActivationFunctionPanel(MultiLayerPerceptronApplet mlpa) {
    this.mlpa = mlpa;
    init();
    initComponents();
    defaultSettings();
  }


  /****************************/
  /* Accessor/Mutator Methods */
  /****************************/


  /******************/
  /* Public Methods */
  /******************/

  /**
   * Respond to an action event occuring over this panel.
   * @param Event evt - the event data object.
   * @param Object arg - .
   * @return boolean - true if event handled here, otherwise event is passed to parent action
   * event handler.
   */

  public boolean action(Event evt, Object arg) {
    // Respond to a button press
    if (evt.target instanceof Button) {
      if (evt.target == applyButton)
        applySettings();
      else
        defaultSettings();
      return true;

    // Responds to a checkbox selection
    } else if (evt.target instanceof Checkbox) {
      if (evt.target == neuron1) {
        displayCurrentSettings(0);
      } else if (evt.target == neuron2) {
        displayCurrentSettings(1);
      } else if (evt.target == neuron3) {
        displayCurrentSettings(2);
      } else if (evt.target == unitStep) {
        parametersLayout.show(parametersPanel, unitStepString);
      } else if (evt.target == sigmoid) {
        parametersLayout.show(parametersPanel, sigmoidString);
      } else if (evt.target == piecewiseLinear) {
        parametersLayout.show(parametersPanel, piecewiseLinearString);
      } else {
        parametersLayout.show(parametersPanel, gaussianString);
      }
       return true;
    } else
      return false;
  }

  /**
   * Display the current settings for the activation function panel.
   */

  public void displayCurrentSettings() {
    displayCurrentSettings(getSelectedNeuron());
  }

  /**
   * Enable this panel's AWT components.
   */

  public void disableComponents() {
    // Enable the check-boxes
    neuron1.disable();
    neuron2.disable();
    neuron3.disable();
    unitStep.disable();
    sigmoid.disable();
    piecewiseLinear.disable();
    gaussian.disable();

    // Enable the textfields and buttons
    beta.disable();
    slope.disable();
    mu.disable();
    applyButton.disable();
    defaultsButton.disable();
  }

  /**
   * Disable this panel's AWT components.
   */

  public void enableComponents() {
    // Enable the check-boxes
    neuron1.enable();
    neuron2.enable();
    neuron3.enable();
    unitStep.enable();
    sigmoid.enable();
    piecewiseLinear.enable();
    gaussian.enable();

    // Enable the textfields and buttons
    beta.enable();
    slope.enable();
    mu.enable();
    applyButton.enable();
    defaultsButton.enable();
  }



  /*******************/
  /* Private Methods */
  /*******************/


  /**
   *
   */

  private void applySettings() {
    int sn = getSelectedNeuron();
    int sf = getSelectedFunction();
    ArtificialNeuron n = an[sn];
    ActivationFunction f = af[sn][sf];
    n.setActivationFunction(f);
    switch (getSelectedFunction()) {
      case 1: ((Sigmoid) f).setBeta(Utils.getFieldAsFloat(beta));
        break;
      case 2: ((PiecewiseLinear) f).setSlope(Utils.getFieldAsFloat(slope));
        break;
      case 3: ((Gaussian) f).setMu(Utils.getFieldAsFloat(mu));
        ((Gaussian) f).setSigma(Utils.getFieldAsFloat(sigma));
        break;
    }
  }

  /**
   *
   */

  private void displayCurrentSettings(int neuron) {
    ActivationFunction af = an[neuron].getActivationFunction();
    if (af instanceof UnitStep) {
      unitStep.setState(true);
      parametersLayout.show(parametersPanel, unitStepString);
    } else if (af instanceof Sigmoid) {
      sigmoid.setState(true);
      parametersLayout.show(parametersPanel, sigmoidString);
      Utils.setFieldToFloat(beta, ((Sigmoid) af).getBeta());
    } else if (af instanceof PiecewiseLinear) {
      piecewiseLinear.setState(true);
      parametersLayout.show(parametersPanel, piecewiseLinearString);
      Utils.setFieldToFloat(slope, ((PiecewiseLinear) af).getSlope());
    } else {
      gaussian.setState(true);
      parametersLayout.show(parametersPanel, gaussianString);
      Utils.setFieldToFloat(mu, ((Gaussian) af).getMu());
      Utils.setFieldToFloat(sigma, ((Gaussian) af).getSigma());
    }
  }

  /**
   *
   */

  private void defaultSettings() {
    for (int i = 0; i < NUM_NEURONS; i++) {
      an[i].setActivationFunction(af[i][ActivationFunction.SIGMOID]);
      for (int j = 0; j < NUM_FUNCTIONS; j++) {
        switch(j) {
          case 1: ((Sigmoid) af[i][j]).setBeta(Sigmoid.DEF_BETA);
            break;
          case 2: ((PiecewiseLinear) af[i][j]).setSlope(PiecewiseLinear.DEF_SLOPE);
            break;
          case 3: {
            ((Gaussian) af[i][j]).setMu(Gaussian.DEF_MU);
            ((Gaussian) af[i][j]).setSigma(Gaussian.DEF_SIGMA);
          }
        }
      }
    }
    Utils.setFieldToFloat(beta, Sigmoid.DEF_BETA);
    Utils.setFieldToFloat(slope, PiecewiseLinear.DEF_SLOPE);
    Utils.setFieldToFloat(mu, Gaussian.DEF_MU);
    Utils.setFieldToFloat(sigma, Gaussian.DEF_SIGMA);
    neuron1.setState(true);
    displayCurrentSettings(0);
  }

  /**
   *
   */

  public int getSelectedNeuron() {
    if (neuronGroup.getCurrent() == neuron1)
      return 0;
    else if (neuronGroup.getCurrent() == neuron2)
      return 1;
    else
      return 2;
  }

  /**
   *
   */

  private int getSelectedFunction() {
    if (functionGroup.getCurrent() == unitStep)
      return 0;
    else if (functionGroup.getCurrent() == sigmoid)
      return 1;
    else if (functionGroup.getCurrent() == piecewiseLinear)
      return 2;
    else
      return 3;
  }

  /**
   *
   */

  private void init() {
    mlp = mlpa.getNetwork();

    // Allocate memory and get references to the networks three artificial neurons
    an = new ArtificialNeuron[NUM_NEURONS];
    an[0] = mlp.getNeuron(0,0);
    an[1] = mlp.getNeuron(0,1);
    an[2] = mlp.getNeuron(1,0);

    // Allocate memory and create one of each activation functions for each of the neurons
    af = new ActivationFunction[NUM_NEURONS][NUM_FUNCTIONS];
    for (int i = 0; i < NUM_NEURONS; i++) {
      for (int j = 0; j < NUM_FUNCTIONS; j++) {
        switch(j) {
          case 0: af[i][j] = new UnitStep();
            break;
          case 1: af[i][j] = new Sigmoid();
            break;
          case 2: af[i][j] = new PiecewiseLinear();
            break;
          case 3: af[i][j] = new Gaussian();
        }
      }
    }
  }

  /**
   * Instantiate and add the AWT components.
   */

  private void initComponents() {
    setLayout(new GridLayout(4,1));

    // Setup the neuron selection panel
    Panel neuronPanel = new Panel();
    neuronPanel.setLayout(new GridLayout(1,4));
    Label l1 = new Label(" Neuron: ");
    neuronPanel.add(l1);
    neuronGroup = new CheckboxGroup();
    neuron1 = new Checkbox("1 ", neuronGroup, true);
    neuronPanel.add(neuron1);
    neuron2 = new Checkbox("2 ", neuronGroup, false);
    neuronPanel.add(neuron2);
    neuron3 = new Checkbox("3 ", neuronGroup, false);
    neuronPanel.add(neuron3);
    add(neuronPanel);

    // Setup the function selection panel
    Panel functionPanel = new Panel();
    functionPanel.setLayout(new GridLayout(1,4));
    Label l2 = new Label(" Function: ");
    functionPanel.add(l2);
    functionGroup = new CheckboxGroup();
    unitStep = new Checkbox(unitStepString, functionGroup, false);
    functionPanel.add(unitStep);
    sigmoid = new Checkbox(sigmoidString, functionGroup, true);
    functionPanel.add(sigmoid);
    piecewiseLinear = new Checkbox(piecewiseLinearString, functionGroup, false);
    functionPanel.add(piecewiseLinear);
    gaussian = new Checkbox(gaussianString, functionGroup, false);
    functionPanel.add(gaussian);
    add(functionPanel);

    // Setup the parameters panel
    parametersPanel = new Panel();
    parametersLayout = new CardLayout();
    parametersPanel.setLayout(parametersLayout);

    // Setup the unitstep panel
    unitStepPanel = new Panel();
    unitStepPanel.add(new Label("No parameters to modify..."));
    parametersPanel.add(unitStepString, unitStepPanel);

    // Setup the sigmoid panel
    sigmoidPanel = new Panel();
    Label l3 = new Label("Beta: ");
    sigmoidPanel.add(l3);
    beta = new TextField("", 6);
    sigmoidPanel.add(beta);
    parametersPanel.add(sigmoidString, sigmoidPanel);

    // Setup the piecewise linear panel
    piecewiseLinearPanel = new Panel();
    Label l4 = new Label("Slope: ");
    piecewiseLinearPanel.add(l4);
    slope = new TextField("", 6);
    piecewiseLinearPanel.add(slope);
    parametersPanel.add(piecewiseLinearString, piecewiseLinearPanel);

    // Setup the gaussian panel
    gaussianPanel = new Panel();
    Label l5 = new Label("Mean: ");
    gaussianPanel.add(l5);
    mu = new TextField("", 6);
    gaussianPanel.add(mu);
    Label l6 = new Label("Standard-deviation: ");
    gaussianPanel.add(l6);
    sigma = new TextField("", 6);
    gaussianPanel.add(sigma);
    gaussianPanel.hide();
    parametersPanel.add(gaussianString, gaussianPanel);
    parametersLayout.show(parametersPanel, sigmoidString);
    add(parametersPanel);

    // Setup the button panel
    Panel buttonPanel = new Panel();
    applyButton = new Button("Apply");
    buttonPanel.add(applyButton);
    defaultsButton = new Button("Defaults");
    buttonPanel.add(defaultsButton);
    add(buttonPanel);

    validate();
  }


  /*******************/
  /* Private Methods */
  /*******************/


} // End of class ActivationFunctionPanel