/**                                                                                                           /**
 * This class is an abstract superclass, providing commom functionality and
 * structure to NeuralNetwork objects. Currently there are two subclasses
 * which can be instantiated: SingleLayerPerceptron and MultiLayerPerceptron.
 *
 * @author  Fred Corbett
 * @version January 2, 1997
 */

package anns;

import java.util.Vector;

public abstract class NeuralNetwork extends Object {


  /*******************/
  /* Class Variables */
  /*******************/

  /**
   * Equality epsilon; used for floating-point equality testing.
   */

  public final static float EE = 1e-10f;

  /**
   * The minimum number of layers in the neural network.
   * NOTE: The network inputs are not counted as a layer.
   */

  public final static int MIN_LAYERS = 1;

  /**
   * The maximum number of layers in the neural network.
   * NOTE: The network inputs are not counted as a layer.
   */

  public final static int MAX_LAYERS = 10;

  /**
   * The minimum number of inputs to the neural network.
   */

  public final static int MIN_INPUTS = 1;

  /**
   * The maximum number of inputs to the neural network.
   */

  public final static int MAX_INPUTS = 100;

  /**
   * The minimum number of artificial neurons in a network layer.
   */

  public final static int MIN_LAYER_NEURONS = 1;

  /**
   * The maximum number of artificial neurons in a network layer.
   */

  public final static int MAX_LAYER_NEURONS = 100;

  /**
   * The minimum learning rate.
   * NOTE: The learning rate must be greater than this value.
   */

  public final static float MIN_LEARNING_RATE = 0f;

  /**
   * The maximum learning rate.
   */

  public final static float MAX_LEARNING_RATE = 1f;

  /**
   * The default value for the learning rate.
   */

  public final static float DEFAULT_LEARNING_RATE = 0.5f;

  /**
   * The minimum error threshold. A network output is considered correct
   * if abs(desired output - actual output) <= error threshold.
   */

  public final static float MIN_ERROR_THRESHOLD = 0f;

  /**
   * The maximum error threshold..
   */

  public final static float MAX_ERROR_THRESHOLD = 0.5f;

  /**
   * The default value for the error threshold
   */

  public final static float DEFAULT_ERROR_THRESHOLD = 0.1f;

  /**
   * The minimum number of iterations for a training session.
   */

  public final static int MIN_ITERATIONS = 1;

  /**
   * The maximum number of iterations for a training session.
   */

  public final static int MAX_ITERATIONS = 10000;

  /**
   * The default number of iterations for a training session.
   */

  public final static int DEFAULT_ITERATIONS = 20;


  /**********************/
  /* Instance Variables */
  /**********************/

  /**
   * Vector of neural network layers. Element 0 is the first hidden layer
   * or output layer in case of SingleLayerPeceptron.
   */

  protected Vector[] nn;

  /**
   * The number of inputs to the network. This is the number of elements
   * in the input layer.
   */

  protected int numInputs;

  /**
   * The inputs to the neural network.
   */

  protected float[] inputs;

  /**
   * The number of outputs from the network. This is the number of neurons
   * in the output layer.
   */

  protected int numLayers;

  /**
   * The sizes (number of neurons) for each network layer. layerSize[0] is
   * the first hidden layer or output layer for the SingleLayerPerceptron.
   */

  protected int[] layerSizes;


  /**********************/
  /* Training Variables */
  /**********************/


  /**
   * The learning rate for the network.
   */

  protected float eta = DEFAULT_LEARNING_RATE;

  /**
   * The error threshold for the network.
   */

  protected float et = DEFAULT_ERROR_THRESHOLD;

  /**
   * The number of training iterations for the network.
   */

  protected int t = DEFAULT_ITERATIONS;


  /****************************/
  /* Accessor/Mutator Methods */
  /****************************/

   /**
   * Accessor for the neurons in the network.
   * @param layer - the network layer.
   * @param index - index of neuron in layer.
   * @return the specified neuron.
   */

  public ArtificialNeuron neuronAt(int layer, int index) {
    try {
      return (ArtificialNeuron) nn[layer].elementAt(index);
    }
    catch (ArrayIndexOutOfBoundsException e) {
      System.err.println("Error in anns.NeuralNetwork.neuronAt()");
      System.err.println(e);
      return new ArtificialNeuron();
    }
  }

  /**
   * Accessor for the number of network inputs.
   * @return int - the number of inputs.
   */

  public int getNumInputs() {
    return numInputs;
  }

  /**
   * Accessor for a specified network input value.
   * @param index - the index of the input (0 to numInputs - 1).
   * @return the specified input, or MAX_VALUE if the index was out of range.
   */

  public float getInput(int index) {
    try {
      return inputs[index];
    }
    catch (ArrayIndexOutOfBoundsException e) {
      System.err.println("Error in anns.NeuralNetwork.getInput()");
      System.err.println("\tindex must be between 0 and " + numInputs);
      System.err.println(e);
      return Float.MAX_VALUE;
    }
  }

  /**
   * Mutator for a specified network input value.
   * @param index - the index of the input (0 to numInputs - 1).
   * @param value - the new input value.
   * @return true if the input was set successfully, false otherwise.
   */

  public boolean setInput(int index, float value) {
    try {
      inputs[index] = value;
      return true;
    }
    catch (ArrayIndexOutOfBoundsException e) {
      System.err.println("Error in anns.NeuralNetwork.setInput()");
      System.err.println("\tindex must be between 0 and " + (numInputs-1));
      System.err.println(e);
      return false;
    }
  }

  /**
   * Accessor for the array of network inputs.
   * @return the array of network inputs.
   */

  public float[] getInputs() {
    return inputs;
  }

  /**
   * Mutator for the array of network inputs.
   * @param inps - array of network inputs.
   * @return true if the inputs were set successfully, false otherwise.
   */

  public boolean setInputs(float[] inps) {
    if (inps.length == numInputs) {
      inputs = inps;
      return true;
    } else {
      System.err.println("Error in anns.NeuralNetwork.setInputs()");
      System.err.println("\tarray passed was not the correct size.");
      return false;
    }
  }

  /**
   * Accessor for the layer sizes of the network
   * @param layer - the layer number. 0 is the first layer; either hidden or
   * output layer for SingleLayerPerceptron.
   * @return the specified layer size, or MAX_VALUE if layer was out of bounds.
   */

  public int getLayerSize(int layer) {
    try {
      return layerSizes[layer];
    }
    catch (ArrayIndexOutOfBoundsException e) {
      System.err.println("Error in anns.NeuralNetwork.getLayerSize()");
      System.err.println("\tlayer must be between 0 and " + (numLayers-1));
      System.err.println(e);
      return Integer.MAX_VALUE;
    }
  }

  /**
   * Accessor for network learning rate.
   * @return the network learning rate.
   */

  public float getLearningRate() {
    return eta;
  }

  /**
   * Mutator for the learning rate. If the argument passed is not within
   * the legal range, no change occurs.
   * @param f - the new learning rate.
   * @return true if the argument is legal, false otherwise.
   */

  public boolean setLearningRate(float f) {
    if (f < MIN_LEARNING_RATE) {
      System.err.println("Error in anns.NeuralNetwork.setLearningRate():");
      System.err.println("\tillegal learning rate: " + f);
      System.err.println("\tmimimum learning rate is: " + MIN_LEARNING_RATE);
      eta = MIN_LEARNING_RATE;
      return false;
    }
    if (f > MAX_LEARNING_RATE) {
      System.err.println("Error in anns.NeuralNetwork.setLearningRate():");
      System.err.println("\tillegal learning rate: " + f);
      System.err.println("\tmaximum learning rate is: " + MAX_LEARNING_RATE);
      eta = MAX_LEARNING_RATE;
      return false;
    }
    eta = f;
    return true;
  }

  /**
   * Accessor for the network error threshold.
   * @return the network error threshold.
   */

  public float getErrorThreshold() {
    return et;
  }

  /**
   * Mutator for the error threshold.
   * @param f - the new error threshold.
   * @return true if the argument is legal, false otherwise.
   */

  public boolean setErrorThreshold(float f) {
    if (f < MIN_ERROR_THRESHOLD) {
      System.err.println("Error in anns.NeuralNetwork.setErrorThreshold()");
      System.err.println("\tillegal error threshold: " + f);
      System.err.println("\tmimimum error threshold is: " + MIN_ERROR_THRESHOLD);
      et = MIN_ERROR_THRESHOLD;
      return false;
    }
    if (f > MAX_ERROR_THRESHOLD) {
      System.err.println("Error in anns.NeuralNetwork.setErrorThreshold()");
      System.err.println("\tillegal error threshold: " + f);
      System.err.println("\tmaximum error threshold is: " + MAX_ERROR_THRESHOLD);
      et = MAX_ERROR_THRESHOLD;
      return false;
    }
    et = f;
    return true;
  }

  /**
   * Accessor for the number of training iterations.
   * @return the number of iterations for a training session.
   */

  public int getIterations() {
    return t;
  }

  /**
   * Mutator for the number of training iterations.
   * @param i - the new number of iterations.
   * @return true if the argument is legal, false otherwise.
   */

  public boolean setIterations(int i) {
    if (i < MIN_ITERATIONS) {
      System.err.println("Error in anns.NeuralNetwork.setIterations()");
      System.err.println("\tillegal number of iterations: " + i);
      System.err.println("\tmimimum number of iterations is: " + MIN_ITERATIONS);
      t = MIN_ITERATIONS;
      return false;
    }
    if (i > MAX_ITERATIONS) {
      System.err.println("Error in anns.NeuralNetwork.setIterations()");
      System.err.println("\tillegal number of iterations: " + i);
      System.err.println("\tmaximum number of iterations is: " + MAX_ITERATIONS);
      t = MAX_ITERATIONS;
      return false;
    }
    t = i;
    return true;
  }


  /******************/
  /* Public Methods */
  /******************/

  /**
   * Clear all of the neuron weights and threshold values.
   */

  public void clear() {
    for (int layer = 0; layer < numLayers; layer++)
      for (int index = 0; index < nn[layer].size(); index++)
        neuronAt(layer,index).clear();
  }

  /**
   * Initialize all of the neuron weights and threshold values to a pseudo-
   * random value between 0 and 1.
   */

  public void initRandom() {
    for (int layer = 0; layer < numLayers; layer++)
      for (int index = 0; index < nn[layer].size(); index++)
        neuronAt(layer,index).initRandom();
  }

  /**
   * Propagates the input signal through the network by calculating the
   * output of each neuron.
   */

  public void calculate() {
    for (int layer = 0; layer < numLayers; layer++)
      for (int index = 0; index < nn[layer].size(); index++)
        neuronAt(layer,index).calculate(inputs);
  }


  /*****************************/
  /* Protected/Private Methods */
  /*****************************/

  /**
   * Mutator for the number of network inputs.
   * @param ni - the number of inputs in the input layer.
   * @return true if the argument is legal, false otherwise.
   */

  private boolean setNumInputs(int ni) {
    if (ni < MIN_INPUTS) {
      System.err.println("Error in anns.NeuralNetwork.setNumInputs()");
      System.err.println("\tillegal number of inputs: " + ni);
      System.err.println("\tmimimum number of inputs is: " + MIN_INPUTS);
      numInputs = MIN_INPUTS;
      return false;
    }
    if (ni > MAX_INPUTS) {
      System.err.println("Error in anns.NeuralNetwork.setNumInputs()");
      System.err.println("\tillegal number of inputs: " + ni);
      System.err.println("\tmaximum number of inputs is: " + MAX_INPUTS);
      numInputs = MAX_INPUTS;
      return false;
    }
    numInputs = ni;
    return true;
  }

  /**
   * Mutator for the number of network layers.
   * @param nl - the number of network layers.
   * @return true if the argument is legal, false otherwise.
   */

  private boolean setNumLayers(int nl) {
    if (nl < MIN_LAYERS) {
      numLayers = MIN_LAYERS;
      return false;
    }
    if (nl > MAX_LAYERS) {
      numLayers = MAX_LAYERS;
      return false;
    }
    numLayers = nl;
    return true;
  }

  /**
   * Mutator for the network layer sizes.
   * @param ls - array of network layer sizes.
   * @return true if the argument is legal, false otherwise.
   */

  private boolean setLayerSizes(int[] ls) {
    boolean inputError = false;

    setNumLayers(ls.length);
    for (int i=0; i < numLayers; i++) {
      if (ls[i] < MIN_LAYER_NEURONS) {
        layerSizes[i] = MIN_LAYER_NEURONS;
        inputError = true;
      }
      if (ls[i] > MAX_LAYER_NEURONS) {
        layerSizes[i] = MAX_LAYER_NEURONS;
        inputError = true;
      }
      layerSizes[i] = ls[i];
    }
    if (inputError) {
      System.err.println("Error in anns.NeuralNetwork.setLayerSizes()");
      System.err.println("\tlayer size(s) out of range");
      return false;
    } else
      return true;
  }

  /**
   * Initialize the neural network. Allocate vectors for the inputs, hidden
   * layers and output layer. Connect the artificial neurons together.
   * @param ni - the number of network inputs.
   * @param ls - array of network layer sizes.
   */

  protected void initNeuralNetwork(int ni, int[] ls) {
    // Allocate the layer size array
    layerSizes = new int[MAX_LAYERS];

    // Initialize and allocate the input layer
    setNumInputs(ni);
    inputs = new float[numInputs];

    // Initialize and allocate memory for the layers
    setLayerSizes(ls);
    nn = new Vector[numLayers];
    int degree = numInputs;
    for (int i = 0; i < numLayers; i++) {
      nn[i] = new Vector(layerSizes[i]);
      for (int j = 0; j < layerSizes[i]; j++) {
        nn[i].insertElementAt(new ArtificialNeuron(degree),j);
      }
      degree = layerSizes[i];
    }
  }

} // End of NeuralNetwork Class