/////////////////////////////////////////////////////////
// Class	    	:   BPlayer
// Author	    	:	F. Meyer
// Description	    :   the panels for a player
//							* computer : the program plays alone
//							* human  : buttons
// Last Modified    :	03.02.1998
/////////////////////////////////////////////////////////

import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import java.lang.*;

public class BPlayer extends Panel implements Runnable{

    // Public constants
    public static final int MAX_PLAYER = 2;
    
    public static final int NONE = 0;
    public static final int HUMAN = 1;
    public static final int COMPUTER = 2;
    public static final int DEALER = -1;

    public static final int NO_ACTION = 0;
    public static final int DEAL = 1;
    public static final int HIT = 2;
    public static final int STAND = 3;
    public static final int WAIT = 4;
    
    // Private variables
    private int card[];
    private int numCard;
    private int points;
    private int type;
    private String name;
    private int strategy, stratNum;
    private boolean canPlayNow;
    private int gamesWon, gamesPlayed;
    private boolean showFirst; // may be false for dealer sometimes, mostly true
    private int curAction;
    private boolean canShowScore;
    private boolean hasWon;
    
    private Applet app;
    private GridBagLayout gbl;
    private GridBagConstraints gbc;
    private Label title;
    private Label info;
    private Canvas dummy;
    private Button hit;
    private Button stand;
    private PlayerCanvas can;
    private BStrategy strat[];
    private Thread mySelf;
    
    
    ////////////////////////////////////////////////////////////////////////////
    // Constructor
    ////////////////////////////////////////////////////////////////////////////

    public BPlayer(int t, Applet app) {
    
    	// initialize some attributes
    	type = t;  // -1 = dealer, 0 = none, 1 = human, 2 = computer
	name = "";
	strategy = 0;
	points = 0;
	numCard = 0;
	canPlayNow = false;	
	card = new int [10];
	gamesWon = 0;
	gamesPlayed = 0;
	curAction = DEAL;
	showFirst = true;
	canShowScore = false;
	hasWon = false;
	
	setBackground(new Color(0, 128, 0));
	
	this.app = app;
	
	
	mySelf = new Thread(this);
	mySelf.start();
	createStrategy(1); // par dfaut...
	setCurrentStrategy(null);

	
	// Graphical interface
	gbl = new GridBagLayout();
	gbc = new GridBagConstraints();
	if (this.type == HUMAN) {
	    title = new Label(this.name = new String("You"));
	}
	else if (this.type == COMPUTER)
	    title = new Label("Computer");
	else if (this.type == DEALER)
	    title = new Label("Dealer");
	else
	    title = new Label("A ghost !!");
	    
	title.setForeground(Color.white);
		
	if (this.type != DEALER) {
	    hit = new Button("Hit");
	    ActionListener hitListener = new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    if (canPlay()) {
			addCard(Cards.next());
			drawCards();
			points = calcPoints();
			if (points<=21) {
			    curAction = HIT;
			    setCanPlay(true);
			}
			else {
			    curAction = WAIT;
			    setCanPlay(false);
			}
			can.repaint();
		    }	  
		}		    
	    };
	    hit.addActionListener(hitListener);
	    stand = new Button("Stand");
	    ActionListener standListener = new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    if (canPlay()) {
			points = calcPoints();
			curAction = STAND;
			setCanPlay(false);
		    }
		}		    
	    };		    
	    stand.addActionListener(standListener);
	    info = new Label(gamesWon + " / " + gamesPlayed);
	    info.setForeground(Color.white);
	}
	dummy = new Canvas();
	can = new PlayerCanvas(app, this);
	setLayout(gbl);
	
	// add the title
	buildConstraints(gbc, 0, 0, 1, 1, 20, 25);
	gbl.setConstraints(title, gbc);
	add(title);
	
	// add the 2 buttons : hit and stand
	if (this.type != DEALER) {
	    buildConstraints(gbc, 0, 1, 1, 1, 20, 25);
	    gbl.setConstraints(hit, gbc);
	    hit.setBackground(Color.lightGray);
	    add(hit);
	    buildConstraints(gbc, 0, 2, 1, 1, 20, 25);
	    gbl.setConstraints(stand, gbc);
	    stand.setBackground(Color.lightGray);
	    add(stand);
	}
	
	// dummy canvas
	buildConstraints(gbc, 0, 3, 1, 1, 20, 25);
	dummy.setSize(80, 2);
	gbl.setConstraints(dummy, gbc);
	add(dummy);
	
	// add the won/played attribute
	if (this.type != DEALER) {
	    buildConstraints(gbc, 0, 4, 1, 1, 20, 25);
	    gbl.setConstraints(info, gbc);
	    add(info);
	}
	// add the canvas
	buildConstraints(gbc, 1, 0, 1, 5, 80, 100);
	gbc.fill = GridBagConstraints.BOTH;
	gbl.setConstraints(can, gbc);
	can.setBackground(new Color(0,160,0));
	can.setSize(340, 90);
	add(can);
	
	updateControls();
    }


    //////////////////////////////////////////////////////
    // Method		:   updateControls
    // Descrition	:   update the panel depending on the 
    //					type of the player
    //////////////////////////////////////////////////////
    
    public void updateControls() {
    
	switch (this.type) {
	case HUMAN : {
	    title.setText(name);
	    hit.setEnabled(true);
	    stand.setEnabled(true);
	    info.setVisible(true);
	    break;
	}   
	case COMPUTER :{
	    title.setText("Computer");
	    hit.setEnabled(false);
	    stand.setEnabled(false);
	    info.setVisible(true);
	    break;
	}
	case DEALER : {
	    title.setText("Dealer");
	    break;
	}
	default : {
	    title.setText("");
	    hit.setEnabled(false);
	    stand.setEnabled(false);
	    info.setVisible(false);
	}
	}
	repaint();
    }
    
    
    //////////////////////////////////////////////////////
    // Method		:	reset   
    // Descrition	:   the player has played no game
    //////////////////////////////////////////////////////

    public void reset() {
	gamesWon = 0;
	gamesPlayed = 0;
	updateControls();
	updateInfo();
    }

    
    //////////////////////////////////////////////////////
    
  
    void buildConstraints(GridBagConstraints gbc, int gx, int gy,
			  int gw, int gh, int wx, int wy) {
        gbc.gridx = gx;
        gbc.gridy = gy;
        gbc.gridwidth = gw;
        gbc.gridheight = gh;
        gbc.weightx = wx;
        gbc.weighty = wy;
    }
    
    
    ////////////////////////////////////////////////////////////////////////////
    // Public methods : get and set for the player
    ////////////////////////////////////////////////////////////////////////////

    public int getPoints() {
    	return points;
    }
    
    public String getName() {
	return name;
    }
    
    public void setName(String name) {
	this.name = name;
    }
    
    public int getType() {
	return type;
    }
    
    public void setType(int type) {
	this.type = type;
    }
    
    
    //////////////////////////////////////////////////////////////////////////////

    public BStrategy getStrategy(int s) {
	if (s == -1 ) 
	    return strat[strategy];
	else if(s>=0 && s<10)
	    return strat[s];
	else return null;
    }
    
    
    //////////////////////////////////////////////////////
    // Method		:	createStrategy   
    // Descrition	:   create the random strategy (i.e default 
    //					strategy for the computer
    //////////////////////////////////////////////////////

    public void createStrategy(int s) {
	strat = new BStrategy [10];
	strat[0] = new BStrategy (s);
	strat[0].setName("Random");
	strat[0].clear();
	
	strategy = 0;
	stratNum = 0;
    }
    
    
    public void addStrategy(BStrategy bs) {
	if (stratNum < 9) {
	    stratNum++;
	    strat[stratNum] = bs;
	}
    }
    
    
    public void setCurrentStrategy(BStrategy bs) {
	if (bs != null)
	    strat[1] = bs;
	else 
	    strat[1] = strat[0];
    }
    
    public void setStrategy(int strategy) {
	if (strategy>=0 && strategy<10)
	    this.strategy = strategy;
    }

    //////////////////////////////////////////////////////////////////////////////
    // methods for updating some variables of the class or getting them
    ////////////////////////////////////////////////////////////////////////////

    public void canPlay(boolean canPlayNow) {
	this.canPlayNow = canPlayNow;
    }
    
    public void clearHand() {
	for (int i=0; i<10; i++)
	    card[i] = 0;
	numCard = 0;
	setAction(DEAL);
    }
    
    public void addCard(int c) {
	card[numCard] = c;
	numCard++;
    }
    
    public int cardNumber() {
	return numCard;
    }
    
    public int getCard(int i) {
	return card[i];
    }
    
    public boolean isPlaying() {
	return (this.type != NONE);
    }
    
    public synchronized void setCanPlay(boolean can) {
	canPlayNow = can;
    }
    
    public synchronized boolean canPlay() {
	return canPlayNow;
    }
    
    public int calcPoints() {
	int i;
	int pts = 0;
	
	for (i=0; i<numCard; i++) {
	    pts += Cards.points[card[i]];
	}
	if (pts>21) 
	    for (i=0; i<numCard; i++) {
		if (Cards.points[card[i]] == 11) {
		    pts -= 10;
		    if (pts>21) continue;
		    break;
		}
	    }
	    
	if(numCard == 2 && pts == 21)	// Black Jack...
	    pts = 0;
	return pts;
    }
    
    public void incWonGames() {
	gamesWon++;
    }
    
    public void resetWonGames() {
	gamesWon = 0;
    }
    
    public int getWonGames() {
	return gamesWon;
    }
    
    public void incPlayedGames() {
	gamesPlayed++;
    }
    
    public void resetPlayedGames() {
	gamesPlayed = 0;
    }
    
    public int getPlayedGames() {
	return gamesPlayed;
    }
    
    public void  updateInfo() {
	info.setText(gamesWon + " / " + gamesPlayed);
    }
    
    public void setShowFirst(boolean can) {
	showFirst = can;
    }
    
    public boolean canShowFirst() {
	return showFirst;
    }
    
    public synchronized void setAction(int act) {
	curAction = act;
    }
    
    public synchronized boolean showScore() {
	return canShowScore;
    }
    
    public synchronized void setShowScore(boolean b) {
	canShowScore = b;
    }
    
    public synchronized boolean Won() {
	return hasWon;
    }
    
    public synchronized void win() {
	hasWon = true;
    }
    
    public void lose() {
	hasWon = false;
    }
    
    public int getAction() {
	return curAction;
    }
    
    public void enableButton() {
	if (this.type == HUMAN) {
	    hit.setEnabled(true);
	    stand.setEnabled(true);
	}
    }
    
    public void disableButton() {
	if (this.type == HUMAN) {
	    hit.setEnabled(false);
	    stand.setEnabled(false);
	}
    }

    
    //////////////////////////////////////////////////////
    // Method		:   run
    // Descrition	:   only for the computer : he plays
    //					here
    //////////////////////////////////////////////////////
   
    public void run() {
	
	int s;
    
	while (true) {
	    try {
		Thread.currentThread().sleep(10);
	    } catch (Exception e)
		{
		    System.out.println("Exception on sleep");
		}
	    if (this.type == COMPUTER) {
		if (canPlay()) {
		    // compute the current state
		    switch (this.strategy) {
		    case 0 : {	// Random Strategy
			s = 0;
			break;
		    }
		    case 1 : {	// Sarsa Strategy
			s = BRL.calcState(card);
			break;
		    }
		    default : {
			s = 0;
		    }
		    }
		    
		    // select the best action associated with s 
		    curAction = strat[strategy].selectAction(s);
		    
		    //take choosed action
		    switch(curAction) {
		    case BStrategy.HIT : {
			addCard(Cards.next());
			drawCards();
			points = calcPoints();
			if (points<=21) {
			    setCanPlay(true);
			}
			else
			    setCanPlay(false);
			can.repaint();
			break;
		    }
		    case BStrategy.STAND : {
			setCanPlay(false);
			break;
		    }
		    default : {setCanPlay(false);}
		    }
		}
	    }
	}
    }      
    
    public Insets getInsets() {
	return new Insets(6, 3, 6, 6);
    }
    
    
    //////////////////////////////////////////////////////
    // Method		:	drawCards   
    // Descrition	:   paint the player cards
    //////////////////////////////////////////////////////

    public void drawCards() {
	can.repaint();
    }
    
    
    //////////////////////////////////////////////////////
    // Method	:   
    // Descrition	:   
    //////////////////////////////////////////////////////

    public void paint(Graphics g) {
	Dimension d = getSize();
	g.setColor(new Color(0, 32, 0));
	g.drawRect (1, 1, d.width-3, d.height-3);
	g.setColor(new Color(0, 192, 0));
	g.drawRect (2, 2, d.width-5, d.height-5);
	can.repaint();
    }
}

class PlayerCanvas extends Canvas {
    final static int OFFSET = 20;
    final static int CARD_WIDTH = 50;
    
    Applet app;
    BPlayer parent;
    
    
    public PlayerCanvas(Applet app, BPlayer parent) {
	this.app = app;
	this.parent = parent;
    }
    
    public void paint(Graphics g) {
	int n = parent.cardNumber();
	int i;
	int top, left;
	Dimension d = getSize();
	int pts = parent.calcPoints();
	String str;
	
	left = OFFSET;
	top = (d.height - Cards.back.getHeight(app)) / 2;
	
    	if (parent.isPlaying()) {
	    
	    if (parent.canShowFirst()) {
		g.drawImage(Cards.card[parent.getCard(0)], left, top, app);
		left += CARD_WIDTH;
	    }
	    else {
		g.drawImage(Cards.back, left, top, app);
		left += CARD_WIDTH;
	    }
	
	    for (i=1; i<n; i++) {
		g.drawImage(Cards.card[parent.getCard(i)], left, top, app);
		left += CARD_WIDTH;
	    }
	
	    if (parent.getType()!=BPlayer.DEALER) {
		if (pts == 0) pts = 21;
		g.drawString(Integer.toString(pts) + " points...", d.width - 60, d.height-15);
	    }  
	    if (parent.showScore()) {
		if (parent.getType()== BPlayer.DEALER) {
		    if (pts == 0) pts = 21;
		    g.drawString(Integer.toString(pts) + " points...", d.width - 60, d.height-15);
		}  
		if (parent.getType()!=BPlayer.DEALER)
		    if (parent.Won()) {
			str = new String("WIN :");
		    }
		    else
			str = new String("LOSE :");
	    	else 
		    str = new String("POINTS :");
	    	g.drawString(str, d.width - 60, d.height-35);
	    }
	}
}
}
