/////////////////////////////////////////////////////////
// Class	    	:   BRLPanel
// Author	    	:	F. Meyer
// Description	    :   - interface with the reinforcement algorithm
//						- navigation between all the option  
//						from here
// Last Modified    :	03.02.1998
/////////////////////////////////////////////////////////

import java.awt.*;
import java.awt.event.*;
import java.applet.Applet;

public class BRLPanel extends Panel {

    // Public constants
    public static final Color fc = Color.white;
    public static final Color bc = new Color(165, 150, 128);
    public static final Color lightbc = bc.brighter();

    // Private variables
    Applet app;
    BRL brl;
    
    private ChoiceCanvas cc;
    private OptionsPanel op;
    
    ///////////////////////////////////////////////////////////
    // Constructeur...
    ///////////////////////////////////////////////////////////
    public BRLPanel (Applet a) {
    
	app =a;
	setSize(500, 400);
	setLayout(null);
	
	// create the reinforcement algorithm
	brl = new BRL(this, 1000, 100);
	
	// create the interface with the algorithm
	cc = new ChoiceCanvas(this);
	cc.setBounds(0,0,500,30);
	cc.setBackground(Color.white);
	op = new OptionsPanel(this);
	op.setBounds(0,30,500, 365);
	op.setBackground(Color.white);
	add(cc);
	add(op);
    }
    
	
    ///////////////////////////////////////////////////////////
    // Start, stop, suspend, resume learning...
    ///////////////////////////////////////////////////////////

    public void startLearning(int ep, int pl) {
	brl = new BRL(this, ep, pl);
	brl.setEpisodes(ep);
	brl.setPlays(pl);
	brl.start();
    }
    
    public void suspendLearning() {
	brl.suspend();
    }
    
    public void continueLearning() {
	brl.resume();
    }
    
    public void stopLearning() {
	brl.stop();
    }
    
    
    public void paint(Graphics g) {
	g.setColor(Color.black);	
    }
    
    
    //////////////////////////////////////////////////////
    // Method		:	displayCard   
    // Descrition	:   display the right panel (defined by theCard)
    /////////////////////////////////////////////////////////
    
    public void displayCard(int theCard) {
	op.selectPanel(theCard);
    }
    
    
    //////////////////////////////////////////////////////
    // Method		:	getStrategy   
    // Descrition	:   return the current strategy calculated 
    //					by the learning algorithm
    /////////////////////////////////////////////////////////
    
    public BStrategy getStrategy() {
	return brl.getStrategy();
    }
    
    
    
    //////////////////////////////////////////////////////
    // Method		:	getBRL   
    // Descrition	:   return the current reinforcement learning
    //					algorithm
    /////////////////////////////////////////////////////////    
    
    public BRL getBRL() {
	return brl;
    }
    
    
    //////////////////////////////////////////////////////
    // Method		:	updateInformations   
    // Descrition	:   return the current strategy calculated 
    //					by the learning algorithm
    /////////////////////////////////////////////////////////        
    
    public void updateInformations(){
	op.updateInformations(brl);
    }
    
    
    //////////////////////////////////////////////////////
    // Method		:	initBRLVariables 
    // Descrition	:  	set all the variables of the RL algorithm
    /////////////////////////////////////////////////////////     
    
    public void initBRLVariables() {
	op.updateBRLVariables(brl);
    }
    
    
    
    //////////////////////////////////////////////////////
    // Method		:	 
    // Descrition	:  
    /////////////////////////////////////////////////////////    
    public void setGraph(BGraph2D gr) {
	op.setGraph(gr);
    }

}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class ChoiceCanvas extends Canvas implements MouseListener {
    // this canvas contains the index for the 4 panels dedicated to
    // the learning algorithm (Learning, Estimate Fiunction, Action Selection
    // and Informations) and allows the navigation between them
	
    int select;
    BRLPanel parent;
    
    final int N = 4;
    final int LENGTH = 120;
    final int RIGHT = 10;
    final int TOP = 5;
    final int HEIGHT = 25;
    final String lab[] = {"Learning", "Estimate Fct.", 
			  "Action Selection", "Informations"};
			    
			    
    //////////////////////////////////////////////////////   
    // Constructor
    //////////////////////////////////////////////////////

    public ChoiceCanvas(BRLPanel parent) {
	this.select = 1;
	this.parent = parent;
	this.addMouseListener(this);
    }   
    
    //////////////////////////////////////////////////////
    // Method		:   mouseClicked
    // Descrition	:   handle the mouse down event in this canvas
    //////////////////////////////////////////////////////
    public void mouseClicked(MouseEvent e) {
	int x = e.getX();
	int y = e.getY();
	if (x>RIGHT && x <RIGHT + N*LENGTH && y>TOP && y<TOP+HEIGHT) {
	    select = (x - RIGHT) / LENGTH + 1;
	}
	repaint();
	parent.displayCard(select);
    }
    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void mousePressed(MouseEvent e) {}
    public void mouseReleased(MouseEvent e) {}             
    
    //////////////////////////////////////////////////////
    // Method		:	update, paint   
    // Descrition	:   draw the canvas
    //////////////////////////////////////////////////////

    public void update(Graphics g){
	paint(g);
    }
    public void paint(Graphics g) {
	doSelect(select, g);
    }
    
    //////////////////////////////////////////////////////
    // Method		:	doSelect   
    // Descrition	:   select the index of the current selected panel
    //////////////////////////////////////////////////////

    public void doSelect(int s, Graphics g) {
	int i;
	int x, y;
		
	// draw the index for non selected panels
	g.setFont(new Font(g.getFont().getName(), Font.PLAIN, g.getFont().getSize()));
	for (i=1; i<=N; i++) {
	    if (i==s) {
	    }
	    else {
		g.setColor(BRLPanel.lightbc);
		x = RIGHT + (i-1)*LENGTH;
		y = TOP;
		g.fillRoundRect(x, y, LENGTH, y+HEIGHT, 8, 8);
		g.setColor(BRLPanel.bc);
		g.drawRoundRect(x, y, LENGTH, y+HEIGHT, 8, 8);
		g.drawString(lab[i-1], x+5, y+HEIGHT-8);
	    }	
	}
		
	// draw the index for the selected panel
	g.setFont(new Font(g.getFont().getName(), Font.BOLD, g.getFont().getSize()));
	g.setColor(BRLPanel.bc);
	x = RIGHT + (s-1)*LENGTH;
	y = TOP;
	g.fillRoundRect(x, y, LENGTH, y+HEIGHT, 8, 8);
	g.setColor(BRLPanel.fc);
	g.drawString(lab[s-1],x+5, y+HEIGHT-8);
    }
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

class OptionsPanel extends Panel {

    // private variables
    final int F_HEIGHT = 30;
    final int C_HEIGHT = 20;
  
    
    final int inter_strat[] = {4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 
			       18, 19, 20, 23, 24, 25, 26, 27, 28, 29, 30, 31};
    
    BRLPanel parent;
    CardLayout cl;
    
    // the Panel with its options and main informations
    Panel learn;
    final int EP_DEFAULT_NB = 1000;
    final int PL_DEFAULT_NB = 100;
    GridLayout lgl;
    Label lEpisodes, lPlays;
    TextField tEpisodes, tPlays;
    Label lCurEpisode, lCurEpNb;
    Label lPlayedGames, lPercentNb;
    Label lWonGames;
    Label lPercent;
    Button bStart, bStop;
    Button bSuspend, bResume;
    
    Button bUpdate, bStats; 
    Button bShowGraph, bHideGraph;
    BTable qTable;
    Frame qFrame;
    Button bShowQTable, bHideQTable;
    
    // the Panel with info and components about the estimate function
    Panel function;
    final double ALPHA_D = 0.01;	// default values for these constantes
    final double GAMMA_D = 0.9;
    final int GOOD_REW_D = 0;
    final int BAD_REW_D = -1;
    final int DEFAULT_REW_D = 0;
    Label lConst;
    Label lAlpha;
    TextField tAlpha;
    Label lGamma;
    TextField tGamma;
    Label lReward;
    Label lGoodReward, lBadReward, lDefaultReward;
    TextField tGoodReward, tBadReward, tDefaultReward;
    Button bFuncSet;
    Button bFuncDefault;
    
    // the Panel with info and components about action selection
    Panel select;
    final int ACTION_D = BRL.E_GREEDY;
    final double EPSILON_D = 0.01;
    final int DEALER_LIMIT = 17;
    CheckboxGroup cGroup;
    Checkbox cEpsilon;
    Label lSelectAction;
    Label lEpsilon;
    TextField tEpsilon;
    Checkbox cSoftMax;    
    Label lDealer;
    Label lDealerLimit;
    TextField tDealerLimit;    
    Button bSelSet;
    Button bSelDefault;
    
    // the panmel with the graph and the value table
    Panel info;
    CardLayout cInfo;
    Panel pInfo;
    BGraph2D gInfo;
    BTable tInfo;
    Button bGraph, bValues, bInfoUpdate;
    
        
    private int curPanel;
    
    // Private variables
    private int nEpisodes, nPlays;
    private int nGoodRew, nBadRew, nDefRew;
    private double nAlpha, nGamma, nEpsilon;
    private int nDealerLimit;
    private int nAction;
    
    //////////////////////////////////////////////////////////////////////////
    // Constructeur
    //////////////////////////////////////////////////////////////////////////

    public OptionsPanel(BRLPanel parent) {
	int i,j;
	
	this.parent = parent;
    	nGamma = GAMMA_D;
	nAlpha = ALPHA_D;
	nEpsilon = EPSILON_D;
	nGoodRew = GOOD_REW_D;
	nBadRew = BAD_REW_D;
	nDefRew = DEFAULT_REW_D;
	nEpisodes = EP_DEFAULT_NB;
	nPlays = PL_DEFAULT_NB;
	nDealerLimit = DEALER_LIMIT;
	nAction = ACTION_D;
		
	cl = new CardLayout(5, 0);
	setLayout(cl);
		
	// Creation of the 'Learning' Panel
	learn = new Panel();
	learn.setBackground(BRLPanel.bc);
	learn.setForeground(BRLPanel.lightbc);
	learn.setLayout(null);
	// component creation
	lEpisodes = new Label("Number of episodes : ");
	lEpisodes.setBounds(2, 10, 190, F_HEIGHT); 
	tEpisodes = new TextField("1000");
	tEpisodes.setForeground(Color.black); // BRLPanel.bc);
	tEpisodes.setBackground(Color.white); // BRLPanel.bc);  // Text field fore color
	tEpisodes.setBounds(200, 10, 100, F_HEIGHT);
	lPlays = new Label("Number of plays / episode : ");
	lPlays.setBounds(2, 50, 190, F_HEIGHT); 
	tPlays = new TextField("100");
	tPlays.setForeground(Color.black); // BRLPanel.bc);
	tPlays.setBackground(Color.white); // BRLPanel.bc);  // Text field fore color
	tPlays.setBounds(200, 50, 100, F_HEIGHT);
	lCurEpisode = new Label("Episode :	");
	lCurEpNb = new Label("0");
	lCurEpisode.setBounds(2, 90, 190, F_HEIGHT);
	lCurEpNb.setBounds(200, 90, 100, F_HEIGHT);
	lPlayedGames= new Label("% of won games :");
	lPercentNb = new Label("0 %");
	lPlayedGames.setBounds(2, 130, 190, F_HEIGHT);
	lPercentNb.setBounds(200, 130, 100, F_HEIGHT);
	lWonGames = new Label ("Won Games : 0 - Lost Games : 0");
	lWonGames.setBounds(2, 170, 300, F_HEIGHT);
	lPercent = new Label ("Pourcentage de partie gagnes : 0%");
	bStart = new Button("Start Learning");
	ActionListener bStartListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		int ep, pl;	
		// get the number of episodes and plays in the text fields
		try {
		    ep = Integer.parseInt(tEpisodes.getText());
		} catch (Exception ex) {
		    ep = EP_DEFAULT_NB;
		    tEpisodes.setText(Integer.toString(ep));
		}
		try {
		    pl = Integer.parseInt(tPlays.getText());
		} catch (Exception ex) {
		    pl = PL_DEFAULT_NB;
		    tPlays.setText(Integer.toString(pl));
		}
		
		// go on with learning
		bStart.setEnabled(false);
		bStop.setEnabled(true);
		bSuspend.setEnabled(true);
		bResume.setEnabled(false);
		nEpisodes = ep;
		nPlays = pl;
		getBRLParent().startLearning(ep, pl);
	    }	    
	};
	bStart.addActionListener(bStartListener);
	bStart.setBounds(10, 210, 140, F_HEIGHT);
	bStop = new Button("Stop Learning");
	ActionListener bStopListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		getBRLParent().stopLearning();
		bStart.setEnabled(true);
		bStop.setEnabled(false);
		bSuspend.setEnabled(false);
		bResume.setEnabled(false);
	    }
	};
	bStop.addActionListener(bStopListener);
	bStop.setBounds(160, 210, 140, F_HEIGHT);
	bSuspend = new Button("Suspend Learning");
	ActionListener bSuspendListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		bStart.setEnabled(false);
		bStop.setEnabled(true);
		bSuspend.setEnabled(false);
		bResume.setEnabled(true);
		getBRLParent().suspendLearning();
	    }
	};
	bSuspend.addActionListener(bSuspendListener);
	bSuspend.setBounds(10, 250, 140, F_HEIGHT);
	bResume = new Button("Continue Learning");
	ActionListener bResumeListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		bStart.setEnabled(false);
		bStop.setEnabled(true);
		bSuspend.setEnabled(true);
		bResume.setEnabled(false);
		getBRLParent().continueLearning();
	    }
	};
	bResume.addActionListener(bResumeListener);
	bResume.setBounds(160, 250, 140, F_HEIGHT);
	bStart.setEnabled(true);
	bStop.setEnabled(false);
	bSuspend.setEnabled(false);
	bResume.setEnabled(false);
	// add the elements to panel
	learn.add (lEpisodes);
	learn.add (tEpisodes);
	learn.add (lPlays);
	learn.add (tPlays);
	learn.add (lCurEpisode);
	learn.add (lCurEpNb);
	learn.add (lPlayedGames);
	learn.add (lPercentNb);
	learn.add (lWonGames);
	learn.add (lPercent);
	learn.add (bStart);
	learn.add (bStop);
	learn.add (bSuspend);
	learn.add (bResume);
    	
		
	// Creation of the 'Estimate Function' panel
	function = new Panel();
	function.setLayout(null);
	function.setBackground(BRLPanel.bc);
	function.setForeground(BRLPanel.lightbc);
	lConst = new Label("Constants");
	lConst.setBounds(15, 10, 100, F_HEIGHT);
	lAlpha = new Label("Alpha : ");
	lAlpha.setBounds(7, 50, 120, F_HEIGHT);
	tAlpha = new TextField(Double.toString(parent.getBRL().getAlpha()));
	tAlpha.setForeground(Color.black); // BRLPanel.bc);  // Text field fore color
	tAlpha.setBackground(Color.white); // BRLPanel.bc);  // Text field fore color	
	tAlpha.setBounds(130, 50, 100, F_HEIGHT);
	lGamma = new Label("Gamma : ");
	lGamma.setBounds(7, 90, 120, F_HEIGHT);
	tGamma = new TextField(Double.toString(parent.getBRL().getGamma()));
	tGamma.setBackground(Color.white); // BRLPanel.bc);  // Text field fore color
	tGamma.setForeground(Color.black);
	tGamma.setBounds(130, 90, 70, F_HEIGHT);
	lReward = new Label ("Rewards");
	lReward.setBounds(258, 10, 100, F_HEIGHT);
	lGoodReward = new Label("Win : ");
	lGoodReward.setBounds(250, 50, 120, F_HEIGHT);
	tGoodReward = new TextField(Integer.toString(parent.getBRL().getGoodReward()));
	tGoodReward.setForeground(Color.black); // BRLPanel.bc);  // Text field fore color
	tGoodReward.setBackground(Color.white); // BRLPanel.bc);  // Text field fore color
	tGoodReward.setBounds(375, 50, 100, F_HEIGHT);
	lBadReward = new Label("Lose : ");
	lBadReward.setBounds(250, 90, 120, F_HEIGHT);
	tBadReward = new TextField(Integer.toString(parent.getBRL().getBadReward()));
	tBadReward.setForeground(Color.black); // BRLPanel.bc);  // Text field fore color
	tBadReward.setBackground(Color.white); // BRLPanel.bc);  // Text field fore color
	tBadReward.setBounds(375, 90, 100, F_HEIGHT);
	lDefaultReward = new Label("Default : ");
	lDefaultReward.setBounds(250, 130, 120, F_HEIGHT);
	tDefaultReward = new TextField(Integer.toString(parent.getBRL().getDefaultReward()));
	tDefaultReward.setForeground(Color.black);  // Text field fore color
	tDefaultReward.setBackground(Color.white); // BRLPanel.bc);  // Text field fore color
	tDefaultReward.setBounds(375, 130, 100, F_HEIGHT);
	bFuncSet = new Button("Set");
	ActionListener bFuncSetListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		double g, a;
		int gr, br, dr;
		BRL brl = getBRLParent().getBRL();
		
		try {
		    g = Double.valueOf(tGamma.getText()).doubleValue();
		} catch (Exception ex) {
		    g = GAMMA_D;
		    tGamma.setText(Double.toString(g));
		}
		try {
		    a = Double.valueOf(tAlpha.getText()).doubleValue();
		} catch (Exception ex) {
		    a = ALPHA_D;
		    tAlpha.setText(Double.toString(a));
		}
		try {
		    gr = Integer.parseInt(tGoodReward.getText());
		} catch (Exception ex) {
		    gr = GOOD_REW_D;
		    tGoodReward.setText(Integer.toString(gr));
		}
		try {
		    br = Integer.parseInt(tBadReward.getText());
		} catch (Exception ex) {
		    br = BAD_REW_D;
		}
		try {
		    dr = Integer.parseInt(tDefaultReward.getText());
		} catch (Exception ex) {
		    dr = DEFAULT_REW_D;
		}
		brl.setAlpha(a);
		brl.setGamma(g);
		
		// for SARSA-algorithm consistency
		if (br<dr && dr<=gr) {
		    brl.setGoodReward(gr);
		    brl.setBadReward(br);
		    brl.setDefaultReward(dr);
		}
		else
		    {
			brl.setGoodReward(GOOD_REW_D);
			brl.setBadReward(BAD_REW_D);
			brl.setDefaultReward(DEFAULT_REW_D);
		    }
		
		// update the fields now. If some values were out of range,
		// they are not changed	    
		nGamma = brl.getGamma();
		nAlpha = brl.getAlpha();
		nGoodRew = brl.getGoodReward();
		nBadRew = brl.getBadReward();
		nDefRew = brl.getDefaultReward();
		
		tAlpha.setText(Double.toString(nAlpha));
		tGamma.setText(Double.toString(nGamma));
		tGoodReward.setText(Integer.toString(nGoodRew));
		tBadReward.setText(Integer.toString(nBadRew));
		tDefaultReward.setText(Integer.toString(nDefRew));
	    }		   
	};
	bFuncSet.addActionListener(bFuncSetListener);
	bFuncSet.setBounds(50, 210, 150, F_HEIGHT);
	bFuncDefault = new Button("Default");
	ActionListener bFuncDefaultListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		BRL brl = getBRLParent().getBRL();

		// set all values to default value
		brl.setAlpha(ALPHA_D);
		brl.setGamma(GAMMA_D);
		brl.setGoodReward(GOOD_REW_D);
		brl.setBadReward(BAD_REW_D);
		brl.setDefaultReward(DEFAULT_REW_D);
		
		nGamma = GAMMA_D;
		nAlpha = ALPHA_D;
		nGoodRew = GOOD_REW_D;
		nBadRew = BAD_REW_D;
		nDefRew = DEFAULT_REW_D;
		
		// update the values fields
		tAlpha.setText(Double.toString(brl.getAlpha()));
		tGamma.setText(Double.toString(brl.getGamma()));
		tGoodReward.setText(Integer.toString(brl.getGoodReward()));
		tBadReward.setText(Integer.toString(brl.getBadReward()));
		tDefaultReward.setText(Integer.toString(brl.getDefaultReward()));
	    }	
	};
	bFuncDefault.addActionListener(bFuncDefaultListener);
	bFuncDefault.setBounds(280, 210, 150, F_HEIGHT);
	function.add(lConst);
	function.add(lAlpha);
	function.add(tAlpha);
	function.add(lGamma);
	function.add(tGamma);
	function.add(lReward);
	function.add(lGoodReward);
	function.add(tGoodReward);
	function.add(lBadReward);
	function.add(tBadReward);
	function.add(lDefaultReward);
	function.add(tDefaultReward);	
	function.add(bFuncSet);
	function.add(bFuncDefault);
		
	// Creation of the 'Action Selection' panel
	select = new Panel();
	select.setLayout(null);
    	select.setBackground(BRLPanel.bc);
	select.setForeground(BRLPanel.lightbc);
	lSelectAction = new Label("Select action with ");
	cGroup = new CheckboxGroup();
	cEpsilon = new Checkbox("Epsilon-Greedy Function", cGroup, true);
	cSoftMax = new Checkbox("SoftMax function", cGroup, false);
	lEpsilon = new Label ("Epsilon : ");
	tEpsilon = new TextField(Double.toString(parent.getBRL().getEpsilon()));	
	tEpsilon.setForeground(Color.black);  // Text field fore color
	tEpsilon.setBackground(Color.white); // BRLPanel.bc);  // Text field fore color
	lDealer = new Label("Dealer");
	lDealerLimit = new Label("Dealer stops hitting at");
	tDealerLimit = new TextField("17");
	tDealerLimit.setForeground(Color.black);  // Text field fore color
	tDealerLimit.setBackground(Color.white); // BRLPanel.bc);  // Text field fore color
	bSelSet = new Button("Set");
	ActionListener bSelSetListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		double eps;
		int dl;
		int act;
		
		// Get and check new values
		try {
		    eps = Double.valueOf(tEpsilon.getText()).doubleValue();
		} catch (Exception ex) {
		    eps = EPSILON_D;
		}
		try {
		    dl = Integer.parseInt(tDealerLimit.getText());
		    if (dl<0 || dl>21)
			dl = getBRLParent().getBRL().getDealerLimit();
		}
		catch (Exception ex) {
		    dl = DEALER_LIMIT;
		}
		// type of action selection
		if (cSoftMax.getState()) {
		    act = BRL.SOFTMAX;
		}
		else
		    act = BRL.E_GREEDY;
		
		// Set the new value
		getBRLParent().getBRL().setEpsilon(eps);
		getBRLParent().getBRL().setDealerLimit(dl);
		getBRLParent().getBRL().setActionSelection(act);
		
		// Update associated field (when the new value is not correct)
		nAction = act;
		nEpsilon = getBRLParent().getBRL().getEpsilon();
		nDealerLimit = getBRLParent().getBRL().getDealerLimit();
		
		tEpsilon.setText(Double.toString(getBRLParent().getBRL().getEpsilon()));
		tDealerLimit.setText(Integer.toString(getBRLParent().getBRL().getDealerLimit()));
	    }	    
	};
	bSelSet.addActionListener(bSelSetListener);
	bSelDefault = new Button("Default");
	ActionListener bSelDefaultListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		// set epsilon to default value and update associated field
		getBRLParent().getBRL().setEpsilon(EPSILON_D);
		tEpsilon.setText(Double.toString(EPSILON_D));
		getBRLParent().getBRL().setDealerLimit(DEALER_LIMIT);
		tDealerLimit.setText(Integer.toString(DEALER_LIMIT));
		nEpsilon = getBRLParent().getBRL().getEpsilon();
		nDealerLimit = getBRLParent().getBRL().getDealerLimit();
	    }
	};
	bSelDefault.addActionListener(bSelDefaultListener);
	lSelectAction.setBounds(18, 10, 150, F_HEIGHT);
	cEpsilon.setBounds(10, 50, 200, F_HEIGHT);
	lEpsilon.setBounds(40, 90, 120, F_HEIGHT);
	tEpsilon.setBounds(170, 90, 100, F_HEIGHT);
	cSoftMax.setBounds(10, 130, 200, F_HEIGHT);
	lDealer.setBounds(20, 170, 100, F_HEIGHT);
	lDealerLimit.setBounds(10, 210, 200, F_HEIGHT);
	tDealerLimit.setBounds(220, 210, 100, F_HEIGHT);
	bSelSet.setBounds(50, 290, 150, F_HEIGHT);
	bSelDefault.setBounds(280, 290, 150, F_HEIGHT);
	select.add(lSelectAction);
	select.add(cEpsilon);
	select.add(lEpsilon);
	select.add(tEpsilon);
	select.add(cSoftMax);	
	select.add(lDealer);
	select.add(lDealerLimit);
	select.add(tDealerLimit);
	select.add(bSelSet);
	select.add(bSelDefault);
		
	// Creation of the 'Informations' panel
	info = new Panel();
	info.setLayout(null);
	info.setBackground(BRLPanel.bc);
	pInfo = new Panel();
	pInfo.setBounds(5, 5, 480, 300);
	cInfo = new CardLayout();
	pInfo.setLayout(cInfo);
	gInfo = new BGraph2D(1000);
	gInfo.setLineColor(Color.red);
	gInfo.setAxesColor(Color.blue);
	gInfo.setBackground(BRLPanel.lightbc);
	tInfo = new BTable(null);
	tInfo.setSize(480, 300);
	tInfo.setBackground(BRLPanel.lightbc);
	tInfo.createCells(14, 6);
	for(i=0; i<6; i++) 
	    for (j=0; j<14; j++) {
		tInfo.setCellBorder(j, i, true, true, true, true);
		tInfo.setCellBorderColor(j, i, Color.blue);
		tInfo.setCellLabelColor(j, i, new Color(96, 0, 0));
	    }
	updateQTable();
	pInfo.add("graph", gInfo);
	pInfo.add("table", tInfo);
	bGraph = new Button("Graph");
	ActionListener bGraphListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		BGraph2D tmp = getBRLParent().getBRL().getGraph();		
		// 'Graph' button is pressed
		    if (tmp != null) {
			cInfo.removeLayoutComponent(gInfo);
			gInfo = tmp;
			pInfo.add("graph", gInfo);
		    }
		    cInfo.show(pInfo, "graph");
		    gInfo.repaint();	
	    }
	};
	bGraph.addActionListener(bGraphListener);
	bValues = new Button("Q-Values");
	ActionListener bValuesListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		cInfo.show(pInfo, "table");
	    }
	};
	bValues.addActionListener(bValuesListener);
	bInfoUpdate = new Button("Update");
	ActionListener bInfoUpdateListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		updateQTable();
		gInfo.repaint();
	    }
	};
	bInfoUpdate.addActionListener(bInfoUpdateListener);
	bGraph.setBounds(20, 320, 100, F_HEIGHT);
	bValues.setBounds(140, 320, 100, F_HEIGHT);
	bInfoUpdate.setBounds(360, 320, 100, F_HEIGHT);
	info.add(pInfo);
	info.add(bGraph);
	info.add(bValues);
	info.add(bInfoUpdate);
	add("learn", learn);
	add("function", function);
	add("select", select);
	add("info", info);
		
		
	curPanel = 1;
    }
    
    //GETTER
    public BRLPanel getBRLParent() {
	return parent;
    }            
    
    //////////////////////////////////////////////////////
    // Method	:   
    // Descrition	:   
    //////////////////////////////////////////////////////

    public void selectPanel(int p) {
	curPanel = p;
	switch (p) {
	case 1 : {
	    cl.show(this, "learn");
	    break;
	}
	case 2 : {
	    cl.show(this, "function");
	    break;
	}
	case 3 : {
	    cl.show(this, "select");
	    break;
	}
	case 4 : {
	    BGraph2D tmp;
	    tmp = parent.getBRL().getGraph();
	    if (tmp != null) {
		cInfo.removeLayoutComponent(gInfo);
		gInfo = tmp;
		pInfo.add("graph", gInfo);
		cInfo.show(pInfo, "graph");
	    }
	    cl.show(this, "info");
	    break;
	}
	default: break;
	}	
    }

//////////////////////////////////////////////////////
// Method		:	paint   
// Descrition	:   paint a few rectangles
//////////////////////////////////////////////////////

public void paint(Graphics g) {
    Graphics gr;
    
    // Drawing on the 'Estimate Function' Panel
    if (curPanel == 2) {
	gr = function.getGraphics();
	gr.drawRect(5, 25, 230, 150);
	gr.drawRect(245, 25, 240, 150);
    }
		
    // Drawing on the 'Action Selection' Panel
    else if (curPanel == 3) {
	gr = select.getGraphics();
	gr.drawRect(5, 25, 320, 135);
	gr.drawRect(5, 185, 320, 70);
    }
}
    

    
    //////////////////////////////////////////////////////////////////////////
// Private methods
//////////////////////////////////////////////////////////////////////////
   
    
//////////////////////////////////////////////////////
// Method		:	updateQTable   
// Descrition	:   update the values table
//					set correct values in all cells of the table
//////////////////////////////////////////////////////

void updateQTable() {
    BStrategy bs;
    int i;
    String s;
		
    bs = parent.getStrategy();
		
    if (bs!=null) {
	tInfo.setCellValue(0,0, new String("Q/S"), BCell.CENTER);
	tInfo.setCellValue(0,1, new String("HIT"), BCell.CENTER);
	tInfo.setCellValue(0,2, new String("STAND"), BCell.CENTER);
	tInfo.setCellValue(0,3, new String("Q/S"), BCell.CENTER);
	tInfo.setCellValue(0,4, new String("HIT"), BCell.CENTER);
	tInfo.setCellValue(0,5, new String("STAND"), BCell.CENTER);
	for (i=0; i<13; i++) {
	    tInfo.setCellValue(i+1, 0, inter_strat[i], BCell.CENTER);
	    s = Float.toString((float)bs.get(inter_strat[i], 0));
	    tInfo.setCellValue(i+1, 1, 
			       s.substring(0,Math.min(9, s.length())), 
			       BCell.RIGHT);
	    s = Float.toString((float)bs.get(inter_strat[i], 1));
	    tInfo.setCellValue(i+1, 2, 
			       s.substring(0, Math.min(9, s.length())),
			       BCell.RIGHT);
			
	    tInfo.setCellValue(i+1, 3, inter_strat[i+13], BCell.CENTER);
	    s = Float.toString((float)bs.get(inter_strat[i+13], 0));
	    tInfo.setCellValue(i+1, 4, 
			       s.substring(0, Math.min(9, s.length())),
			       BCell.RIGHT);
	    s = Float.toString((float)bs.get(inter_strat[i+13], 1));
	    tInfo.setCellValue(i+1, 5, 
			       s.substring(0, Math.min(9, s.length())),
			       BCell.RIGHT);
	}
	tInfo.repaint();
    }
}
    
    
    
//////////////////////////////////////////////////////
// Method		:	updateInformations   
// Descrition	:   update the informations in the panel
//					'Learning'
//////////////////////////////////////////////////////

void updateInformations(BRL brl) {
    int wonGames = brl.getWonGames();
    int lostGames = brl.getLostGames();
    int episode = brl.getEpisode();
    lWonGames.setText("Won Games : "+ wonGames + "   Lost Games : "+lostGames);
    lCurEpNb.setText(Integer.toString(episode));
    lPercentNb.setText(Double.toString((double)wonGames*100.0/(double)(wonGames + lostGames))+ " %");
}
    
    
//////////////////////////////////////////////////////
// Method		:	updateBRLVariables   
// Descrition	:   Set all values of the learning algorithm 
//					with the one defined here
//////////////////////////////////////////////////////
    
void updateBRLVariables(BRL brl) {
    brl.setGamma(nGamma);
    brl.setAlpha(nAlpha);
    brl.setGoodReward(nGoodRew);
    brl.setBadReward(nBadRew);
    brl.setDefaultReward(nDefRew);
    brl.setEpsilon(nEpsilon);
    brl.setDealerLimit(nDealerLimit);
    brl.setActionSelection(nAction);
}
    
    
    
    public void setGraph(BGraph2D gr) {
	gInfo = gr;
    }
   
//////////////////////////////////////////////////////////////////////////

}
