/**
 * The package which implements your RC filter.
 * @author Sebastien Baehni
 * @version 1.0
 */
package Algorithme;
/**
 * The package which implements your RC filter.
 */

import java.awt.Color;


/**
 * The class implementing the algorithme for the retinal vision.
 * @author Sebastien Baehni
 * @version 1.0
 */
public class Algorithme {
   
      
    /**
     * This is the input stimuli
     */
    protected int[] input;
    /**
     * This is the first output stimuli
     */
    protected int[] output1;
    /**
     * This is the second output stimuli
     */
    protected int[] output2;
    /**
     * This is the width of the image.
     */
    protected int width;
    /**
     * This is the height of the image.
     */
    protected int height;

    /**
     * Our constructor.
     * @param input  The input stimuli.
     * @param width  The width of our input image.
     * @param height The height of our input image.
     */
    public Algorithme(int[] input, int width, int height) {

	/* Initialisation of our variables. */
	this.width = width;
	this.height = height;
	this.input = new int[this.width*this.height];
	this.output1 = new int[this.width*this.height];
	this.output2 = new int[this.width*this.height];
	
	System.arraycopy(input,0,this.input(),0,input.length);

    }

    /**
     * This method initialize the different parameters in order to use the
     * algorithme. 
     */
    public void init() {
	     
	/* Init of all our array. */
	for (int y=0; y<this.height; y++) {
	    for (int x=0; x<this.width; x++) {
		this.output1[(y*this.width)+x] = 255<<24 | 255<<16 | 255<<8 | 0;
		this.output2[(y*this.width)+x] = 255<<24 | 255<<16 | 255<<8 | 0;	
	    }
	}
    }

    /**
     * Our algorithme. It simules a RC filter.
     * It can be seen like a mask of pixels applied on the center pixel of the mask.
     * This is a general vision of it, but it's much more complicate.
     * @exception                 AlgorithmeError
     * @param input               The input of our filter.
     * @param output              The response of our filter.
     * @param deltaT              The time derivative discretisation.
     * @param synapticResistance  The synaptic resistance.
     * @param jonctionResistance  The jonction resistance.
     * @param membranarResistance The resistance which refine our neural membran caracteristic.
     * @param membranarCapacity   The capacity of our neural membran.
     * @return The response of the RCFilter
     */
    public int[] RCFilter(int[] input,
			  int[] output,			
			  float deltaT,
			  float synapticResistance, 
			  float jonctionResistance,
			  float membranarResistance,
			  float membranarCapacity) throws AlgorithmeError {
	
	/**
	 * The result of our mask.
	 */
	int result;

	/** 
	 * The final response.
	 */
	int[] resultArray = new int[input.length];

	/**
	 * The constants of the filter.
	 */
	float alpha;
	float tau;
	float beta;
	
	/**
	 * Test on the validity of our constants.
	 */
	if (jonctionResistance == 0.0f || membranarResistance == 0.0f || membranarCapacity == 0.0f) {
	    throw new AlgorithmeError("The jonctionResistance is null");
	}
	if (membranarResistance == 0.0f ) {
	    throw new AlgorithmeError("The membranarResistance is null");
	}
	if (membranarCapacity == 0.0f) {
	    throw new AlgorithmeError("The membranarCapacity is null");
	}
	else if (synapticResistance == 0.0f) {
	    return input;
	}
	else {
	    alpha = synapticResistance/jonctionResistance;
	    beta  = synapticResistance/membranarResistance;
	    tau   = synapticResistance*membranarCapacity;	    
	}

	/**
	 * If 1+4*alpha+beta == 0 then the retina is satured and the output response is 
	 * white.
	 */
	if ( (1+beta+4*alpha) == 0.0f) {
	    for (int i=0; i<resultArray.length; i++) {
		resultArray[i] = 255<<24 | 255<<16 | 255<<8 | 255;
	    }
	    return resultArray;
	}
	
	/* We change our array in paying attention to the edge of the image. */
	for (int y=0; y<this.height; y++) {
	    for (int x=0; x<this.width; x++) {				
		    
		/* The higher left corner. */
		if (x==0 && y ==0) {			   		   
		    result = (int)(
				   (deltaT/tau)*this.grayComponent(input[(y*this.width)+x]) + 
				   alpha*(deltaT/tau)*(
						       255+
						       this.grayComponent(output[(y*this.width)+x+1])+
						       255 + 
						       this.grayComponent(output[((y+1)*this.width)])) -
				   (1+beta+4*alpha)*(deltaT/tau)*this.grayComponent(output[(y*this.width)+x]) +
				   this.grayComponent(output[(y*this.width)+x]) 
				   );		   				    
		}
		/* The higher right corner. */
		else if (x+1==this.width && y == 0) {			   
		    result = (int)(
				   (deltaT/tau)*this.grayComponent(input[(y*this.width)+x]) + 
				   alpha*(deltaT/tau)*(
						       this.grayComponent(output[(y*this.width)+x-1]) +
						       255 +
						       255 + 
						       this.grayComponent(output[((y+1)*this.width)])) -
				   (1+beta+4*alpha)*(deltaT/tau)*this.grayComponent(output[(y*this.width)+x]) +
				   this.grayComponent(output[(y*this.width)+x]) 
				   );			
		}
		/* The bottom right corner. */
		else if (x+1 == width && y+1 == height) {		   		   
		    result =  (int)(
				    (deltaT/tau)* this.grayComponent(input[(y*this.width)+x]) + 
				    alpha*(deltaT/tau)*(
							this.grayComponent(output[(y*this.width)+x-1]) + 
							255 + 
							this.grayComponent(output[((y-1)*this.width)+x]) +
							255) -
				    (1+beta+4*alpha)*(deltaT/tau)*this.grayComponent(output[(y*this.width)+x]) +
				    this.grayComponent(output[(y*this.width)+x])
				    );		   		    
		}
		/* The bottom left corner. */
		else if (x == 0 && y+1 == this.height) {		  		   
		    result =  (int)(
				    (deltaT/tau)*this.grayComponent(input[(y*this.width)+x]) + 
				    alpha*(deltaT/tau)*(
							255 + 
							this.grayComponent(output[(y*this.width)+x+1]) +
							this.grayComponent(output[((y-1)*this.width)+x]) +
							255) -
				    (1+beta+4*alpha)*(deltaT/tau)*this.grayComponent(output[(y*this.width)+x]) +
				    this.grayComponent(output[(y*this.width)+x])
				    );		   		    
		}
		/* The first line. */
		else if (x==0) {		   
		    result = (int)(
				   (deltaT/tau)* this.grayComponent(input[(y*this.width)+x]) + 
				   alpha*(deltaT/tau)*(
						       255+
						       this.grayComponent(output[(y*this.width)+x+1]) +
						       this.grayComponent(output[((y-1)*this.width)+x]) +
						       this.grayComponent(output[((y+1)*this.width)])) -
				   (1+beta+4*alpha)*(deltaT/tau)*this.grayComponent(output[(y*this.width)+x]) +
				   this.grayComponent(output[(y*this.width)+x])
				   );		   		
		}
		/* The first column. */
		else if (y==0) {			    
		    result =  (int)(
				    (deltaT/tau)* this.grayComponent(input[(y*this.width)+x]) + 
				    alpha*(deltaT/tau)*(
							this.grayComponent(output[(y*this.width)+x-1]) + 
							this.grayComponent(output[(y*this.width)+x+1]) +
							255 + 
							this.grayComponent(output[((y+1)*this.width)])) -
				    (1+beta+4*alpha)*(deltaT/tau)*this.grayComponent(output[(y*this.width)+x]) +
				    this.grayComponent(output[(y*this.width)+x])
				    );		  		
		}     
		/* The last line. */
		else if (x+1 == width) {			    
		    result =  (int)(
				    (deltaT/tau)* this.grayComponent(input[(y*this.width)+x]) + 
				    alpha*(deltaT/tau)*(
							this.grayComponent(output[(y*this.width)+x-1]) +
							255 +
							this.grayComponent(output[((y-1)*this.width)+x]) +
							this.grayComponent(output[((y+1)*this.width)+x])) -
				    (1+beta+4*alpha)*(deltaT/tau)*this.grayComponent(output[(y*this.width)+x]) +
				    this.grayComponent(output[(y*this.width)+x])
				    );				    
		}
		/* The last column. */
		else if (y+1 == height) {			    
		    result = (int)(
				   (deltaT/tau)*this.grayComponent(input[(y*this.width)+x]) + 
				   alpha*(deltaT/tau)*(
						       this.grayComponent(output[(y*this.width)+x-1]) +
						       this.grayComponent(output[(y*this.width)+x+1]) +
						       this.grayComponent(output[((y-1)*this.width)+x]) +
						       255) -
				   (1+beta+4*alpha)*(deltaT/tau)*this.grayComponent(output[(y*this.width)+x]) +
				   this.grayComponent(output[(y*this.width)+x])
				   );		 		    
		}
		/* All others cases. */
		else {		    
		    result = (int)(
				   (deltaT/tau)*this.grayComponent(input[(y*this.width)+x]) + 
				   alpha*(deltaT/tau)*(
						       this.grayComponent(output[(y*this.width)+x-1]) +
						       this.grayComponent(output[(y*this.width)+x+1]) +
						       this.grayComponent(output[((y-1)*this.width)+x]) +
						       this.grayComponent(output[((y+1)*this.width)+x])) -
				   (1+beta+4*alpha)*(deltaT/tau)*this.grayComponent(output[(y*this.width)+x]) +
				   this.grayComponent(output[(y*this.width)+x])
				   );		  		
		}
		/**
		 * We pay attention to the result. It must be in the color space.
		 */
		if (result > 255) {
		    result = 255;
		}
		else if (result < 0) {
		    result = 0;
		}	
		/**
		 * We create a new color which rgb integer is (255 << 24 | result << 16 | result << 8 | result).
		 */
		resultArray[(y*this.width)+x] = 255 << 24 | result << 16 | result << 8 | result;		  
	    }
	}
	    
	return resultArray;
    }    		

    /**
     * Return the gray component of a given integer. This optimize your algorithme because
     * instead of creating a new Color every times and get is blue color we just have to apply
     * a mask on the input to have the gray component.
     * @param input The integer of which we want to know the gray component.
     * @return The gray Component of an integer.
     */
    private int grayComponent(int input) {
	/**
	 * The red, green and blue component are equal (we
	 * compute only gray images) so it's equal if we
	 * take the green, red or blue component. 
	 */
	return (input & 0x000000ff);		
    }	

    /**
     * The input stimuli.
     * @return The input stimuli.
     */
    public int[] input() {
	return this.input;
    }

    /**
     * The output stimuli after the first low pass filter.
     * @return The first output stimuli.
     */
    public int[] output1() {
	return this.output1;
    }

    /**
     * We acceed at one component of our output stimuli after the first pass filter.
     * @exception AlgorithmeError
     * @param index The index we want to acceed.
     * @return The value of the first output stimuli at index index or an error if index isn't in the first output array
     */
    public int output1(int index) throws AlgorithmeError {
	if (index < 0 || index > this.output1.length) {
	    throw new AlgorithmeError("The index you want to acceed is not available");
	}
	else {
	    return this.output1[index];
	}
    }

    /**
     * The output sitmuli after the second low pass filter.
     * @return the second output stimuli.
     */
    public int[] output2() {
	return this.output2;
    }


    /**
     * We acceed at one component of our output stimuli after the second pass filter.
     * @exception AlgorithmeError
     * @param index The index we want to acceed.
     * @return The value of the second output stimuli at index index or an error if index isn't in the second output array
     */
    public int output2(int index) throws AlgorithmeError {
	if (index < 0 || index > this.output2.length) {
	    throw new AlgorithmeError("The index you want to acceed is not available");
	}
	else {
	    return this.output2[index];
	}
    }

}


