/**
 * This is the package which defined procedures and function to manipulate easily memory image source.
 * @author Sebastien Baehni
 * @version 1.0
 */
package AnimatedImage;

import java.awt.image.MemoryImageSource;
import java.awt.Image;
import java.awt.Color;
import java.awt.Toolkit;


/**
 * This class provide a method to animate images with memory image source.
 * We can manipulate the animated image with an array of pixels.
 * @author Sebastien Baehni
 * @version 1.0
 */
public class AnimatedImage {

		
    /**	 
     * This represents the image of our memory image source.
     */
    protected Image image;
    /**
     * This represents our memory image source.
     */
    protected MemoryImageSource memoryImageSource;
    /**
     * This represents the pixels of the memory image source.
     */
    protected int[] pixels;
    /**
     * This represents the width of the memory image source
     */
    protected int width;
    /**
     * This represents the height of the memory image source
     */
    protected int height;

    /**
     * The constructor. It creates an animatedImage with only the width, the height and the background color.
     * @param width           The width of our animatedImage.
     * @param height          The height of our animatedImage.
     * @param backgroundColor The background color of our animatedImage.
     */
    public AnimatedImage(int width, int height, Color backgroundColor) {

	/**
	 * Initialisation of all our variables.
	 */
	this.width = width;
	this.height = height;
	this.pixels = new int[this.width*this.height];

	for (int y=0; y<this.height; y++) {
	    for (int x=0; x<this.width; x++) {
		this.pixels[(y*this.width)+x] = backgroundColor.getRGB();
	    }
	}

	/**
	 * Creation of our memoryImageSource.
	 */
	this.memoryImageSource = new MemoryImageSource(this.width,
						       this.height,
						       this.pixels,
						       0,
						       this.width);
	
	/**
	 * Our animatedImage is a multi frame animation.
	 */
	this.memoryImageSource.setAnimated(true);
	
	/**
	 * We create a toolkit to have an image via the memoryImageSource.
	 */
	
	this.image = Toolkit.getDefaultToolkit().createImage(this.memoryImageSource);	
		 
    }		

    /**
     * This method initialize our animatedImage with the white color.
     */
    public void init(Color color) {
	for (int y=0; y<this.height; y++) {
	    for (int x=0; x<this.width; x++) {
		this.pixels[(y*this.width)+x] = color.getRGB();
	    }
	}
    }
    
    /**
     * Method which send the whole buffer of pixels to the memory image source in order
     * to update it and to do the animation.
     */
    public void updateAnimatedImage() {
	this.memoryImageSource.newPixels();
    }

    /**
     * Method which tells the memory image source to update only one part of his buffer.
     * @param xOrigin The x origin of the buffer.
     * @param yOrigin The y origin of the buffer.
     * @param width   The width of the buffer.
     * @param height  The height of the buffer.
     */
    public void updateAnimatedImage(int xOrigin, int yOrigin, int width, int height) {
	this.memoryImageSource.newPixels(xOrigin,yOrigin,width,height);
    }

    /**
     * Method which returns the image created via the animatedImage.
     * @return The image created via the animatedImage.
     */
    public Image image() {
	return this.image;
    }
    
    /**
     * Method which return the array of pixels of our animatedImage.
     * @return The array of pixels of the animatedImage.
     */
    public int[] pixels() {
	return this.pixels;	
    }

    /**
     * Method which replace one pixel of the animatedImage'pixels array with the indicated value.
     * @throws AnimatedImage.AnimatedImageError
     * @param index The pixel we want to change.
     * @param value The new value of the pixel.
     */
    public void pixels(int index, int value) throws AnimatedImageError {
	/**
	 * We test if the pixel exist in the pixel's array.
	 */
	if (index < 0 || index > this.pixels.length) {
	    throw new AnimatedImageError("The pixel you want to change is not in the array space");
	}
	else {
	    this.pixels[index] = value;
	}
    }

    /**
     * Method which return the width of the animatedImage.
     * @return The width of the animatedImage.
     */
    public int width() {
	return this.width;
    }

    /**
     * Method which return the height of the animatedImage.
     * @return The height of the animatedImage.
     */
    public int height() {
	return this.height;
    }
     
}
