/**
 * Package which manipulate Animated Image in order to draw them on a canvas.
 * @author Sebastien Baehni
 * @see DrawableSurface.DrawableSurface
 * @version 1.0
 */
package DrawableSurface;

import java.awt.Color;
import java.awt.event.*;

import AnimatedImage.*;
import Algorithme.*;

import PanelConfiguration.ToolPanel;

/**
 * This is the class which return the image modified by the retina algorithme.
 * @author Sebastien Baehni
 * @version 1.0
 */
public class ModifiedPaintSurface extends DrawableSurface implements Runnable, 
								     MouseListener {   
    /**
     * The toolpanel.
     * @see PanelConfiguration.ToolPanel
     * @serial
     */
    protected ToolPanel toolPanel;
    /**
	 * The retina algorithme.
	 * @see Algorithme.Algorithme
	 * @serial
	 */
    protected Algorithme algorithme;  
    /**
	 * The surface that we are going to take the pixels in order to put them in the algorithme.
	 * @serial
	 */
    protected PaintSurface paintSurface;
	
    /**
	 * The value of the cone synaptic resistance.
	 * @serial
	 */
    protected float coneSynapticResistanceFloat;
    /**	
	 * The value of the cone membran resistance.
	 * @serial
	 */
    protected float coneMembranarResistanceFloat; 
    /**
	 * The value of the cone membran capacity.
	 * @serial
	 */
    protected float coneMembranarCapacityFloat; 
    /**
	 * The value of the cone jonction resistance.
	 * @serial
	 */
    protected float coneJonctionResistanceFloat; 
    /**
	 * The value of the horizontal synaptic resistance.
	 * @serial
	 */
    protected float horizontalSynapticResistanceFloat;
    /**
	 * The value of the horizontal membran resistance.
	 * @serial
	 */
    protected float horizontalMembranarResistanceFloat;
    /**
	 * The value of the horizontal membran capacity.
	 * @serial
	 */
    protected float horizontalMembranarCapacityFloat;
    /**
	 * The value of the horizontal jonction resistance.
	 * @serial
	 */
    protected float horizontalJonctionResistanceFloat;	
    /**
	 * The value of the deltatT.
	 * @serial
	 */
    protected final float deltaT = 0.01f;
	
    /**
	 * The number of millis second we want to sleep in the algorithme.
	 * This is very useful for someone who want to see the algorithme step by step.
	 * @serial
	 */
    protected long numberOfMillisToSleep = 0;
	
    /**
	 * This boolean allow the user to stop/restart the algorithme.
	 * @serial
	 */
    protected boolean simulationStopped = false;
    
    /**
	 * The first constructor. It is identical of the drawable surface constructor except that
	 * it creates an new instance of the retina algorithme.
	 * @see DrawableSurface.DrawableSurface
	 * @param width           The width of our modifiedPaintSurface
	 * @param height          The height of our modifiedPaintSurface
	 * @param backgroundColor The background color of our modifiedPaintSurface
	 */
    public ModifiedPaintSurface(int width,		
				int height,
				Color background_color) {																
	/**
	 * Call to the super constructor.
	 */
	super(width, height, background_color);
	/**
	 * We create a new instance of the algoritme.
	 */
	this.algorithme = new Algorithme(this.animatedImage().pixels(),width,height);
    }
    
    /**
	 * The second constructor. It is identical of the drawable surface constructor except that
	 * it creates an new instance of the retina algorithme.
	 * @see DrawableSurface.DrawableSurface
	 * @param width           The width of our modifiedPaintSurface
	 * @param height          The height of our modifiedPaintSurface	 
	 */
    public ModifiedPaintSurface(int width,
				int height) {
	/**
		 * Call to this constructor whith a white background color.
		 */
	this(width,height,Color.white);
    }
    
    /**
	 * Initialisation of our modifiedPaintSurface.
	 * @see PanelConfiguration.ToolPanel
	 * @param paintSurface The paintSurface. This is it that will give us the pixels for the input
	 * of the retina algorithme.
	 * @param toolPanel The panel in which we are going to display the help message.
	 */
    public void init(PaintSurface paintSurface,ToolPanel toolPanel) {
	/**
	 * Initialisation of the algorithme and we assign this paintSuface to the parameter.
	 */
	this.toolPanel = toolPanel;
	this.algorithme.init();
	this.addMouseListener(this);
	this.paintSurface = paintSurface;
    }   
   
    /**
	 * Method which initialize the modifiedPaintSurface's parameters with the specified values.
	 * @param coneSynapticResistanceFloat        The value for the synaptic resistance of the cone layer.
	 * @param coneMembranarResistanceFloat       The value for the membranar resistance of the cone layer.
	 * @param coneMembranarCapacityFloat         The value for the membranar capacity of the cone layer.
	 * @param coneJonctionResistanceFloat        The value for the jonction resistance of the cone layer.
	 * @param horizontalSynapticResistanceFloat  The value for the synaptic resistance of the horizontal layer.
	 * @param horizontalMembranarResistanceFloat The value for the membranar resistance of the horizontal layer.
	 * @param horizontalMembranarCapacityFloat   The value for the membranar capacity of the horizontal layer.
	 * @param horizontalJonctionResistanceFloat  The value for the jonction resistance of the horizontal layer.
	 * @param numberOfMillisToSleepLong          The value of which we ara going to sleep in the algorithme.
	 */
    public void initSimulation(float coneSynapticResistanceFloat,
			       float coneMembranarResistanceFloat,
			       float coneMembranarCapacityFloat,
			       float coneJonctionResistanceFloat,
			       float horizontalSynapticResistanceFloat,
			       float horizontalMembranarResistanceFloat,
			       float horizontalMembranarCapacityFloat,
			       float horizontalJonctionResistanceFloat,
			       long numberOfMillisToSleepLong) {
	
	this.coneSynapticResistanceFloat = coneSynapticResistanceFloat;
	this.coneMembranarResistanceFloat = coneMembranarResistanceFloat;
	this.coneMembranarCapacityFloat = coneMembranarCapacityFloat;
	this.coneJonctionResistanceFloat = coneJonctionResistanceFloat;
	this.horizontalSynapticResistanceFloat = horizontalSynapticResistanceFloat;
	this.horizontalMembranarResistanceFloat = horizontalMembranarResistanceFloat;
	this.horizontalMembranarCapacityFloat = horizontalMembranarCapacityFloat;
	this.horizontalJonctionResistanceFloat = horizontalJonctionResistanceFloat;
	this.numberOfMillisToSleep = numberOfMillisToSleepLong;
	this.algorithme.init();
	
	/**
	 * We can now begin the simulation.
	 */
	this.simulationStopped = false;
    }
	 
    /**
	 * This method is invoqued when we start the algorithme. This is an individual thread.
	 * We stop it when we want to do something else (repaint another surface, etc..)
	 */
    public synchronized void run() {
	try {	         	 	    	    
	    /**
	     * We run the algorithme until when the user said that it is enough.
	     */
	    while(!this.simulationStopped) {	
		/**
		 * We apply the first RCFilter (cone layer) of the input stimuli. We copy the result in an array
		 * in order to pass it in the second filter.
		 */
		System.arraycopy(
				 this.algorithme.RCFilter(this.paintSurface.animatedImage().pixels(),
							  this.algorithme.output1(),
							  this.deltaT,
							  this.coneSynapticResistanceFloat,
							  this.coneJonctionResistanceFloat,
							  this.coneMembranarResistanceFloat,
							  this.coneMembranarCapacityFloat),
				 0,
				 this.algorithme.output1(),
				 0,
				 this.algorithme.output1().length);		
		/**
		 * The output of the first RCFilter is passed then in another RCFilter (horizontal layer).
		 * We copy the result in order to make the difference of the two output.
		 */
		System.arraycopy(
				 this.algorithme.RCFilter(this.algorithme.output1(),
							  this.algorithme.output2(),
							  this.deltaT,
							  this.horizontalSynapticResistanceFloat,
							  this.horizontalJonctionResistanceFloat,
							  this.horizontalMembranarResistanceFloat,
							  this.horizontalMembranarCapacityFloat),	
				 0,
				 this.algorithme.output2(),
				 0,
				 this.algorithme.output2().length);	   	   	
		
		/**
		 * We make the difference of the two output. So we can have
		 */
		for (int k=0; k<this.animatedImage().pixels().length; k++) {
		    int result =  (this.algorithme.output1(k) & 0x000000ff) - (this.algorithme.output2(k) & 0x000000ff);	
		    /**
		     * We pay attention that the result is in the scale of the rgb color.
		     */
		    if (result > 255) {
			result = 255;
		    }
		    else if(result<0) {		
			result=0;
		    }		 
		    if (this.paintSurface.negativeChoosen()) {
			result = 255-result;
		    }
		
		    /**
		     * We change the color of the pixels of the animatedImage.
		     */
		    this.animatedImage().pixels(k,(255 << 24 | result << 16 | result << 8 | result));
		}
		//System.arraycopy(this.algorithme.output2(),0,this.animatedImage().pixels(),0,this.algorithme.output2().length);	
				
		/**
		 * We repaint the animatedImage.
		 */
		this.animatedImage().updateAnimatedImage();
		
		/**
		 * We allow the other thread to do what they must do.
		 */
		Thread.yield();

		/**
		 * If the user wants to see the algorithme more slower he can.
		 */
		Thread.sleep(this.numberOfMillisToSleep);
	
	    }
	    
	}
	catch(AlgorithmeError error) {
	    System.out.println("An error occured in the algorithme. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	}
	catch(InterruptedException error) {
	    System.out.println("An error occured in the run thread. Please try again");	    
	}
	catch(AnimatedImageError error) {
	    System.out.println("An error occured in the animatedImage. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	    error.printStackTrace();
	}
    }
	
    /**
	 * This method is called when the user wants to stop the simulation.
	 */
    public void stop() {		
	this.simulationStopped=true;		
    }
	
    public void mouseClicked(MouseEvent e) {}
    public void mouseEntered(MouseEvent e) {
	this.toolPanel.setText("In this area you can see the output of the retina with the above input stimuli.");
    }
    public void mouseExited(MouseEvent e) {}
    public void mousePressed(MouseEvent e) {}
    public void mouseReleased(MouseEvent e) {}
		    
}
