/**
 * Package which manipulate Animated Image in order to draw them on a canvas.
 * @author Sebastien Baehni
 * @see DrawableSurface.DrawableSurface
 * @version 1.0
 */
package DrawableSurface;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.event.*;
import java.awt.Point;

import Forms.*;
import AnimatedImage.*;
import Algorithme.*;
import Texture.*;
import PanelConfiguration.ToolPanel;
import PanelConfiguration.FormsMakerPanel;

/** 
 * This class defines a surface on which a user can draw forms and can manipulate them. 
 * @author Sebastien Baehni
 * @version 1.0
 */
public class PaintSurface extends DrawableSurface implements MouseListener, 
							     MouseMotionListener {		
	
    /**
     * This is the panel on which we can give some explanation about what the user must do.
     * @serial 
     */
    protected ToolPanel toolPanel;
    /**
     * We must take the different parameters the user specified (texture, color, type of forms).
     * @serial 
     */
    protected FormsMakerPanel formsMakerPanel;

    /**
     * Administrator for the texture.
     * @see Forms.FormsAdministrator     
     * @serial 
     */
    protected FormsAdministrator formsAdministrator;
    /**
     * Administrator for the texture.
     * @see	Texture.TextureAdministrator
     * @serial
     */
    protected TextureAdministrator textureAdministrator;

    /**
     * We keep in memory the first click of the user. This is center of a circle.
     * @serial
     */
    protected Point circleCenter;
    /**
     * We keep in memory the first click of the user. This is the corner of a rectange.
     * @serial
     */
    protected Point rectangleFirstPoint;

    /**
     * This variable is very useful for drawing a polygon. We can also change the help message 
     * with this variable. When the user has clicked once to specify the center of a circle
     * (by exemple) the help message change.
     * @serial
     */
    protected int numberOfClicks = 0; 
	
    /**
     * We keep in mind if the user has clicked on a circle.
     * @serial
     */
    protected int indexCircleChoosen = Integer.MIN_VALUE;
    /**
     * We keep in mind if the user has clicked on a rectangle.
     * @serial
     */
    protected int indexRectangleChoosen = Integer.MIN_VALUE;
    /**
     * We keep in mind if the user has clicked on a polygon.
     * @serial
     */
    protected int indexPolygonChoosen = Integer.MIN_VALUE;
      
    /** 
     * The first constructor.
     * @param width            The width of our PaintSurface.
     * @param height           The height of our PaintSurface.
     * @param backgroundColor The background color of our PaintSurface.
     */
    public PaintSurface(int width,
			int height,		
			Color backgroundColor) {
	/**
	 * Call to the super constructor.
	 */
	super(width,height,backgroundColor);		
	
	/**
	 * A paintSurface is sensible to the mouse.
	 */
	this.addMouseListener(this);
	this.addMouseMotionListener(this);
	
	/**
	 * Creation of our administrators.
	 */
	this.formsAdministrator = new FormsAdministrator();
	this.textureAdministrator = new TextureAdministrator();
    }
    
    /**
     * A second constructor. The background color is white.
     * @param width   The width of our PaintSurface.
     * @param height  The height of our PaintSurface.
     */
    public PaintSurface(int width,
			int height) {
	this(width,height,Color.white);
    }   

    /**
     * Initialisation of the paintSurface.
     * @param toolPanel       The panel on which we are going to display help messages.
     * @param formsMakerPanel That panel is going to give us the information about waht the user want to do.
     */
    public void init(ToolPanel toolPanel, FormsMakerPanel formsMakerPanel) {		  	
	
  	this.toolPanel = toolPanel;
  	this.formsMakerPanel = formsMakerPanel;  	
	/**
	 * In a paintSurface, the cursor is a crosshair.
	 */
  	this.setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR)); 	
    }

    /**
     * When the user clicked on the right or middle mouse button, we clear all the surface.
     */
    private void clearAll() {
	this.numberOfClicks = 0;
	this.circleCenter = null;
	this.rectangleFirstPoint = null;
	/**
	 * The different index we have found before are deprecated.
	 */
	this.indexCircleChoosen = Integer.MIN_VALUE;
	this.indexRectangleChoosen = Integer.MIN_VALUE;
	this.indexPolygonChoosen = Integer.MIN_VALUE;
	/**
	 * There is no more circle, polygon or rectangle in the surface.
	 */
	this.formsAdministrator.clearAll();
	this.textureAdministrator.clearAll();
	this.eraseSurface(Color.white);
	this.animatedImage.updateAnimatedImage();
    }

    /**
     * Test if the user has choosen the negative image or not.
     * @return True if the negative image has been choosen.
     */
    public boolean negativeChoosen() {
	return this.formsMakerPanel.negativeChoosen();
    }
	
    /**
     * The different methods implemented for a mouse listener.
     */
    /**
     * When the mouse is pressed.  
     */
    public void mousePressed(MouseEvent e){}	
    /**
     * When the mouse is released. 
     */
    public void mouseReleased(MouseEvent e){}
    /**
     * When the mouse enters the area.
     */
    public void mouseEntered(MouseEvent e){
	/**
	 * If the formsMakerPanel is active it's that we are drawing a form. Otherwise it's that
	 * the simulation has begun and we must display another help message.
	 */
  	if (this.formsMakerPanel.formsMakerEnable()) {
	    /**
	     * We test now if the user is going to draw something on the surface for the first time (his first click).
	     * If it's that we display the appropriate help message. Otherwise we display another help message.
	     */
	    if (this.formsMakerPanel.formsMakerPaintingDisable()) {	
		switch (this.formsMakerPanel.formsChoice()) {
		case 0:
		    this.toolPanel.setText("Click once to specify a corner of the rectangle.");
		    break;
		case 1:
		    this.toolPanel.setText("Click once to specify the first point of the polygon.");
		    break;
		case 2:		    
		    this.toolPanel.setText("Click once to specify the center of the circle.");
		    break;	
		default:
		    break;
		}
	    }
	    else {
		switch (this.formsMakerPanel.formsChoice()) {
		case 0:
		    this.toolPanel.setText("Now click to specify the opposite corner of the rectangle.");
		    break;
		case 1:		    
		    this.toolPanel.setText("Click several times to build the polygon.\nTo finish it you must click near the first point.");
		    break;
		case 2:
		    this.toolPanel.setText("Now click to specify the radius of the circle.");
		    break;	
		default:
		    break;
		}
	    }
  	}
  	else {	  
	    this.toolPanel.setText("You must click on a shape to choose it.\nThen you can move it simply by moving the mouse.");
	}			  	
    }  

    /**
     * When the mouse exits the area.  
     */
    public void mouseExited(MouseEvent e){
	/**
	 * We deselect any forms that the user has clicked on.
	 */
	this.indexCircleChoosen    = Integer.MIN_VALUE;
	this.indexRectangleChoosen = Integer.MIN_VALUE;
	this.indexPolygonChoosen   = Integer.MIN_VALUE;
    }
    /**
     * When the user clicked on the area.    
     */
    public void mouseClicked(MouseEvent e){	
	/**
	 * If the formsMaker is enable it's that the user is drawing something on it. Otherwise it's
	 * that he wants to move a form.
	 */
	if (this.formsMakerPanel.formsMakerEnable()) {
	    /**
	     * The user want to clear the whole surface.
	     */
	    if (e.getModifiers() != InputEvent.BUTTON1_MASK) {
		this.clearAll();	
	    }
	    else {
		/**
		 * If this is the first click, we draw the user's specified form.
		 */
		if (this.numberOfClicks == 0) {
		    this.formsMakerPanel.disableFormsMakerPainting();
		}			
		switch (this.formsMakerPanel.formsChoice()) {
		case 0:
		    this.drawRectangle(e);		 
		    break;
		case 1:		  
		    this.drawPolygon(e);
		    break;
		case 2:
		    this.drawCircle(e);
		    break;	
		default:
		    break;
		}			
	    }
	}	
	else {	
	    /**
	     * The user wants to deselect a form.
	     */
	    if (e.getModifiers() != InputEvent.BUTTON1_MASK) {	
		this.indexCircleChoosen    =Integer.MIN_VALUE;
		this.indexRectangleChoosen = Integer.MIN_VALUE;
		this.indexPolygonChoosen   = Integer.MIN_VALUE;	
		this.toolPanel.setText("The shape is not selected anymore.\nClick with the left mouse button on a shape to select it.");
	    }    
	    else {
		/**
		 * The user wants to move a specified form. We test if the click corresponds of one of
		 * the forms he has drawn.
		 */
		try {
		    this.indexRectangleChoosen = Integer.MIN_VALUE;
		    this.indexPolygonChoosen   = Integer.MIN_VALUE;
		    this.indexCircleChoosen    = this.formsAdministrator.returnIndexCircle(e.getX(),e.getY());		
		    if (this.indexCircleChoosen == Integer.MIN_VALUE) {
			this.indexRectangleChoosen = this.formsAdministrator.returnIndexRectangle(e.getX(),e.getY());
		    }	
		    if (this.indexRectangleChoosen == Integer.MIN_VALUE && this.indexCircleChoosen == Integer.MIN_VALUE) {
			this.indexPolygonChoosen = this.formsAdministrator.returnIndexPolygon(e.getX(),e.getY());
		    }				
		    if (this.indexCircleChoosen == Integer.MIN_VALUE && 
			this.indexRectangleChoosen == Integer.MIN_VALUE && 
			this.indexPolygonChoosen == Integer.MIN_VALUE) {	
			this.toolPanel.setText("There is no shape where you clicked.");		
		    }
		    
		    if (this.indexCircleChoosen != Integer.MIN_VALUE) {
			this.toolPanel.setText("Circle selected. You can move it now.\nTo deselect it just click on the right mouse button.");	
		    }
		    else if (this.indexRectangleChoosen != Integer.MIN_VALUE) {
			this.toolPanel.setText("Rectangle selected. You can move it now.\nTo deselect it just click on the right mouse button.");	
		    }
		    else if (this.indexPolygonChoosen != Integer.MIN_VALUE) {
			this.toolPanel.setText("Polygon selected. You can move it now.\nTo deselect it just click on the right mouse button.");	
		    }			
		}
		catch (FormsAdministratorError error) {
		    System.out.println("An error occured in the formsAdministrator. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
		    error.printStackTrace();
		}
	    }	
	}
    }
    
    /**
     * When the mouse is dragged.   
     */
    public void mouseDragged(MouseEvent e) {}
    /**
     * When the mouse is moved on the area.
     */
    public void mouseMoved(MouseEvent e) {
	/**
	 * If the formsMaker is not enabled it's that the user wants to move a form.
	 */
	if (!this.formsMakerPanel.formsMakerEnable()) {			
	    try {			
		if (this.indexPolygonChoosen != Integer.MIN_VALUE || this.indexRectangleChoosen != Integer.MIN_VALUE || this.indexCircleChoosen != Integer.MIN_VALUE) {
			
		    /**
		     * First we erase the surface. Then we test which forms has been clicked. If there is only
		     * one form to repaint we make an optimisation. Otherwise we repaint the clicked form where it 
		     * has moved.
		     * 
		     */
		    boolean onlyOneFormsToDraw = true;
		    this.eraseSurface(Color.white);
		    this.animatedImage.updateAnimatedImage();
		    /**
		     * We test the polygon.
		     */
		    for (int i=0; i<this.formsAdministrator.indexPolygon(); i++) {
			if (i != this.indexPolygonChoosen) {
			    this.formsAdministrator.returnPolygon(i).drawPolygonWithTexture(this.animatedImage.pixels(),
											    this.animatedImage.width(),
											    this.animatedImage.height(),
											    this.textureAdministrator.returnPolygonTexture(i));
			    onlyOneFormsToDraw = false;
			}		   
		    }
		    /**
		     * We test the rectangle.
		     */
		    for (int i=0; i<this.formsAdministrator.indexRectangle(); i++) {
			if (i != this.indexRectangleChoosen) {
			    this.formsAdministrator.returnRectangle(i).drawRectangleWithTexture(this.animatedImage.pixels(),
												this.animatedImage.width(),
												this.animatedImage.height(),
												this.textureAdministrator.returnRectangleTexture(i));				
			    onlyOneFormsToDraw = false;
			}
		    }
		    /**
		     * We test the circle.
		     */
		    for (int i=0; i<this.formsAdministrator.indexCircle(); i++) {
			if (i != this.indexCircleChoosen) {
			    this.formsAdministrator.returnCircle(i).drawCircleWithTexture(this.animatedImage.pixels(),
											  this.animatedImage.width(),
											  this.animatedImage.height(),
											  this.textureAdministrator.returnCircleTexture(i));
			    onlyOneFormsToDraw = false;
			}
		    }
		    /**
		     * We repaint the polygon.
		     */
		    if (this.indexPolygonChoosen != Integer.MIN_VALUE) {
			this.formsAdministrator.returnPolygon(this.indexPolygonChoosen).translate(e.getX(),e.getY(),this.animatedImage.width(),this.animatedImage.height());
			this.formsAdministrator.returnPolygon(this.indexPolygonChoosen).drawPolygonWithTexture(this.animatedImage.pixels(),
													       this.animatedImage.width(),
													       this.animatedImage.height(),
													       this.textureAdministrator.returnPolygonTexture(this.indexPolygonChoosen));
			if (onlyOneFormsToDraw) {
			    this.formsAdministrator.returnPolygon(this.indexPolygonChoosen).optimizeDrawing(this.animatedImage);
			}
			else {
			    this.animatedImage.updateAnimatedImage();
			}
		    }
		    /**
		     * We repaint the rectangle.
		     */
		    else if (this.indexRectangleChoosen != Integer.MIN_VALUE) {
			this.formsAdministrator.returnRectangle(this.indexRectangleChoosen).translate(e.getX(),e.getY(),this.animatedImage.width(),this.animatedImage.height());
			this.formsAdministrator.returnRectangle(this.indexRectangleChoosen).drawRectangleWithTexture(this.animatedImage.pixels(),
														     this.animatedImage.width(),
														     this.animatedImage.height(),
														     this.textureAdministrator.returnRectangleTexture(this.indexRectangleChoosen));
			if (onlyOneFormsToDraw) {
			    this.formsAdministrator.returnRectangle(this.indexRectangleChoosen).optimizeDrawing(this.animatedImage);
			}
			else {
			    this.animatedImage.updateAnimatedImage();
			}
		    }
		    /**
		     * We repaint the circle.
		     */
		    else if (this.indexCircleChoosen != Integer.MIN_VALUE) {
			this.formsAdministrator.returnCircle(this.indexCircleChoosen).translate(e.getX(),e.getY(),this.animatedImage.width(),this.animatedImage.height());
			this.formsAdministrator.returnCircle(this.indexCircleChoosen).drawCircleWithTexture(this.animatedImage.pixels(),
													    this.animatedImage.width(),
													    this.animatedImage.height(),
													    this.textureAdministrator.returnCircleTexture(this.indexCircleChoosen));
			if (onlyOneFormsToDraw) {
			    this.formsAdministrator.returnCircle(this.indexCircleChoosen).optimizeDrawing(this.animatedImage);
			}
			else {
			    this.animatedImage.updateAnimatedImage();
			}
		    }       	
		}      
	    }
	    catch (FormsAdministratorError error) {
		System.out.println("An error occured in the formsAdministrator. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
		error.printStackTrace();
	    }
	    catch (TextureAdministratorError error) {
		System.out.println("An error occured in the textureAdministrator. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
		error.printStackTrace();
	    } 
	    catch (TextureError error) {
		System.out.println("An error occured in the texture. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
		error.printStackTrace();
	    } 
	}		    
    }	
    
    /**
     * The private method to draw a circle.
     * @param e The mouse event.
     */
    private void drawCircle(MouseEvent e) {
	try {
	    /**
	     * The user specifies the center of the circle.
	     */
	    if (this.numberOfClicks == 0) {
		this.circleCenter = e.getPoint();
		this.numberOfClicks++;
		this.toolPanel.setText("Now click a second time to specify the radius of the circle.");
	    }
	    /**
	     * The user has specified another point. We can now draw the circle with the texture.
	     */
	    else {			
		this.toolPanel.setText("Filling the circle, please be patient...");
		int radius = (new Double(Math.sqrt(Math.pow((this.circleCenter.x-e.getX()),2.0)+
						   Math.pow((this.circleCenter.y-e.getY()),2.0)))).intValue(); 		
		if (this.circleCenter.x + radius > this.width) { 	
		    radius = this.width - this.circleCenter.x;					
		}
		if (this.circleCenter.x - radius < 0) {					
		    radius = this.circleCenter.x;
		}
		if (this.circleCenter.y + radius > this.height) {					
		    radius = this.height - this.circleCenter.y;
		}
		if (this.circleCenter.y - radius < 0) {					
		    radius = this.circleCenter.y;
		}
		this.formsAdministrator.addCircle(this.circleCenter.x,
						  this.circleCenter.y,
						  radius);		
		/**
		 * We test if the user has specified a color or a texture.
		 */
		if (this.formsMakerPanel.textureChoosen()) {
		    this.textureAdministrator.addCircleTexture(2*radius,2*radius,this.formsMakerPanel.textureChoice());
		    (this.formsAdministrator.returnCircle(this.formsAdministrator.indexCircle()-1)).
			drawCircleWithTexture(this.animatedImage.pixels(),
					      this.animatedImage.width(),
					      this.animatedImage.height(),
					      this.textureAdministrator.returnCircleTexture(this.textureAdministrator.indexCircle()-1));
		}
		else {
		    this.textureAdministrator.addCircleTexture(-this.formsMakerPanel.colorChoosen());
		    (this.formsAdministrator.returnCircle(this.formsAdministrator.indexCircle()-1)).
			drawCircleWithTexture(this.animatedImage.pixels(),
					      this.animatedImage.width(),
					      this.animatedImage.height(),
					      this.textureAdministrator.returnCircleTexture(this.textureAdministrator.indexCircle()-1));
		}
		/**
		 * We repaint the circle.
		 */
		(this.formsAdministrator.returnCircle(this.formsAdministrator.indexCircle()-1)).
		    optimizeDrawing(this.animatedImage);
		this.formsMakerPanel.enableFormsMaker();
		this.numberOfClicks=0;
		this.toolPanel.setText("Click once to specify the center of the circle.");			
	    }
	}
	catch (FormsAdministratorError error) {
	    System.out.println("An error occured in the formsAdministrator. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	    error.printStackTrace();
	}
	catch (TextureAdministratorError error) {
	    System.out.println("An error occured in the textureAdministrator. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	    error.printStackTrace();
	} 
	catch (TextureError error) {
	    System.out.println("An error occured in the texture. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	    error.printStackTrace();
	}    
    }
    
    /**
     * The private method to draw a rectangle.
     * @param e The mouse event.
     */
    private void drawRectangle(MouseEvent e) {
	try {
	    /**
	     * The user clicks to specify the first corner of the rectangle.
	     */
	    if (this.numberOfClicks == 0) {
		this.rectangleFirstPoint = e.getPoint();
		this.numberOfClicks++;
		this.toolPanel.setText("Now click a second time to specify the opposite corner of the rectangle.");
	    }
	    /**
	     * The user has clicks twice. We can now fill the rectangle with the texture or the color.
	     */
	    else {
		this.toolPanel.setText("Filling the rectangle, please be patient...");
		int width = Math.abs(this.rectangleFirstPoint.x-e.getX());
		int height = Math.abs(this.rectangleFirstPoint.y-e.getY());
		int lastPointX = e.getX();
		int lastPointY = e.getY();
		/**
		 * Tests to create a new rectangle with the correct parameters (we must specify the 
		 * higher left corner and the lower right one).
		 */
		if (this.rectangleFirstPoint.x >= lastPointX && this.rectangleFirstPoint.y <= lastPointY) {
		    this.rectangleFirstPoint.x = this.rectangleFirstPoint.x - width;
		    lastPointX = lastPointX + width;
		}
		else if (this.rectangleFirstPoint.x >= lastPointX && this.rectangleFirstPoint.y > lastPointY) {
		    this.rectangleFirstPoint.x = lastPointX;
		    this.rectangleFirstPoint.y = lastPointY;
		    lastPointX = lastPointX+width;
		    lastPointY = lastPointY+height;
		}
		else if (this.rectangleFirstPoint.x < lastPointX && this.rectangleFirstPoint.y >= lastPointY) {
		    this.rectangleFirstPoint.x = lastPointX-width;
		    this.rectangleFirstPoint.y = lastPointY;					
		    lastPointY = lastPointY+height;
		}														
		
		this.formsAdministrator.addRectangle(this.rectangleFirstPoint.x,
						     this.rectangleFirstPoint.y,
						     Math.max(this.rectangleFirstPoint.x,lastPointX)-Math.min(this.rectangleFirstPoint.x,lastPointX),
						     Math.max(this.rectangleFirstPoint.y,lastPointY)-Math.min(this.rectangleFirstPoint.y,lastPointY));
		/**
		 * The user has either choosen a color or a texture.
		 */
		if (this.formsMakerPanel.textureChoosen()) {
		    this.textureAdministrator.addRectangleTexture(this.formsAdministrator.returnRectangleWidth(this.
													       formsAdministrator.
													       indexRectangle()-1),
								  this.formsAdministrator.returnRectangleHeight(this.
														formsAdministrator.
														indexRectangle()-1),
								  this.formsMakerPanel.textureChoice());
		    (this.formsAdministrator.returnRectangle(this.formsAdministrator.indexRectangle()-1)).
			drawRectangleWithTexture(this.animatedImage.pixels(),
						 this.animatedImage.width(),
						 this.animatedImage.height(),
						 this.textureAdministrator.
						 returnRectangleTexture(this.textureAdministrator.indexRectangle()-1));
		}
		else {
		    this.textureAdministrator.addRectangleTexture(-this.formsMakerPanel.colorChoosen());
		    (this.formsAdministrator.returnRectangle(this.formsAdministrator.indexRectangle()-1)).
			drawRectangleWithTexture(this.animatedImage.pixels(),
						 this.animatedImage.width(),
						 this.animatedImage.height(),
						 this.textureAdministrator.
						 returnRectangleTexture(this.textureAdministrator.indexRectangle()-1));
		}	
		/**
		 * We repaint the rectangle.
		 */
		(this.formsAdministrator.returnRectangle(this.formsAdministrator.indexRectangle()-1)).
		    optimizeDrawing(this.animatedImage);
		this.formsMakerPanel.enableFormsMaker();
		this.numberOfClicks=0;
		this.toolPanel.setText("Click once to specify a corner of the rectangle.");		
	    }
	}
	catch (FormsAdministratorError error) {
	    System.out.println("An error occured in the formsAdministrator. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	    error.printStackTrace();
	}
	catch (TextureAdministratorError error) {
	    System.out.println("An error occured in the textureAdministrator. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	    error.printStackTrace();
	}
	catch (TextureError error) {
	    System.out.println("An error occured in the texture. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	    error.printStackTrace();
	} 
    }
	
    /**
     * The private method to draw a polygon.
     * @param e The mouse event.
     */
    private void drawPolygon(MouseEvent e) {
	try {
	    /**
	     * The user clicks once to specify the first point of the polygon.
	     */
	    if (this.numberOfClicks == 0) {
		this.formsAdministrator.addPolygon(15);
		(this.formsAdministrator.returnPolygon(this.formsAdministrator.indexPolygon()-1)).
		    drawPolygon(this.animatedImage.pixels(),
				this.animatedImage.width(),
				this.animatedImage.height(),
				e.getX(),
				e.getY(),
				Color.black);
		this.numberOfClicks++;
		this.toolPanel.setText("Now click several times to build the polygon.\nTo finish it you must click near the first point.");
	    }
	    /**
	     * The user specifies more points to continue drawing the polygon.
	     */
	    else {								
		(this.formsAdministrator.returnPolygon(this.formsAdministrator.indexPolygon()-1)).
		    drawPolygon(this.animatedImage.pixels(),
				this.animatedImage.width(),
				this.animatedImage.height(),
				e.getX(),
				e.getY(),
				Color.black);
		
		/**
		 * We test if the polygon is drawn or not.
		 */
		if ( (this.formsAdministrator.returnPolygon(this.formsAdministrator.indexPolygon()-1)).polygonDrawn() ) {  
		    this.toolPanel.setText("Filling the polygon, please be patient...");
		    /**
		     * The polygon could either have a texture or a color.
		     */
		    if (this.formsMakerPanel.textureChoosen()) {
			this.textureAdministrator.addPolygonTexture(this.formsAdministrator.returnPolygonWidth(this.
													       formsAdministrator.
													       indexPolygon()-1),
								    this.formsAdministrator.returnPolygonHeight(this.
														formsAdministrator.
														indexPolygon()-1),
								    this.formsMakerPanel.textureChoice());			
			(this.formsAdministrator.returnPolygon(this.formsAdministrator.indexPolygon()-1)).
			    drawPolygonWithTexture(this.animatedImage.pixels(),
						   this.animatedImage.width(),
						   this.animatedImage.height(),
						   this.textureAdministrator.
						   returnPolygonTexture(this.textureAdministrator.indexPolygon()-1)
						   );
		    }
		    else {
			this.textureAdministrator.addPolygonTexture(-this.formsMakerPanel.colorChoosen());
			(this.formsAdministrator.returnPolygon(this.formsAdministrator.indexPolygon()-1)).
			    drawPolygonWithTexture(this.animatedImage.pixels(),
						   this.animatedImage.width(),
						   this.animatedImage.height(),
						   this.textureAdministrator.
						   returnPolygonTexture(this.textureAdministrator.indexPolygon()-1)
						   );
		    }					
		    /**
		     * We repaint our polygon.
		     */
		    (this.formsAdministrator.returnPolygon(this.formsAdministrator.indexPolygon()-1)).
			optimizeDrawing(this.animatedImage);			
		    this.formsMakerPanel.enableFormsMaker();
		    this.numberOfClicks=0;	
		    this.toolPanel.setText("Click once to specify the first point of the polygon.");
		}
		else {
		    this.animatedImage.updateAnimatedImage();
		}
	    }			
	}
	catch (FormsAdministratorError error) {
	    System.out.println("An error occured in the formsAdministrator. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	    error.printStackTrace();
	}
	catch (TextureAdministratorError error) {
	    System.out.println("An error occured in the textureAdministrator. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	    error.printStackTrace();
	}
	catch (TextureError error) {
	    System.out.println("An error occured in the texture. Please mail Sebastien.Baehni@epfl.ch for this with the following string");
	    error.printStackTrace();
	} 
    }
}
