/**
 * This package provides useful primitives to draw forms like circles, polygon or rectangle.
 * @author Sebastien Baehni
 * @version 1.0
 */
package Forms;

import java.awt.Color;

import AnimatedImage.*;
import Texture.*;

/**
 * This class construct a new circle.
 * @author Sebastien Baehni
 * @see Forms.Forms
 * @version 1.0
 */
public class MyCircle extends Forms {

    /**
     * The radius of the circle.
     */
    protected int radius;   
    /**
     * The x coord of the center.
     */
    protected int xCenter;
    /**
     * The y coord of the center.
     */
    protected int yCenter;	

    /**
     * The constructor.
     * @param xCenter The x coordinate of the center of the circle.
     * @param yCenter The y coordinate of the center of the circle.
     * @param radius  The radius of the circle.
     */
    public MyCircle(int xCenter, int yCenter, int radius) {
	super(xCenter-radius, yCenter-radius, 2*radius, 2*radius);
	this.xCenter=xCenter;
	this.yCenter=yCenter;
	this.radius = radius;  
    }
	
    /**
     * This method return the x coordinate of the center of the circle.
     * @return The x coordinate of the center of the circle.
     */
    public int xCenter() {
	return this.xCenter;
    }
	
    /**
     * This method assign the x coordinate of the center to a new value.
     * @param xCenter The new x coordinate for the center.
     */
    public void xCenter(int xCenter) {
	this.xCenter = xCenter;
    }
    
    /**
     * This method return the y coordinate of the center of the circle.
     * @return The y coordinate of the center of the circle.
     */
    public int yCenter() {
	return this.yCenter;
    }
    
    /**
     * This method assign the y coordinate of the center to a new value.
     * @param yCenter The new y coordinate for the center.
     */
    public void yCenter(int yCenter) {
	this.yCenter = yCenter;
    }

    /**
     * This method return the radius of the circle.
     * @return The radius of the circle.
     */
    public int radius() {
	return this.radius;
    }
  
    /**
     * This method assign the radius of the circle at a new value.
     * @param radius The new value for the radius.
     */
    public void radius(int radius) {
	this.radius = radius;
    }    
        
    /**
     * This method draw a circle in a array of pixels.
     * @param pixels The array of pixels we want to draw the circle.
     * @param width  The width of the array of pixels.
     * @param height The height of the array of pixels.
     * @param color  The color the circle is painted.
     */
    public synchronized void drawCircle(int[] pixels,				   
					int width,
					int height,
					// int pointWidth,
					Color color) {
	
	int c=0;
	int delta = 0;
	int[] values = new int[2];
       
	
	/**
	 * We draw on times the circle with the y value in the equation.
	 */
	for (int y=Math.max(this.yOrigin(),0); y<Math.min(this.yOrigin()+2*this.radius,height); y++) {	    
	   
	    /**
	     * We find the parameters of the circle.
	     */
	    c = this.returnC(y,this.xCenter,this.yCenter,this.radius);
	    delta = this.solveDelta(1,-2*(this.xCenter),c);
	    
	    /**
	     * We assign a pixel to the color if it is on the border of the circle.
	     */
	    if (delta >= 0) {
		values = this.returnSolutions(1,-2*(this.xCenter),delta);
		if (values[0] >= 0 && values[0] < width) {		
		    pixels[(y*width)+values[0]]=color.getRGB();
		}
		if (values[1] >= 0 &&  values[1] < width) {	
		    pixels[(y*width)+values[1]]=color.getRGB();
		}
	    }	   
	}   
	
	/** 
	 * We make the same thing but with the x axis. This prevent us from having hole (border that are not drawing because of the precision) in the drawing form.
	 */
	for (int x=Math.max(this.xOrigin(),0); x<Math.min(this.xOrigin()+2*this.radius,width); x++) {
	    
	    c = this.returnC(x,this.yCenter,this.xCenter,this.radius);
	    delta = this.solveDelta(1,-2*this.yCenter,c);
	    
	    if (delta >=0) {
		values = this.returnSolutions(1,-2*this.yCenter,delta);
		if (values[0] >= 0 &&   values[0] < height) {		
		    pixels[(values[0]*width)+x] = color.getRGB();
		}
		if (values[1] >= 0 &&  values[1] < height) {		
		    pixels[(values[1]*width)+x] = color.getRGB();
		}
	    }	   
	}
    }

    /**
     * We draw the circle with a texture. 
     * @throws Texture.TextureError
     * @param pixels  The array of pixels we want to draw the circle.
     * @param width   The width of the array of pixels.
     * @param height  The height of the array of pixels.
     * @param texture The texture for the circle.
     */
    public synchronized void drawCircleWithTexture(int[] pixels,				   
						   int width,
						   int height,
						   Texture texture) throws TextureError{
			
	/**
	 * If the texture mode is positive then we have a true texture. Otherwise we have a uniform color for the texture.
	 */
	if (texture.mode() > 0) {
	    /**
	     * If we have the mode 1 for the texture than we draw an empty circle.
	     */
	    if (texture.mode() == 1) {
		this.drawCircle(pixels,width,height,Color.black);
	    }
	    else {							
		int index=0;	
		for (int y=Math.max(this.yOrigin(),0); y<Math.min(this.yOrigin()+2*this.radius,height); y++) {
		    for (int x=Math.max(this.xOrigin(),0); x<Math.min(this.xOrigin()+2*this.radius,width); x++) {	    
			if ((int)Math.pow((double)(x-this.xCenter),2.0) +
			    (int)Math.pow((double)(y-this.yCenter),2.0) <= 
			    (int)Math.pow((double)this.radius,2.0)) {	    		    
			    pixels[(y*width)+x] = texture.arraytexture()[index];			 
			}
			index++;
		    }
		}
	    }
	}
	else {
	    for (int y=Math.max(this.yOrigin(),0); y<Math.min(this.yOrigin()+2*this.radius,height); y++) {
		for (int x=Math.max(this.xOrigin(),0); x<Math.min(this.xOrigin()+2*this.radius,width); x++) {	    
		    if ((int)Math.pow((double)(x-this.xCenter),2.0) +
			(int)Math.pow((double)(y-this.yCenter),2.0) <= 
			(int)Math.pow((double)this.radius,2.0)) {	    
		    	pixels[(y*width)+x] = texture.color().getRGB();			 
		    }		
		}
	    }
	} 
    }
	
    /**
     * This method optimize the drawing of the circle on a animatedImage in drawing only the bounding rectangle of the circle.
     * @param animatedImage The animatedImage on which the circle must be drawn.
     */
    public void optimizeDrawing(AnimatedImage animatedImage) {
	animatedImage.updateAnimatedImage(this.xOrigin(),this.yOrigin(),this.width()+1,this.height()+1);
    }
    
    /**
     * Translation of the circle to a new point.
     * @param x         The x coordinate the center of the circle must be translated.
     * @param y         The y coordinate the center of the circle must be translated.
     * @param maxWidth  The maximum width the circle must be translated. This prevent the circle from going out of the the animatedImage.
     * @param maxHeight The maximum height the circle must be translated. This prevent the circle from going out of the the animatedImage.
     */
    public void translate(int x, int y, int maxWidth, int maxHeight) {

	/**
	 * We translate the circle with parameters upper.
	 */
	if (x-this.radius < 0) {
	    this.xOrigin(0);
	    this.xCenter=this.radius;
	}
	else if (x+this.radius>maxWidth) {
	    this.xOrigin(maxWidth-2*this.radius);
	    this.xCenter=maxWidth - this.radius;
	}
	else {
	    this.xCenter=x;
	    this.xOrigin(this.xCenter-this.radius);
	}
	if (y-this.radius < 0) {
	    this.yOrigin(0);
	    this.yCenter=this.radius;
	}
	else if (y+this.radius>maxHeight) {
	    this.yOrigin(maxHeight-2*this.radius);
	    this.yCenter=maxHeight - this.radius;
	}
	else {
	    this.yCenter=y;
	    this.yOrigin(this.yCenter-this.radius);
	} 

    }

    /**
     * This method tell if the forms is selected or not with the specified point.
     * @param x The x coordinate of the point.
     * @param y The y coordinate of the point.
     * @return  True if the point is in the forms false otherwise.
     */
    public boolean isSelected(int x, int y) {
	if ((int)Math.pow((double)(x-this.xCenter),2.0) +
	    (int)Math.pow((double)(y-this.yCenter),2.0) <= 
	    (int)Math.pow((double)this.radius,2.0)) {	    
	    return true;
	}
	else {
	    return false;
	}
    }

    /**
     * This method find the c value of the equation: (x-p)**2 + (var-q)**2 == r**2.
     * @param var The variable in the upper equation.
     * @param p   The x center coordinate of the circle.
     * @param q   The y center coordinate of the circle.
     * @param r   The radius of the circle.
     * @return    The C value of the equation.
     */     
    protected int returnC(int var, int p, int q, int r) {
	return (p*p + var*var -2*q*var + q*q - r*r);
    }

    /**
     * This method solves the delta equation (b-4ac).
     * @param a The a in the upper equation.
     * @param b The b in the upper equation.
     * @param c The c in the upper equation.
     * @return  The result of the equation upper.
     */
    protected int solveDelta (int a, int b, int c) {
	return (b*b - 4*a*c);
    }

    /**
     * This method returns the solutions of a second degree equation.
     * @param a     The a value in a second degree equation.
     * @param b     The b value in a second degree equation.
     * @param delta The delta value in a second degree equation. It can be found with the upper methods.
     * @return The solutions of a second degree equation
     */
    protected int[] returnSolutions(int a, int b, int delta) {
	int[] resultat = new int[2];
	resultat[0] = (int)(((float)-b+(float)Math.sqrt((double)delta))/((float)(2*a)));
	resultat[1] = (int)(((float)-b-(float)Math.sqrt((double)delta))/((float)(2*a)));
	return resultat;
    }

}
	
