/**
 * This package provides useful primitives to draw forms like circles, polygon or rectangle.
 * @author Sebastien Baehni
 * @version 1.0
 */
package Forms;

import java.awt.Color;
import java.awt.Polygon;

import AnimatedImage.*;
import Texture.*;

/**
 * This class construct a new polygon.
 * @author Sebastien Baehni
 * @see Forms.Forms
 * @version 1.0
 */
public class MyPolygon extends Forms {
    
    /**
     * Boolean to know if the polygon is drawn or not.
     */
    protected boolean polygonDrawn;
    /**
     * The precision of the polygon i.e: The user must specifies the last point of the 
     * polygon at least precision near the first point.
     */
    protected int precision;   
		
    /**
     * The last x coordinate the polygon has been translated.
     */
    protected int lastX=0;
    /**
     * The last y coordinate the polygon has been translated.
     */
    protected int lastY=0;
	
    /**
     * The polygon. This polygon class is given by java. This will be very useful when we must know 
     * if a point is in or out of a polygon.
     */
    protected Polygon polygon;	
	
    /**
     * The constructor.
     * @param precision The precision of the polygon (see the variable upper).
     */
    public MyPolygon(int precision) {
  	polygon = new Polygon();
	this.polygonDrawn = false;
  	this.precision = precision;
    }	
	
    /**
     * This method draw a polygon in a given array of pixels with the specified color.
     * @param pixels The array of pixels on which we want to draw the polygon.
     * @param width  The width of the array of pixels.
     * @param height The height of the arra of pixels.
     * @param x      The x coordinate of the current point to be drawn.
     * @param y      The y coordinate of the current point to be drawn.
     * @param color  The color we want to draw the polygon.
     */
    public synchronized void drawPolygon(int[] pixels,
					 int width,
					 int height,
					 int x, 
					 int y,
					 Color color) {	
	
	/** 
	 * We test if the current point is possible for our program.
	 */
	if (x<0) {
	    x=0;
	}
	if (x>width) {
	    x=width;
	}
	if (y<0) {
	    y=0;
	}
	if (y>height) {
	    y=height;
	}
	
	/**
	 * If the number of points of the polygon is upper than 2 then we can form a polygon.
	 * Otherwise we only draw segments.
	 */
	if (polygon.npoints > 2) {
	   	   	    
	    /**
	     * We test if the point is near the first one => we construct a polygon.
	     */	    
	    if ( ( ((x >= this.polygon.xpoints[0] - this.precision) 
		    && (x <= this.polygon.xpoints[0] + this.precision)) &&
		   (( y >= this.polygon.ypoints[0]  - this.precision) &&
		    (y <= this.polygon.ypoints[0]  +this.precision)) ) )
		{		
		    int[] lastPoint = new int[2];
		    lastPoint[0] = this.polygon.xpoints[this.polygon.npoints-1];
		    lastPoint[1] = this.polygon.ypoints[this.polygon.npoints-1];
		    int[] firstPoint = new int[2];
		    firstPoint[0] = this.polygon.xpoints[0];
		    firstPoint[1] = this.polygon.ypoints[0];
		   	
		    /**
		     * We draw the segment between the first and the last point.
		     */
		    this.drawSegment(pixels,
				     width,
				     height,
				     lastPoint,
				     firstPoint,				
				     color);
		    this.polygonDrawn = true;		
		    this.rectangle(this.polygon.getBounds());			  
		}
	    else {
		/**
		 * We have not finished our polygon. We add the current point, we draw the new segment.
		 */
		int[] currentPoint = {x,y};
		int[] lastPoint = new int[2];
		lastPoint[0] = this.polygon.xpoints[this.polygon.npoints-1];
		lastPoint[1] = this.polygon.ypoints[this.polygon.npoints-1];				
				
		this.polygon.addPoint(x,y);
				
	    	this.drawSegment(pixels,
				 width,
				 height,
				 lastPoint,				 
				 currentPoint,			      
				 color);		
	    }		    
	}	
	/**
	 * If the user has specified only two points we add the points and we draw the segments.
	 */
	else {
	    int[] currentPoint = {x,y};      	
		
	    this.polygon.addPoint(x,y);
	  	   
	    if (this.polygon.npoints > 1) {	    
	    	int[] lastPoint = new int[2];
		lastPoint[0] = this.polygon.xpoints[this.polygon.npoints-2];
		lastPoint[1] = this.polygon.ypoints[this.polygon.npoints-2];		    	  
	    	
		this.drawSegment(pixels,
				 width,
				 height,
				 lastPoint,
				 currentPoint,
				 color);				
	    }	    	    
	}	  
    }   
   
    /**
     * This method draw a segment in a given array of pixels.
     * @param pixels       The array of pixels on which we want to draw the segment.
     * @param width        The width of the array of pixels.
     * @param height       The height of the array of pixels.
     * @param firstPoint   The first point of the segment.
     * @param secondPoint  The second point of the segment.
     * @param colorSegment The color we want to draw the segment.
     */
    public synchronized void drawSegment(int[] pixels,			 
					 int width,
					 int height,
					 int[] firstPoint, 			   
					 int[] secondPoint, 			  
					 Color colorSegment) 
    {


	/**
	 * We find the min and maximum widht and height between of the two points.
	 * So we can draw a segment and not a line.
	 */
	int minWidth = Math.min(firstPoint[0],secondPoint[0]);
	int minHeight = Math.min(firstPoint[1],secondPoint[1]);
	int maxWidth = Math.max(firstPoint[0],secondPoint[0]);
	int maxHeight = Math.max(firstPoint[1],secondPoint[1]);	
	
	float[] parameters = this.returnLineParameters(firstPoint, secondPoint);
	
	/**
	 * We draw the segment in paying attention of the differents cases (m = infinity). 
	 * To know the m and h of the equation (y = mx +h), we use the methods defined below.
	 */
	for (int y = 0; y<height; y++) {
	    for (int x = 0; x<width; x++) {		
		if (parameters[0] == Float.POSITIVE_INFINITY) {
		    if (x == parameters[1] && y >= minHeight && y <= maxHeight) {
			pixels[(y*width)+x] = colorSegment.getRGB();
		    }		
		}
		else {		    
		    if (x>= minWidth && x<= maxWidth && y>= minHeight && y<= maxHeight) {
			if (parameters[0]>=-1 && parameters[0]<=1 ) {
			    if (y == (int)((float)x*parameters[0] + parameters[1])) {
				pixels[(y*width)+x] = colorSegment.getRGB();
			    }		
			}
			else {
			    if (x == (int)((float)y/parameters[0] - parameters[1]/parameters[0])) {
				pixels[(y*width)+x] = colorSegment.getRGB();				
			    		
			    }								
			}					    
		    }
		}
	    }				
	}			    										
    }         

    /**
     * This method draw a polygon with the given texture.
     * @throws Texture.TextureError
     * @param pixels  The array of pixels on which we want to draw the polygon.
     * @param width   The width of the array of pixels.
     * @param height  The height of the array of pixels.
     * @param texture The texture of the polygon.
     */
    public synchronized void drawPolygonWithTexture(int[] pixels,
						    int width,
						    int height,
						    Texture texture) throws TextureError
    {		
	/**
	 * If the mode of the texture is positive then we have a true texture. Otherwise we have a uniform color.
	 */
	if (texture.mode()>0) {
	    /**
	     * If the texture's mode equals one then we draw an empty polygon.
	     */
	    if (texture.mode() ==1) {
		int[] currentPoint = new int[2];
		int[] nextPoint = new int[2];
		/**
		 * We draw segment by segment.
		 */
		for (int i=0; i<this.polygon.npoints-1; i++) {
		    currentPoint[0] = this.polygon.xpoints[i];
		    currentPoint[1] = this.polygon.ypoints[i];
		    nextPoint[0] = this.polygon.xpoints[i+1];
		    nextPoint[1] = this.polygon.ypoints[i+1];
		    this.drawSegment(pixels,width,height,currentPoint,nextPoint,Color.black);
		}
		/**
		 * Segment between the first and the last point.
		 */
		currentPoint[0] = this.polygon.xpoints[0];
		currentPoint[1] = this.polygon.ypoints[0];
		nextPoint[0] = this.polygon.xpoints[this.polygon.npoints-1];
		nextPoint[1] = this.polygon.ypoints[this.polygon.npoints-1];
		this.drawSegment(pixels,width,height,currentPoint,nextPoint,Color.black);				
	    }
	    else {
		/**
		 * We draw the polygon with the texture.
		 */
		int index =0;				
		for (int y=Math.max(this.yOrigin(),0); y<Math.min(this.yOrigin()+this.height(),height); y++) {
		    for (int x=Math.max(this.xOrigin(),0); x<Math.min(this.xOrigin()+this.width(),width); x++) {			
			if (polygon.contains(x,y)) {
			    pixels[(y*width)+x] = texture.arraytexture()[index];					
			}	    	
			index++;
		    }				
		}				
	    }
	}
	else {
	    /**
	     * We draw the polygon with a uniform color.
	     */
	    for (int y=Math.max(this.yOrigin(),0); y<Math.min(this.yOrigin()+this.height(),height); y++) {
		for (int x=Math.max(this.xOrigin(),0); x<Math.min(this.xOrigin()+this.width(),width); x++) {					   
		    if (polygon.contains(x,y)) {
			pixels[(y*width)+x] = texture.color().getRGB();					
		    }	    	
		}				
	    }
	}
    }		
	
    /**
     * This method optimize the drawing of the polygon. It refreshes only the bounding box of the polygon.
     */
    public void optimizeDrawing(AnimatedImage animatedImage) {	
	animatedImage.updateAnimatedImage(this.xOrigin(),this.yOrigin(),this.width(),this.height());
    }
	
    /**
     * This method return the line parameters m and h.
     * @param firstPoint  The first point of the line.
     * @param secondPoint The second point of the line.
     * @return An array of two elements containing the line parameters. The first element is m and the second one is h.
     */
    protected float[] returnLineParameters(int[] firstPoint, int[] secondPoint) 
    {	
	float parameters[] = new float[2];
	/**
	 * When we have a non-deterministic equation (by exemple when firstPoint[0] ==
	 * firstPoint[1] == secondPoint[0] == secondPoint[1], the returned value is NAN.
	 * When we have an vertical line, the returned value for m is + Infinity 
	 * and for h is firstPoint[0] or secondPoint[0].
	 * In all other cases we calculate the right m and h parameters. 
	 */
	
	/**
	 *When we solve the equation y1 = mx1 + h and y2 = mx2 + h we must pay attention
	 * to some cases: x2 = 0, x1 = x2.
	 */
	if (secondPoint[0] == 0.0f) {
	    /**
	     * When x2 == 0.0f we must pay attention to x1 == 0.0f.
	     */
	    if (firstPoint[0] == 0.0f) {
		/**
		 * We have the y axis for the line. 
		 */
		parameters[1] = 0.0f;
		parameters[0] = Float.POSITIVE_INFINITY;
	    }
	    else {						
		parameters[1] = secondPoint[1];
		parameters[0] = (firstPoint[1] - secondPoint[1])/firstPoint[0];
	    }
	}     
	else if (firstPoint[0] == secondPoint[0]) {    
	    if (firstPoint[1] == secondPoint[1]) {			
		/**
		 * NOT DETERMINED.
		 */ 		    
		parameters[1] = Float.NaN;
		parameters[0] = Float.NaN;         
	    }               
	    else {		    
		/** 
		 * This is a vertical line.
		 */		    
		parameters[1] = firstPoint[0];		    
		parameters[0] = Float.POSITIVE_INFINITY;  					
	    }		
	}	
	else {	    
	    /**
	     * This is a regular line. 
	     */
	    /** 
	     * This is h.
	     */
	    parameters[1] = 
		(firstPoint[1] - (((float)firstPoint[0]/secondPoint[0])*secondPoint[1]))/(1-((float)firstPoint[0]/secondPoint[0]));
	    /**
	     * This is m.
	     */
	    parameters[0] = ((float)(secondPoint[1] - parameters[1]))/secondPoint[0];	
	}		
	return parameters;	    
    }
   
    /**
     * This method tells if a point is in the polygon or not.
     * @param x The x coordinate of the point we test if it's containing in the polygon.
     * @param y The y coordinate of the point we test if it's containing in the polygon.
     * @return True if the point is in the polygon, false otherwise.
     */
    public boolean isSelected(int x, int y) {			
	return this.polygon.contains(x,y);		    
    }
  
    /**
     * This method translate the polygon to the specified point.
     * @param x         The x coordinate the center of the polygon must be translated.
     * @param y         The y coordinate the center of the polygon must be translated.
     * @param maxWidth  The maximum width the polygon must be translated. This prevent the polygon from going out of the the animatedImage.
     * @param maxHeight The maximum height the polygon must be translated. This prevent the polygon from going out of the the animatedImage.     
     */
    public void translate(int x, int y, int maxWidth, int maxHeight) {
	
	/**
	 * We translate the polygon with the upper parameters.
	 */
	if (x-this.width()/2 < 0) {
	    /**
	     * WARNING: We can't put x to zero because something strange happens.
	     */
	    x=1;	    
	}
	else if (x+this.width()/2 > maxWidth) {
	    x=maxWidth-this.width()-1;	    
	}
	else {
	    x=x-this.width()/2;	      
	}
	if (y-this.height()/2 < 0) {
	    /**
	     * WARNING: We can't put y to zero because something strange happens.
	     */
	    y=1;
	}
	else if (y+this.height()/2 > maxHeight) {
	    y=maxHeight-this.height()-1;	    
	}
	else {
	    y=y-this.height()/2;	    
	}		
	
	/**
	 * The translate method of the polygon class add all coordinates of the polygon
	 * the number specified in its method. So in order to not have a polygon that 
	 * is translated to +10 then +10+20 ... We must translate it to -10 then to +20 then
	 * we have the right position.
	 */
	this.polygon.translate(this.lastX,this.lastY);
	this.polygon.translate(x-this.xOrigin(),y-this.yOrigin());		
	this.rectangle(this.polygon.getBounds());
	/**
	 * We keep in mind the lastX and the lastY position.
	 */
	this.lastX= -x+this.xOrigin();
	this.lastY = -y+this.yOrigin();		

    }	
  
    /**
     * This method tells if the polygon is finished or not.
     * @return True if the polygon is finished, false otherwise.
     */
    public boolean polygonDrawn() {
	return this.polygonDrawn;
    }			      			      
	
}
   
