/**
 * This package provides useful primitives to draw forms like circles, polygon or rectangle.
 * @author Sebastien Baehni
 * @version 1.0
 */
package Forms;

import java.awt.Color;

import AnimatedImage.*;
import Texture.*;

/**
 * This class construct a new rectangle.
 * @author Sebastien Baehni
 * @see Forms.Forms
 * @version 1.0
 */
public class MyRectangle extends Forms {     	

    
    /**
     * The constructor.
     * @param xOrigin The x origin of a corner of the rectangle.
     * @param yOrigin The y origin of a corner of the rectangle.
     * @param width   The width of the rectangle.
     * @param height  The height of the rectangle.
     */
    public MyRectangle(int xOrigin, int yOrigin, int width, int height) 
    {
	super(xOrigin,yOrigin,Math.abs(width),Math.abs(height));  	
    }           
	
    /**
     * This method draw a rectangle in an array of pixels with the specified color.
     * @param pixels The array of pixels on which we want to draw the rectangle.
     * @param width  The width of the array of pixels.
     * @param height The height of the arra of pixels.
     * @param color  The color we want to draw the rectangle.
     */
    public synchronized void drawRectangle(int[] pixels,
					   int width,
					   int height,
					   Color color) 
    {		
	  
	for (int y=Math.max(this.yOrigin(),0); y<Math.min(this.yOrigin()+this.height(),height); y++) {
	    for (int x=Math.max(this.xOrigin(),0); x<Math.min(this.xOrigin()+this.width(),width); x++) {
		/**
		 * The first horizontal segment.
		 */
		if (y == this.yOrigin()) {
		    pixels[(y*width)+x] = color.getRGB();
		}
		/**
		 * The second horizontal segment.
		 */	
		if (y == this.yOrigin()+this.height()-1) {
		    pixels[(y*width)+x] = color.getRGB();
		}
		/**
		 * The first vertical segment.
		 */
		if (x == this.xOrigin()) {
		    pixels[(y*width)+x] = color.getRGB();
		}
		/**
		 * The second vertical segment.
		 */	
		if (x == this.xOrigin()+this.width()-1) {
		    pixels[(y*width)+x] = color.getRGB();
		}
	    }
	}		
    }

    /**
     * This method draw a rectangle with the given texture.
     * @throws Texture.TextureError
     * @param pixels  The array of pixels on which we want to draw the rectangle.
     * @param width   The width of the array of pixels.
     * @param height  The height of the array of pixels.
     * @param texture The texture of the rectangle.
     */
    public synchronized void drawRectangleWithTexture(int[] pixels,
						      int width,
						      int height,
						      Texture texture) throws TextureError
    {		  	  	
	
	/**
	 * If the mode of the texture is positive then we have a true texture. Otherwise we have a uniform color.
	 */
	if (texture.mode()>0) {
	    /**
	     * If the texture's mode equals one then we draw an empty rectangle.
	     */
	    if (texture.mode() == 1) {	
		this.drawRectangle(pixels,width,height,Color.black);
	    }
	    /**
	     * We draw the rectangle with the texture.
	     */
	    else {
		int index =0;
		for (int y=Math.max(this.yOrigin(),0); y<Math.min(this.yOrigin()+this.height(),height); y++) {
		    for (int x=Math.max(this.xOrigin(),0); x<Math.min(this.xOrigin()+this.width(),width); x++) {	
			pixels[(y*width)+x] = texture.arraytexture()[index];		  
			index++;
		    }
		}	
	    }
	}	
	/**
	 * We draw the rectangle with a uniform color.
	 */
	else {
	    for (int y=Math.max(this.yOrigin(),0); y<Math.min(this.yOrigin()+this.height(),height); y++) {
		for (int x=Math.max(this.xOrigin(),0); x<Math.min(this.xOrigin()+this.width(),width); x++) {		
		    pixels[(y*width)+x] = texture.color().getRGB();		  	
		}
	    }
	}
	
    }

    /**
     * This method optimize the drawing of the rectangle. It refreshes only the bounding box of the rectangle.
     */
    public void optimizeDrawing(AnimatedImage animatedImage) 
    {
	animatedImage.updateAnimatedImage(this.xOrigin(),this.yOrigin(),this.width(), this.height());
    }
     
    /**
     * This method tells if a point is in the rectangle or not.
     * @param x The x coordinate of the point we test if it's containing in the rectangle.
     * @param y The y coordinate of the point we test if it's containing in the rectangle.
     * @return True if the point is in the rectangle, false otherwise.
     */
    public boolean isSelected(int x, int y) {	      
  	
  	return (this.rectangle.contains(x,y));
  		
    }
    
    /**
     * This method translate the rectangle to the specified point.
     * @param x         The x coordinate the center of the rectangle must be translated.
     * @param y         The y coordinate the center of the rectangle must be translated.
     * @param maxWidth  The maximum width the rectangle must be translated. This prevent the rectangle from going out of the the animatedImage.
     * @param maxHeight The maximum height the rectangle must be translated. This prevent the rectangle from going out of the the animatedImage.     
     */
    public void translate(int x, int y, int maxWidth, int maxHeight) {

	/**
	 * We translate the rectangle with the upper parameters.
	 */
	if (x-this.width()/2 < 0) {
	    this.xOrigin(0);	   
	}
	else if (x+this.width()/2 > maxWidth) {
	    this.xOrigin(maxWidth-this.width());
	}
	else {
	    this.xOrigin(x-this.width()/2);
	}
	if (y-this.height()/2 < 0) {
	    this.yOrigin(0);
	}
	else if (y+this.height()/2 > maxHeight) {
	    this.yOrigin(maxHeight-this.height());
	}
	else {
	    this.yOrigin(y-this.height()/2);
	}	
    }
}
