/**
 * This package implements a gray color chooser.
 * @author Sebastien Baehni
 * @version 1.0
 */
package GrayColorChooser;

import java.awt.Canvas;
import java.awt.Cursor;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.event.*;
import java.awt.image.MemoryImageSource;

/**
 * This class give us a canvas to paint the gray colors.
 * @author Sebastien Baehni
 * @version 1.0
 */
public class GrayCanvas extends Canvas {  
            
    /**
     * Our image which is going to be drawn in the canvas.
     * @serial
     */
    protected Image image;          
    
    /**
     * We are sure that the image is created before we make a call to the repaint method.
     * @serial
     */
    protected boolean painted = false;					             
    
    /**
     * Principal Constructor.
     * @param mouseListener       The canvas is sensible to the mouse.
     * @param mouseMotionListener the canvas is sensible to the motion of the mouse.
     */
    public GrayCanvas(MouseListener mouseListener, MouseMotionListener mouseMotionListener) {	
	
	/**
	 * The width and the height of the image. 
	 */
	int width = 256;
	int height = 256;
	
	/**
	 * This our array describing the MemoryImageSource.
	 */
	int[] pixels = new int[width*height]; 
	
	/**
	 * Creation of our memory image source. This is an image in wich we can see all colors composed with 
	 * all red, all green and a blue component of 128.
	 */
	for (int y=0; y<height; y++) {
	    for (int x=0; x<width; x++) {		
		pixels[(y*width)+x] = (255<<24)|(x<<16)|(x<<8)|x;			
	    }
	}	
	
	/**
	 * Our memory image source to optimized the drawing .
	 */
	MemoryImageSource memory_image_source = new MemoryImageSource(width,height,pixels,0,width);
	image = createImage(memory_image_source);
	
	/**
	 * The canvas is a mouse listener in oder to know the mouse position on the image.
	 */
	this.addMouseListener(mouseListener);
	/**
	 * The canvas as a mousemotion listener so we can know where the mouse when it is dragged.
	 */
	this.addMouseMotionListener(mouseMotionListener);
	/**
	 * A crosshair cursor for our canvas.
	 */
	this.setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));  
	/**
	 * Our canvas is 256 width so we can paint all the gray colors.
	 */
	this.setSize(width,height);
	
    }
    
    /**
     * Method in which we create the differant images.
     */
    public void start() {
	
	/**
	 * Now we can repaint the graphics.
	 */
	painted = true;
	this.repaint();
	
    }

    /**
     * This method repaint the canvas.
     * @see java.awt.Graphics
     * @param g The graphics on which we paint the image.
     */
    public void paint(Graphics g) {      
	/**
	 * We only paint the image on the graphics if the image has as value.
	 */
	if (painted) {
	    g.drawImage(image,0,0,this);
	}
    }
    
    /**
     * Optimized method for the repaint.
     * @see java.awt.Graphics
     * @param g The graphics on which we paint the image.
     */
    public void update(Graphics g) {
	/**
	 * We don't need to clear the background.
	 */
	if (painted) {
	    this.paint(g);
	}
    }
}
