/**
 * This package provides tools to draw the help in a frame from a simple file.
 * @author Sebastien Baehni
 * @version 1.0
 */
package Help;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.*;
import java.awt.Font;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.Image;
import java.awt.Label;
import java.awt.List;
import java.awt.Panel;
import java.awt.TextArea;
import java.awt.Toolkit;

import java.io.*;
import java.net.*;

/**
 * This is the class that provides the frame in which we draw the help.
 * @author Sebastien
 * @version 1.0
 */
public class Help extends Frame implements WindowListener {

    /**
     * The menu list.
     * @serial
     */
    protected List mainIndex;
    /**
     * The sub menu list
     * @serial
     */
    protected List subIndex;
    /**
     * The area in which we draw the help.
     * @serial
     */
    protected TextArea textArea;
    /**
     * The structure in which there is the help.
     * @serial
     */
    protected MyMenu[] menu;
	
    /**
     * The label for the menu list.
     * @serial
     */
    protected Label menuIndex;
    /**
     * The label for the sub menu list.
     * @serial
     */
    protected Label subMenuIndex;

	
    //Panels for the frame.	
    /**
     * The panel that contains the indexes.
     * @serial
     */
    protected Panel indexPanel;
    /**
     * The panel that contains the image and the index.
     * @serial
     */
    protected Panel mainPanel;
    /**
     * The panel that contains the main menu index and its label.
     * @serial
     */
    protected Panel indexNorthPanel;
    /**
     * The panel that contains the main menu index.
     * @serial
     */
    protected Panel menuIndexPanel;	
    /**
     * The panel that contains the sub menu index and its label.
     * @serial
     */
    protected Panel indexSouthPanel;
    /**
     * The panel that contains the sub menu index.
     * @serial
     */
    protected Panel subMenuIndexPanel;

    /**
     * Actionlistener for the menu list.
     * @serial
     */
    protected ActionListener mainIndexActionListener;
    /**
     * Actionlistener for the sub menu list.
     * @serial
     */
    protected ActionListener subIndexActionListener;
	
    /**
     * The canvas in which we draw the image.
     * @serial
     */
    protected HelpImageCanvas helpImageCanvas;
	
    /**
     * We can know if the frame is open or not.
     * @serial
     */
    protected boolean helpFrameOpen = false;
	
    /**
     * This is a boolean to know if the help file has been correctly read or not.
     * @serial
     */
    protected boolean helpFileProblem = false;
	
    /**
     * This is a string to tell the user what's wrong with the help file.
     * @serial
     */
    protected String problemString = "";

    /**
     * The main constructor. It creates all the panel and get the help from a file
     * that was parsed with the help file parser.
     * @see Help.HelpFileParser
     * @param fileName The fileName of the help file.
     */
    public Help(URL fileName) {
	/**
	 * We create all the structures.
	 */
	super("Help about the Retina Applet");
	this.helpImageCanvas = new HelpImageCanvas();
	this.mainIndex = new List();
	this.subIndex = new List();
	this.textArea = new TextArea();
	this.menuIndex = new Label("Menu Index");
	this.subMenuIndex = new Label("SubMenu Index");
	this.menuIndexPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.subMenuIndexPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.indexPanel = new Panel(new GridLayout(2,1));
	this.indexNorthPanel = new Panel(new BorderLayout());
	this.indexSouthPanel = new Panel(new BorderLayout());
	this.mainPanel = new Panel(new GridLayout(2,1));
	this.setLayout(new BorderLayout());
	/**
	 * We get the help from a help file and we put it in a menu structure.
	 * @see Help.MyMenu
	 */
	this.menu = this.readStructure(fileName);				
	
	if (!this.helpFileProblem) {
	    this.initListsItems();
	    this.initActionListener();
	}
    }

 /**
     * The main constructor. It creates all the panel and get the help from a file
     * that was parsed with the help file parser.
     * @see Help.HelpFileParser
     * @param fileName The fileName of the help file.
     */
    public Help(String fileName) {
	/**
	 * We create all the structures.
	 */
	super("Help about the Retina Applet");
	this.helpImageCanvas = new HelpImageCanvas();
	this.mainIndex = new List();
	this.subIndex = new List();
	this.textArea = new TextArea();
	this.menuIndex = new Label("Menu Index");
	this.subMenuIndex = new Label("SubMenu Index");
	this.menuIndexPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.subMenuIndexPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.indexPanel = new Panel(new GridLayout(2,1));
	this.indexNorthPanel = new Panel(new BorderLayout());
	this.indexSouthPanel = new Panel(new BorderLayout());
	this.mainPanel = new Panel(new GridLayout(2,1));
	this.setLayout(new BorderLayout());
	/**
	 * We get the help from a help file and we put it in a menu structure.
	 * @see Help.MyMenu
	 */
	this.menu = this.readStructure(fileName);				
	
	if (!this.helpFileProblem) {
	    this.initListsItems();
	    this.initActionListener();
	}
    }

    /**
     * Init of our frame. 
     * @param image The image that we will display in our frame.
     */
    public void init(Image image) {		
	/**
	 * We add all the panel, the listener and we initialize them.
	 */
	this.addWindowListener(this);
	this.mainIndex.addActionListener(this.mainIndexActionListener);	
	this.subIndex.addActionListener(this.subIndexActionListener);
	this.add(this.indexPanel,BorderLayout.WEST);
	this.add(this.mainPanel,BorderLayout.CENTER);
	this.menuIndex.setFont(new Font("Dialog",Font.BOLD,16));
	this.menuIndex.setForeground(Color.red);
	this.subMenuIndex.setFont(new Font("Dialog",Font.BOLD,14));
	this.subMenuIndex.setForeground(Color.blue);
		
	this.textArea.setEditable(false);
	this.menuIndexPanel.add(this.menuIndex);
	this.subMenuIndexPanel.add(this.subMenuIndex);
	this.indexNorthPanel.add(this.menuIndexPanel,BorderLayout.NORTH);
	this.indexNorthPanel.add(this.mainIndex,BorderLayout.CENTER);
	this.indexSouthPanel.add(this.subMenuIndexPanel,BorderLayout.NORTH);
	this.indexSouthPanel.add(this.subIndex,BorderLayout.CENTER);
	this.indexPanel.add(this.indexNorthPanel);
	this.indexPanel.add(this.indexSouthPanel);
	
	this.mainPanel.add(this.helpImageCanvas);
	this.mainPanel.add(this.textArea);			
	this.helpImageCanvas.init(image);	
	this.helpImageCanvas.repaint();		
    }
	
    /**
     * Init of our frame used when we have a nullpointerexception. 
     */
    public void init() {		
	/**
	 * We add all the panel, the listener and we initialize them.
	 */
	this.addWindowListener(this);
	this.mainIndex.addActionListener(this.mainIndexActionListener);	
	this.subIndex.addActionListener(this.subIndexActionListener);
	this.add(this.indexPanel,BorderLayout.WEST);
	this.add(this.mainPanel,BorderLayout.CENTER);
	this.menuIndex.setFont(new Font("Dialog",Font.BOLD,16));
	this.menuIndex.setForeground(Color.red);
	this.subMenuIndex.setFont(new Font("Dialog",Font.BOLD,14));
	this.subMenuIndex.setForeground(Color.blue);
		
	this.textArea.setEditable(false);
	this.menuIndexPanel.add(this.menuIndex);
	this.subMenuIndexPanel.add(this.subMenuIndex);
	this.indexNorthPanel.add(this.menuIndexPanel,BorderLayout.NORTH);
	this.indexNorthPanel.add(this.mainIndex,BorderLayout.CENTER);
	this.indexSouthPanel.add(this.subMenuIndexPanel,BorderLayout.NORTH);
	this.indexSouthPanel.add(this.subIndex,BorderLayout.CENTER);
	this.indexPanel.add(this.indexNorthPanel);
	this.indexPanel.add(this.indexSouthPanel);
			
	this.mainPanel.add(this.textArea);									
    }

    /**
     * This method show the frame with the right size.
     */
    public void start() {	
	this.setSize(800,600);	
	Dimension dimension = Toolkit.getDefaultToolkit().getScreenSize();
	this.setLocation(Math.abs((dimension.width-this.getSize().width)/2),Math.abs((dimension.height-this.getSize().height)/2));
	this.show();
	if (!this.helpFileProblem) {
	    this.setText("\nDouble click on a menu to see the help about it");
	}
	else {
	    this.setText(this.problemString);
	}
	this.subIndex.removeAll();
	this.helpFrameOpen = true;
    }

    /**
     * This method init the different action listener of the lists.
     */
    protected void initActionListener() {
	this.mainIndexActionListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		/**
		 * If the user click on the list, we update the textarea and the sublist.
		 */
		for (int i=0; i<menu().length; i++) {
		    if ( ((i+1)+". "+(menu()[i].name())).compareTo(e.getActionCommand()) == 0 ) {
			updateSubList(i);
		    }
		}
	    }
	};
	this.subIndexActionListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		/**
		 * If the user click on the sub list, we display a help menu.
		 */
		for (int i=0; i<menu().length; i++) {
		    if (menu()[i].hasSubMenu()) {
			for (int j=0; j<menu()[i].subMenu().length; j++) {						
			    if ( ((j+1)+". "+(menu()[i].subMenu()[j]).name()).compareTo(e.getActionCommand()) == 0) {
				updateSubText(i,j);
			    }
			}
		    }
		} 																				  								
	    }
	};
    }  

    /**
     * We create the title in the main menu.
     */
    protected void initListsItems() {
	for (int i=0; i<this.menu.length; i++) {			
	    this.mainIndex.add((i+1)+". "+this.menu[i].name());
	}
    }
	
    /**
     * This method updates the sublist with the right menu. It draws in the textarea a help
     * message.
     * @param index The number of the menu that the user has clicked on.
     */
    public void updateSubList(int index) {		
	this.subIndex.removeAll();
	if (this.menu[index].hasSubMenu()) {
	    for (int i=0; i<(this.menu[index]).subMenu().length; i++) {
		this.subIndex.add((i+1)+". "+(this.menu[index]).subMenu()[i].name());
	    }
	}
	this.setText(this.menu[index].text());
    }
	
    /**
     * This method display the help message corresponding to the sub title that the user
     * has clicked on.
     * @param menuIndex The main menu index.
     * @param subMenuIndex The index of the sub menu the user has clicked on.
     */
    public void updateSubText(int menuIndex, int subMenuIndex) {
	this.setText(this.menu[menuIndex].subMenu()[subMenuIndex].text());
    }
	
    /**
     * This method read the structure of a help file that has been parsed by the 
     * HelpFileParser.
     * @see Help.HelpFileParser
     * @see Help.MyMenu
     * @param  fileName The name of the file we want to get the structure.
     * @return The menu structure.
     */
    protected MyMenu[] readStructure(String fileName) {
	MyMenu[] menu = null;
	try {
	    /**
	     * We get the structure from the file.
	     */			
	    FileInputStream fileIStream = new FileInputStream(fileName);
	    ObjectInputStream stream = new ObjectInputStream(fileIStream);
		
	    menu = (MyMenu[])stream.readObject();		
	
	    fileIStream.close();
	}
	catch (FileNotFoundException e) {
	    this.problemString="\nThe specified help file was not found.\nBe sure to have a help.txt.str file in the helpfiles directory.";
	    this.helpFileProblem=true;			
	}
	catch (ClassNotFoundException e) {
	    this.problemString="\nClass of a serialized object cannot be found.\nPlease mail to Sebastien.Baehni@epfl.ch about this";
	    this.helpFileProblem=true;			
	}
	catch (InvalidClassException e) {
	    this.problemString="\nSomething is wrong with a class used by serialization.\nPlease mail to Sebastien.Baehni@epfl.ch about this";
	    this.helpFileProblem=true;
	}
	catch (OptionalDataException e) {
	    this.problemString="\nPrimitive data was found in the stream instead of objects.\nPlease mail to Sebastien.Baehni@epfl.ch about this";
	    this.helpFileProblem=true;
	}
	catch (StreamCorruptedException e) {
	    this.problemString="\nThe file specified in the helpfiles directory has a bad type.\nBe sure to have a correct .str file";
	    this.helpFileProblem=true;
	}
	catch(IOException e) {
	    this.problemString="\nAn input output exception occurs.\nPlease try again.";
	    this.helpFileProblem=true;			
	}					
	return menu; 
    }

    /**
     * This method read the structure of a help file that has been parsed by the 
     * HelpFileParser.
     * @see Help.HelpFileParser
     * @see Help.MyMenu
     * @param  fileName The name of the file we want to get the structure.
     * @return The menu structure.
     */
    protected MyMenu[] readStructure(URL fileName) {
	MyMenu[] menu = null;
	try {
	    /**
	     * We get the structure from the file.
	     */			
	    InputStream fileIStream = fileName.openStream();
	    ObjectInputStream stream = new ObjectInputStream(fileIStream);
		
	    menu = (MyMenu[])stream.readObject();		
	
	    fileIStream.close();
	}
	catch (FileNotFoundException e) {
	    this.problemString="\nThe specified help file was not found.\nBe sure to have a help.txt.str file in the helpfiles directory.";
	    this.helpFileProblem=true;			
	}
	catch (ClassNotFoundException e) {
	    this.problemString="\nClass of a serialized object cannot be found.\nPlease mail to Sebastien.Baehni@epfl.ch about this";
	    this.helpFileProblem=true;			
	}
	catch (InvalidClassException e) {
	    this.problemString="\nSomething is wrong with a class used by serialization.\nPlease mail to Sebastien.Baehni@epfl.ch about this";
	    this.helpFileProblem=true;
	}
	catch (OptionalDataException e) {
	    this.problemString="\nPrimitive data was found in the stream instead of objects.\nPlease mail to Sebastien.Baehni@epfl.ch about this";
	    this.helpFileProblem=true;
	}
	catch (StreamCorruptedException e) {
	    this.problemString="\nThe file specified in the helpfiles directory has a bad type.\nBe sure to have a correct .str file";
	    this.helpFileProblem=true;
	}
	catch(IOException e) {
	    this.problemString="\nAn input output exception occurs.\nPlease try again.";
	    this.helpFileProblem=true;			
	}					
	return menu; 
    }

    /**
     * This method set a help message on the TextArea with a specified color.
     * @param string The help message.
     * @param color  The color of the help message.
     */
    public void setText(String string) {
	this.textArea.setText(string);
    }
	
    /**
     * This method returns the structures of all the different menus.
     * @see Help.MyMenu
     * @return The menu structure.
     */
    public MyMenu[] menu() {
	return this.menu;
    }
  
    /**
     * If the help frame is open or not.
     * @return True if the frame is open. False otherwise.
     */
    public boolean helpFrameOpen() {
	return this.helpFrameOpen;
    }
	
    /**
     * To close the window.  
     * @param e The window event.
     */
    public void windowClosing(WindowEvent e) {	
	this.dispose();  		
	this.helpFrameOpen = false;
    }
    
    /**
     * When you open the window.   
     * @param e The window event.
     */     
    public void windowOpened(WindowEvent e){}
    /**
     * When you iconify the window.   
     * @param e The window event.
     */     
    public void windowIconified(WindowEvent e) {}
    /**
     * When you deiconify the window.   
     * @param e The window event.
     */     
    public void windowDeiconified(WindowEvent e) {}
    /**
     * When you deactivate the window.   
     * @param e The window event.
     */     
    public void windowDeactivated(WindowEvent e) {}
    /**
     * When you close the window.   
     * @param e The window event.
     */     
    public void windowClosed(WindowEvent e) {}
    /**
     * When you activate the window.   
     * @param e The window event.
     */     
    public void windowActivated(WindowEvent e) {} 
    

}
