/**
 * This package provides tools to draw the help in a frame from a simple file.
 * @author Sebastien Baehni
 * @version 1.0
 */
package Help;

import java.io.*;
import java.util.Vector;
import java.lang.ClassNotFoundException;

/**
 * This class provides the parser for reading a help file. After the file is parser, we 
 * write another file that contains the structure of menu we have just found.
 * @author Sebastien Baehni
 * @version 1.0
 * @see Help.MyMenu
 */
public class HelpFileParser{

    /**
     * A random access file.
     */
    RandomAccessFile randomFile;
    /**
     * The two string that we are going to search in the file.
     * Before we have a title we have the "/title;" tag and before we have
     * a subtitle we have the "/stitle;" tag.
     */
    String title = "/title;";
    String subTitle = "/stitle;"; 
    /**
     * The current position of the file pointer.
     */
    long currentPosition=0;
        
    /**
     * The main constructor. It creates a new random access file.
     * @param fileName The name of the file we want to parse.
     */
    public HelpFileParser(String fileName) {
	try {
	    this.randomFile = new RandomAccessFile(fileName,"r");
	}
	catch (FileNotFoundException e) {
	    System.out.println("File "+fileName+" not found");
	    System.exit(-1);
	}
    }

    /**
     * This method finds a title and all his subtitles.
     * @param startPosition The position of the file pointer.
     * @return An array of long that contains all the position of the differents title. The title is in first position.
     */
    protected Long[] findTitleAndHisSubTitles(long startPosition) {
	Vector titleVector = new Vector();	
	/**
	 * The position of the title.
	 */
	titleVector.add(new Long(startPosition));
	/**
	 * We find all his subtitles.
	 */
	this.findSubTitles(titleVector);		
	
	/**
	 * We create our array of long.
	 */
        Long[] paragraph = new Long[titleVector.size()];
	titleVector.toArray(paragraph);
	return paragraph;
    } 

    /**
     * This method find all the subtitles between two titles.
     * @param titleVector The vector in which the position of the title and the subtitles are.
     */
    protected void findSubTitles(Vector titleVector) {
	String str;
	try {
	    /**
	     * We put in the vector the previous position of the subtitle.
	     */
	    long previousPosition = this.randomFile.getFilePointer();
	    while( (str = this.randomFile.readLine()) != null) {	
		/**
		 * We have found a title, we stop the loop.
		 */
		if (str.regionMatches(true,0,this.title,0,this.title.length())) {				  		
		    this.currentPosition = previousPosition;
		    break;
		}
		/**
		 * We found the sub title.
		 */
		if (str.regionMatches(true,0,this.subTitle,0,this.subTitle.length())) {				  
		    titleVector.add(new Long(previousPosition));
		}
		previousPosition =  this.randomFile.getFilePointer();
	    }
	    /**
	     * We are at the end of the file. We stop searching.
	     */
	    if (str == null) {
		this.currentPosition = this.randomFile.length();
	    }
	}
	catch (IOException e) {
	    e.printStackTrace();
	}
    }    

    /**
     * This method find all the paragraphs (titles and subtitles)in a file.
     * @return A vector that contains multiple array of long in which they are the position of the titles and subtitle.
     *         an element of the vector represents a paragraph.
     */
    protected Vector findAllParagraphs() {
	Vector paragraphs = new Vector();
	try {	 	 
	    /**
	     * If the text doesn't begin with a title tag.
	     */
	    while ( !this.randomFile.readLine().regionMatches(true,0,this.title,0,this.title.length()) ) {
		this.currentPosition = this.randomFile.getFilePointer();
	    }	 	  
	    /**
	     * We find all the paragraphs and we put them in the vector.
	     */
	    while (this.currentPosition < this.randomFile.length()) {
		paragraphs.add(this.findTitleAndHisSubTitles(this.currentPosition));	
	    }
	}
	catch (IOException e) {
	    e.printStackTrace();
	}
	catch (NullPointerException e) {
	    System.out.println("The file doesn't contain any title");
	    System.exit(-1);
	}
	return paragraphs;
    }
	   
    /**
     * This method create a submenu structure. This is a menu with a name (the title name),
     * a text and no sub menu.
     * @see Help.MyMenu
     * @param startPosition The start position of the file pointer.
     * @param stopPosition The stop position of the file pointer.
     */
    protected MyMenu createSubTitleMenu(long startPosition, long stopPosition) {
	String subTitleName="";
	String subTitleText="";
	try {
	    this.randomFile.seek(startPosition);
	    /**
	     * We get the tag.
	     */		
	    this.randomFile.readLine();
	    /**
	     * We get the title name.
	     */
	    subTitleName = this.randomFile.readLine();
	    /**
	     * We get the text.
	     */
	    while ( this.randomFile.getFilePointer() < stopPosition) {
		subTitleText += this.randomFile.readLine() + "\n";
	    }	
	}
	catch (IOException e) {
	    e.printStackTrace();
	}
	return (new MyMenu(subTitleName,subTitleText));    
    }

    /**
     * This method creates all the submenu of a main menu.
     * @see Help.MyMenu
     * @param position The array containing the all the different position of submenu in a main menu.
     * @param lastValue The last position of the subtitles in a file.
     * @return An array of sub menu.
     */
    protected MyMenu[] createSubTitlesMenu(Long[] position, long lastValue) {
	Vector subTitleVector = new Vector();
	/**
	 * We create all the sub menu.
	 */
	for (int i=1; i<position.length-1; i++) {
	    subTitleVector.add(this.createSubTitleMenu(position[i].longValue(),position[i+1].longValue()));
	}
	subTitleVector.add(this.createSubTitleMenu(position[position.length-1].longValue(),lastValue));	
	
	MyMenu[] mySubTitles = new MyMenu[subTitleVector.size()];
	subTitleVector.toArray(mySubTitles);
	return mySubTitles;
    }

    /**
     * This method creates the main menu structure.
     * @see Help.MyMenu
     * @param startPosition The start position of the file pointer.
     * @param stopPosition The stop position of the file pointer.
     * @param mySubMenu The structure of submenu of the main menu.
     * @return The main menu structure.
     */
    protected MyMenu createTitleMenu(long startPosition, long stopPosition, MyMenu[] mySubMenu) {
	String titleName="";
	String titleText="";
	try {
	    this.randomFile.seek(startPosition);
	    /**
	     * We get the title tag.
	     */
	    this.randomFile.readLine();
	    /**
	     * We get the title name.
	     */
	    titleName = this.randomFile.readLine();
	    /**
	     * We get the text.
	     */
	    while ( this.randomFile.getFilePointer() < stopPosition) {
		titleText += this.randomFile.readLine() + "\n";
	    }		    
	}
	catch (IOException e) {
	    e.printStackTrace();
	}       
	return (new MyMenu(titleName,titleText,mySubMenu));
    }    
	
    /**
     * This method creates the main menu structure with no sub menu.
     * @see Help.MyMenu
     * @param startPosition The start position of the file pointer.
     * @param stopPosition The stop position of the file pointer.
     * @return The main menu structure.
     */
    protected MyMenu createTitleMenu(long startPosition, long stopPosition) {
	String titleName="";
	String titleText="";
	try {
	    this.randomFile.seek(startPosition);
	    /**
	     * We get the title tag.
	     */
	    this.randomFile.readLine();
	    /**
	     * We get the title name.
	     */
	    titleName = this.randomFile.readLine();
	    /**
	     * We get the title text.
	     */
	    while ( this.randomFile.getFilePointer() < stopPosition) {
		titleText += this.randomFile.readLine() + "\n";
	    }		    
	}
	catch (IOException e) {
	    e.printStackTrace();
	}       
	return (new MyMenu(titleName,titleText));
    }    

    /**
     * This method creates all the menu from a file.
     * @see Help.MyMenu
     * @return An array of main menus.
     */
    protected MyMenu[] createAllMenus() {
		
	Vector menuVector =  this.findAllParagraphs();
	MyMenu[] myMenu = new MyMenu[menuVector.size()];
	try {
	    /**
	     * We get all the main menus.
	     */
	    for (int i=0; i<menuVector.size()-1; i++) {			
		if ( ((Long[])menuVector.elementAt(i)).length > 1) {
		    MyMenu[] subMenu = this.createSubTitlesMenu((Long[])menuVector.elementAt(i),((Long[])menuVector.elementAt(i+1))[0].longValue());
		    myMenu[i] = this.createTitleMenu(((Long[])menuVector.elementAt(i))[0].longValue(),((Long[])menuVector.elementAt(i))[1].longValue(),subMenu);
		}
		/**
		 * If the main menu has no sub menu, we create a menu without sub menu.
		 */
		else {
		    myMenu[i] = this.createTitleMenu(((Long[])menuVector.elementAt(i))[0].longValue(),((Long[])menuVector.elementAt(i+1))[0].longValue());	
		}
	    }
	    /**
	     * There is a special case when we arrives at the last menu.
	     */
	    if ( ((Long[])menuVector.elementAt(menuVector.size()-1)).length > 1) {
		MyMenu[] subMenu = this.createSubTitlesMenu((Long[])menuVector.elementAt(menuVector.size()-1),this.randomFile.length());		
		myMenu[myMenu.length-1] = this.createTitleMenu(((Long[])menuVector.elementAt(menuVector.size()-1))[0].longValue(),((Long[])menuVector.elementAt(menuVector.size()-1))[1].longValue(),subMenu);
	    }
	    else {
		myMenu[myMenu.length-1] = this.createTitleMenu(((Long[])menuVector.elementAt(menuVector.size()-1))[0].longValue(),((Long[])menuVector.elementAt(menuVector.size()-1))[1].longValue());
	    }
	}
	catch (IOException e) {
	    e.printStackTrace();
	}

	return myMenu;
    }
	
    /**
     * This method write an array of menu in a file.
     * @see Help.MyMenu
     * @param fileName The name of the file we want to write the structure into.
     * @param menu The structure containing all the menu.
     */
    protected void writeStructure(String fileName, MyMenu[] menu) {
	try {
	    /**
	     * We write the structure in the file.
	     */
	    File newFile = new File(fileName);
	    FileOutputStream fileOStream = new FileOutputStream(newFile);				
	    ObjectOutputStream stream = new ObjectOutputStream(fileOStream);
			
	    stream.writeObject(menu);	
	    stream.flush();
	    fileOStream.close();
	}
	catch (FileNotFoundException e) {
	    e.printStackTrace();
	}
	catch(IOException e) {
	    e.printStackTrace();
	}				
    }
	
    /**
     * This method read an array of menu from a file.
     * @see Help.MyMenu
     * @param fileName The name of the file we want to read the structure.
     * @return The array of menu. Null if the array has not been found.
     */
    protected MyMenu[] readStructure(String fileName) {
	MyMenu[] menu = null;
	try {
	    /**
	     * We read the file.
	     */
	    FileInputStream fileIStream = new FileInputStream(fileName);
	    ObjectInputStream stream = new ObjectInputStream(fileIStream);
		
	    menu = (MyMenu[])stream.readObject();		
	
	    fileIStream.close();
	}
	catch (FileNotFoundException e) {
	    e.printStackTrace();
	}
	catch(IOException e) {
	    e.printStackTrace();
	}	
	catch (ClassNotFoundException e) {
	    e.printStackTrace();
	}
	return menu; 
    }


    /**
     * The main method that read the file, creates all the menu, write them in a file,
     * read the written file (to be sure that the operation is succesful) 
     * and put in the console some parameters of the file.
     * @param args The argument of the command line.
     */
    public static void main(String[] args) {
		
	/**
	 * We create all the menus.
	 */
	if (args.length != 1) {
	    System.out.println("Usage: java Help.HelpFileParser filename");
	    System.exit(-1);
	}
	System.out.println("Trying to parse the file");
	HelpFileParser help = new HelpFileParser(args[args.length-1]);
	MyMenu[] menu = help.createAllMenus();		
	System.out.println("Parsing succeeded");
	
	/**
	 * We write the menus in a file.
	 */
	System.out.println("\nTrying to write the output file");
	help.writeStructure(args[args.length-1]+".str",menu);
	System.out.println("Write in "+args[args.length-1]+".str"+" succeeded");
	
	/**
	 * We read the file.
	 */
	System.out.println("\nTrying to read the output file");
	MyMenu[] newMenu = help.readStructure(args[args.length-1]+".str");
	System.out.println("Read succeeded");	
	
	/**
	 * We display the parameters of the file.
	 */
	System.out.println("\n***  Caracteristics  ***");
	System.out.println("Number of menu: "+newMenu.length);

	for (int i=0; i<newMenu.length; i++) {
	    if (newMenu[i].hasSubMenu()) {
		System.out.println("Menu "+i+" has "+(newMenu[i].subMenu).length+" submenu(s)");
	    }
	    else {
		System.out.println("Menu "+i+" has no submenu");
	    }	    
	}
	
	System.out.println("***  End  ***");

	
    } 

}
    
    
