/**
 * This package provides some tools to create and manipulate a panel.
 * @author Sebastien baehni
 * @version 1.0
 */
package PanelConfiguration;

import java.awt.Button;
import java.awt.Color;
import java.awt.Choice;
import java.awt.Checkbox;
import java.awt.Label;
import java.awt.event.*;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.Panel;

import GrayColorChooser.*;

/**
 * This class create a panel that can be used to draw forms.
 * @author Sebastien baehni
 * @see PanelConfiguration.FormsMakerPanel
 * @version 1.0
 */
public class FormsMakerPanel extends PanelConfiguration implements MouseListener {
       
    
    //The different labels buttons and choice.     
    /**
     * The forms label.
     * @serial
     */
    protected Label formsLabel;
    /**
     * The texture label.
     * @serial
     */
    protected Label textureLabel;
    /**
     * The color button.
     * @serial
     */
    protected Button colorButton;  
    /**
     * The shape choice.
     * @serial
     */
    protected Choice formsChoice;
    /**
     * The texture choice.
     * @serial
     */
    protected Choice textureChoice;

    /**
     * The checkbox for the negative image.
     * @serial
     */
    protected Checkbox negativeCheckbox;

    /**
     * The panel that contains the negative checkbox and the color button.
     * @serial
     */
    protected Panel eastPanel;
    /**
     * The panel that contains the shapes choice and the texture choice.
     * @serial
     */
    protected Panel westPanel;
    /**
     * The panel that contains the negative checkbox.
     * @serial
     */
    protected Panel northEastPanel;
    /**
     * The panel that contains the color button.
     * @serial
     */
    protected Panel southEastPanel;
    /**
     * The panel that contains the shapes choice.
     * @serial
     */
    protected Panel northWestPanel;
    /**
     * The panel that contains the texture choice.
     * @serial
     */
    protected Panel southWestPanel;
    
    
    //The mouse listener for the upper components.    	
    /**
     * The mouse listener for the shape choice.
     * @serial
     */
    protected MouseListener formsMouseListener;
    /**
     * The mouse listener for the texture choice.
     * @serial
     */
    protected MouseListener textureMouseListener;
    /**
     * The mouse listener for the color button.
     * @serial
     */
    protected MouseListener colorMouseListener;   
    /**
     * The mouse listener for the negative checkbox.
     * @serial
     */
    protected MouseListener negativeMouseListener;
    
    /**
     * This is the panel on which we display the help message.
     * @see PanelConfiguration.ToolPanel
     * @serial
     */
    protected ToolPanel toolPanel;
    /**
     * The gray color chooser.
     * @see GrayColorChooser.GrayColorChooser
     * @serial
     */
    protected GrayColorChooser grayColorChooser;
    
    /**
     * If the user has choosen a texture or a color.
     * @serial
     */
    protected boolean textureChoosen = true;	
    /**
     * If the user is simulating.
     * @serial
     */
    protected boolean formsMakerEnable = true;
    /**
     * If the user is drawing a form.
     * @serial
     */
    protected boolean formsMakerPaintingDisable = true;
    /**
     * The color the user has choosen.
     * @serial
     */
    protected int lastColorClicked = 128;
    
    /**
     * The constructor.
     * @param backgroundColor The background color of the panel.
     */
    public FormsMakerPanel(Color backgroundColor) {
	/**
	 * Creation of the panel and all the labels.
	 */
	super(backgroundColor);
	this.formsLabel = new Label("Shape");
	this.textureLabel = new Label("Texture");
	this.colorButton = new Button("Color");
	this.formsChoice = new Choice();
	this.textureChoice = new Choice();
	this.negativeCheckbox = new Checkbox("Negative Image",false);
	this.grayColorChooser = new GrayColorChooser();

	this.eastPanel = new Panel(new GridLayout(2,2));
	this.westPanel = new Panel(new GridLayout(2,2));
	this.northEastPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.northWestPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.southEastPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.southWestPanel = new Panel(new FlowLayout(FlowLayout.CENTER));

	this.initMouseListener();
	
	this.setLayout(new FlowLayout(FlowLayout.CENTER));
    }

    /**
     * The second constructor. The background color is white.
     */
    public FormsMakerPanel() {
	this(Color.white);
    } 
    
    /**
     * Initialisation of the mouse listener.
     */
    protected void initMouseListener() {			
	this.formsMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("Choose your shape.");
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.textureMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("Choose your texture.\nThis will invalidate any choice of color.");
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) { 			
		textureChoosen(true);
		grayColorChooser().clicked(false);
	    }
	    public void mouseReleased(MouseEvent e) {}
	};
	this.colorMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {	
		if (colorButtonEnabled()) {
		    textureChoosen(false);
		    if (!grayColorChooser().colorFrameOpen()) {
			grayColorChooser().start();
		    }	
		    else {
			grayColorChooser().toFront();
		    }
		}
	    }
	    public void mouseEntered(MouseEvent e) {
		setText("Use this button to choose a color.\nThis will invalidate any choice of texture.");
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.negativeMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("You can specify if you want to see the real image or the negative one.");
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
    }
    
    /**
     * Initialisation of the panel. We add the different button, choice and label component.
     * We add also the the mouse listener to the components.
     * @see PanelConfiguration.ToolPanel
     * @param toolPanel The panel on which we display the help.
     */
    public void init(ToolPanel toolPanel) {
		
	this.colorButton.addMouseListener(this.colorMouseListener);
	this.formsChoice.addMouseListener(this.formsMouseListener);
	this.textureChoice.addMouseListener(this.textureMouseListener);
	this.negativeCheckbox.addMouseListener(this.negativeMouseListener);
	this.addMouseListener(this);
	
	this.formsLabel.addMouseListener(this);
	this.textureLabel.addMouseListener(this);
	this.eastPanel.addMouseListener(this);
	this.westPanel.addMouseListener(this);
	this.northEastPanel.addMouseListener(this);
	this.northWestPanel.addMouseListener(this);
	this.southEastPanel.addMouseListener(this);
	this.southWestPanel.addMouseListener(this);

	this.toolPanel = toolPanel;
	
	this.formsChoice.add("Rectangle");
	this.formsChoice.add("Polygon");
	this.formsChoice.add("Circle");
	
	this.textureChoice.add("None");
	this.textureChoice.add("Upper left - Lower right");
	this.textureChoice.add("Upper right - Lower left");
	this.textureChoice.add("Lower left - Upper right");
	this.textureChoice.add("Lower right - Upper left");
	this.textureChoice.add("Left - Right");
	this.textureChoice.add("Right - Left");
	this.textureChoice.add("Up - Down");
	this.textureChoice.add("Down - Up");
	this.textureChoice.add("Random");
	this.textureChoice.add("Sinus");
	
	this.add(this.westPanel);
	this.add(this.eastPanel);

	this.westPanel.add(this.northWestPanel);
	this.westPanel.add(this.southWestPanel);
	this.eastPanel.add(this.northEastPanel);
	this.eastPanel.add(this.southEastPanel);

	this.northWestPanel.add(this.formsLabel);
	this.northWestPanel.add(this.formsChoice);
	this.northEastPanel.add(this.negativeCheckbox);	

	this.southWestPanel.add(this.textureLabel);
	this.southWestPanel.add(this.textureChoice);
	this.southEastPanel.add(this.colorButton);
    }
    
    /**
     * This method enable the panel and all his component.
     */
    public void enableFormsMaker() {
	this.colorButton.setEnabled(true);
	this.formsChoice.setEnabled(true);
	this.textureChoice.setEnabled(true);	
	this.Enable();
	this.formsMakerEnable = true;	
	this.formsMakerPaintingDisable = true;
    }
    
    /**
     * This method disable the panel and all his component.
     */
    public void disableFormsMaker() {
	this.colorButton.setEnabled(false);
	this.formsChoice.setEnabled(false);
	this.textureChoice.setEnabled(false);
	this.Disable();
	this.formsMakerEnable = false;	
    }
    
    /**
     * This method disable the panel and all the component but only when the user is painting forms.
     */
    public void disableFormsMakerPainting() {
	this.colorButton.setEnabled(false);
	this.formsChoice.setEnabled(false);
	this.textureChoice.setEnabled(false);	
	this.Disable();
	this.formsMakerPaintingDisable = false;
    }
    
    /**
     * This method return if the panel is enable or not.
     * @return True if the panel is enable.
     */
    public boolean formsMakerEnable() {
	return this.formsMakerEnable;
    }
    
    /**
     * This method return if the panel is enable or not (used when the user is painting).
     * @return True if the user is painting a form. False otherwise.
     */
    public boolean formsMakerPaintingDisable() {
	return this.formsMakerPaintingDisable;
    }
    
    /**
     * Return if the user has choosen a texture or a color.
     * @return True if the user has choosen a texture, false if he has choosen a color.
     */
    public boolean textureChoosen() {
	if (this.grayColorChooser().clicked()) {			
	    return false;			
	}
	else {
	    return this.textureChoosen;
	}
    }
    
    /**
     * Return the texture mode.
     * @return The texture mode.
     */
    public int textureChoice() {
	return (this.textureChoice.getSelectedIndex()+1);	
    }
    
    /**
     * Return the forms that the user has choosen.
     * @return 0 for the rectangle, 1 for the circle and 2 for the polygon.
     */
    public int formsChoice() {
	return this.formsChoice.getSelectedIndex();
    }
    
    /**
     * Return the rgb value of the color that the user has choosen.
     * @return The rgb value of the gray color the user has choosen.
     */
    public int colorChoosen() {
	return this.grayColorChooser().colorClicked();
    }

    /**
     * Return the gray color chooser. Very useful to acceed the gray color chooser
     * in the internal mouse listener.
     * @see GrayColorChooser.GrayColorChooser
     * @return The GrayColorChooser.
     */
    public GrayColorChooser grayColorChooser() {
	return this.grayColorChooser;
    } 
    
    /**
     * Assign the textureChoosen boolean to a new value.
     * @param choosen The new value we want to set the boolean textureChoosen
     */
    public void textureChoosen(boolean choosen) {
	this.textureChoosen = choosen;
    }
    
    /**
     * This method return if the color button is enabled or not. Very useful for the 
     * mouselistener.
     * @return True is the button is enabled, false otherwise.
     */
    public boolean colorButtonEnabled() {
	return this.colorButton.isEnabled();
    }

    /**
     * Return the state of the checkbox.
     * @return True is the checkbox is checked, false otherwise.
     */
    public boolean negativeChoosen() {
	return this.negativeCheckbox.getState();
    }

    /**
     * This method set the help message.
     * @param string The help message.
     */
    public void setText(String string) {
	this.toolPanel.setText(string);
    }
	
    /**
     * When the mouse is clicked.
     * @param e The MouseEvent.
     */
    public void mouseClicked(MouseEvent e) {}
    /**
     * When the mouse enters.
     * @param e The MouseEvent.
     */
    public void mouseEntered(MouseEvent e) {
  	this.setText("Here you can specify all the parameters for drawing the shapes.");
    }
    /**
     * When the mouse exits.
     * @param e The MouseEvent.
     */
    public void mouseExited(MouseEvent e) {}
    /**
     * When the mouse is pressed.
     * @param e The MouseEvent.
     */
    public void mousePressed(MouseEvent e) {}
    /**
     * When the mouse is released.
     * @param e The MouseEvent.
     */
    public void mouseReleased(MouseEvent e) {}
   
}
