/**
 * This package provides some tools to create and manipulate a panel.
 * @author Sebastien baehni
 * @version 1.0
 */
package PanelConfiguration;

import java.awt.Button;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.Image;
import java.awt.Label;
import java.awt.Panel;
import java.awt.TextField;
import java.awt.event.*;

/**
 * This class provides a panel in order to add on it some useful textfield for the retina algorithme simulation.
 * By exemple, TextFields for the cone layer, horizontal layer...
 * @see PanelConfiguration.PanelConfiguration
 * @author Sebastien baehni
 * @version 1.0
 */ 
public class RetineConfigurationPanel extends PanelConfiguration implements MouseListener {
   
    
    //The different panel we are going to use to have their variables.    
    /**
     * The panel in which we are going to display help messages.
     * @see PanelConfiguration.ToolPanel
     * @serial
     */
    protected ToolPanel toolPanel;
    /**
     * The panel for drawing forms.
     * @see PanelConfiguration.FormsMakerPanel
     * @serial
     */
    protected FormsMakerPanel formsMakerPanel;
    /**
     * The panel that contains the result of the algorithm.
     * @see PanelConfiguration.ModifiedPaintSurfacePanel
     * @serial
     */
    protected ModifiedPaintSurfacePanel modifiedPaintSurfacePanel;

    /* The different Labels, TextFields, Button in order to enter the parameters
       and to start the retina algorithme. */     
    /**
     * The label for the cone layer.
     * @serial
     */
    protected Label coneLayerLabel;
    /**
     * The label for the horizontal layer.
     * @serial
     */
    protected Label horizontalLayerLabel;
    /**
     * The label for the cone synaptic resistance.
     * @serial
     */
    protected Label coneSynapticResistanceLabel;
    /**
     * The label for the cone membran resistance.
     * @serial
     */
    protected Label coneMembranarResistanceLabel;
    /**
     * The label for the cone jonction resistance.
     * @serial
     */
    protected Label coneJonctionResistanceLabel;
    /**
     * The label for the cone membran capacity.
     * @serial
     */
    protected Label coneMembranarCapacityLabel;
    /**
     * The label for the horizontal synaptic resistance.
     * @serial
     */
    protected Label horizontalSynapticResistanceLabel;
    /**
     * The label for the horizontal membran resistance.
     * @serial
     */
    protected Label horizontalMembranarResistanceLabel;
    /**
     * The label for the horizontal jonction resistance.
     * @serial
     */
    protected Label horizontalJonctionResistanceLabel;
    /**
     * The label for the horizontal membran capacity.
     * @serial
     */
    protected Label horizontalMembranarCapacityLabel;
    /**
     * The label for the sleep textfield.
     * @serial
     */
    protected Label numberOfMillisToSleepLabel;
    /**
     * The labels for the unit.
     * @serial
     */
    protected Label[] unitLabel;
    
    /**
     * The textfield for the cone synaptic resistance.
     * @serial
     */
    protected TextField coneSynapticResistanceTextField;
    /**
     * The textfield for the cone membran resistance.
     * @serial
     */
    protected TextField coneMembranarResistanceTextField;
    /**
     * The textfield for the cone jonction resistance.
     * @serial
     */
    protected TextField coneJonctionResistanceTextField;
    /**
     * The textfield for the cone membran capacity.
     * @serial
     */
    protected TextField coneMembranarCapacityTextField;
    /**
     * The textfield for the horizontal synaptic resistance.
     * @serial
     */
    protected TextField horizontalSynapticResistanceTextField;
    /**
     * The textfield for the horizontal membran resistance.
     * @serial
     */
    protected TextField horizontalMembranarResistanceTextField;
    /**
     * The textfield for the horizontal jonction resistance.
     * @serial
     */
    protected TextField horizontalJonctionResistanceTextField;
    /**
     * The textfield for the horizontal membran capacity.
     * @serial
     */
    protected TextField horizontalMembranarCapacityTextField;
    /**
     * The textfield for sleeping in the algorithm.
     * @serial
     */
    protected TextField numberOfMillisToSleepTextField;

    /**
     * The start button.
     * @serial
     */
    protected Button startButton;
    /**
     * The stop button.
     * @serial
     */
    protected Button stopButton;    
    /**
     * The reset button.
     * @serial
     */
    protected Button resetButton;

    /**
     * The value for the cone synaptic resistance.
     * @serial
     */
    protected float coneSynapticResistanceFloat;
    /**
     * The value for the cone membran resistance.
     * @serial
     */
    protected float coneMembranarResistanceFloat;
    /**
     * The value for the cone jonction resistance.
     * @serial
     */
    protected float coneJonctionResistanceFloat;
    /**
     * The value for the cone membran capacity.
     * @serial
     */
    protected float coneMembranarCapacityFloat;
    /**
     * The value for the horizontal synaptic resistance.
     * @serial
     */
    protected float horizontalSynapticResistanceFloat;
    /**
     * The value for the horizontal membran resistance.
     * @serial
     */
    protected float horizontalMembranarResistanceFloat;
    /**
     * The value for the horizontal jonction resistance.
     * @serial
     */
    protected float horizontalJonctionResistanceFloat;
    /**
     * The value for the horizontal membran capacity.
     * @serial
     */
    protected float horizontalMembranarCapacityFloat;  
    /**
     * The value for the sleeping (in ms).
     * @serial
     */
    protected long numberOfMillisToSleepLong;
    
    // The mouse listeners for all the component upper.     	
    /**
     * The mouse listener for the cone synaptic resistance.
     * @serial
     */
    protected MouseListener coneSynapticResistanceMouseListener;
    /**
     * The mouse listener for the cone membran resistance.
     * @serial
     */
    protected MouseListener coneMembranarResistanceMouseListener;
    /**
     * The mouse listener for the cone jonction resistance.
     * @serial
     */
    protected MouseListener coneJonctionResistanceMouseListener;
    /**
     * The mouse listener for the cone membran capacity.
     * @serial
     */
    protected MouseListener coneMembranarCapacityMouseListener;
    /**
     * The mouse listener for the horizontal synaptic resistance.
     * @serial
     */
    protected MouseListener horizontalSynapticResistanceMouseListener;
    /**
     * The mouse listener for the horizontal membran resistance.
     * @serial
     */
    protected MouseListener horizontalMembranarResistanceMouseListener;
    /**
     * The mouse listener for the horizontal jonction resistance.
     * @serial
     */
    protected MouseListener horizontalJonctionResistanceMouseListener;
    /**
     * The mouse listener for the horizontal membran capacity.
     * @serial
     */
    protected MouseListener horizontalMembranarCapacityMouseListener;
    /**
     * The mouse listener for the sleeping (in ms).
     * @serial
     */
    protected MouseListener numberOfMillisToSleepMouseListener;
    /**
     * The mouse listener for the start button.
     * @serial
     */
    protected MouseListener startMouseListener;
    /**
     * The mouse listener for the stop button.
     * @serial
     */
    protected MouseListener stopMouseListener;
    /**
     * The mouse listener for the reset button.
     * @serial
     */
    protected MouseListener resetMouseListener;

    /**
     * Boolean very useful to know if the user has entered valid parameters.
     * @serial
     */
    protected boolean initSucceed = false;
    
    // The panels for the frame.
    /**
     * The panels for the parameters fields.
     * @serial
     */
    protected Panel[] panel;
    /**
     * The panel that contains the cone layer parameters
     * @serial
     */
    protected Panel centerGridPanel;
    /**
     * The panel that contains the horizontal layer parameters
     * @serial
     */
    protected Panel southGridPanel;
    /**
     * The north panel (see the code for more informations).
     * @serial
     */
    protected Panel northPanel;
    /**
     * The north center panel (see the code for more informations).
     * @serial
     */
    protected Panel northCenterPanel;
    /**
     * The north north panel (see the code for more informations).
     * @serial
     */
    protected Panel northNorthPanel;
    /**
     * The north east panel (see the code for more informations).
     * @serial
     */
    protected Panel northEastPanel;
    /**
     * The center panel (see the code for more informations).
     * @serial
     */
    protected Panel centerPanel;
    /**
     * The center center panel (see the code for more informations).
     * @serial
     */
    protected Panel centerCenterPanel;
    /**
     * The center north panel (see the code for more informations).
     * @serial
     */
    protected Panel centerNorthPanel;
    /**
     * The center east panel (see the code for more informations).
     * @serial
     */
    protected Panel centerEastPanel;
    /**
     * The south panel (see the code for more informations).
     * @serial
     */
    protected Panel southPanel;     
    /**
     * The south south panel (see the code for more informations).
     * @serial
     */
    protected Panel southSouthPanel;
    /**
     * The south north panel (see the code for more informations).
     * @serial
     */
    protected Panel southNorthPanel;
    
    /**
     * The constructor. It creates a new RetineConfigurationPanel with all the upper components.
     * @param backgroundcolor The background color of the panel.
     */
    public RetineConfigurationPanel(Color backgroundColor) {
	super(backgroundColor);
	this.coneLayerLabel = new Label("Cone layer");
	this.horizontalLayerLabel = new Label("Horizontal layer");
	this.coneSynapticResistanceLabel = new Label("Synaptic resistance (rc)");
	this.coneMembranarResistanceLabel = new Label("Membrane resistance (Rmc)");
	this.coneMembranarCapacityLabel = new Label("Membrane capacity (Cc)");
	this.coneJonctionResistanceLabel = new Label("Junction resistance (Rc)");
	this.horizontalSynapticResistanceLabel = new Label("Synaptic resistance (rh)");
	this.horizontalMembranarResistanceLabel = new Label("Membrane resistance (Rmh)");
	this.horizontalMembranarCapacityLabel = new Label("Membrane capacity (Ch)");
	this.horizontalJonctionResistanceLabel = new Label("Junction resistance (Rh)");
	this.numberOfMillisToSleepLabel = new Label("Waiting time between each epoch");
	
	
	this.coneSynapticResistanceTextField = new TextField("0.01",15);
	this.coneMembranarResistanceTextField = new TextField("1",15);
	this.coneJonctionResistanceTextField = new TextField("0.6",15);
	this.coneMembranarCapacityTextField = new TextField("1",15);
	this.horizontalSynapticResistanceTextField = new TextField("0.08",15);
	this.horizontalMembranarResistanceTextField = new TextField("1",15);
	this.horizontalJonctionResistanceTextField = new TextField("0.06",15);
	this.horizontalMembranarCapacityTextField = new TextField("1",15);	
	this.numberOfMillisToSleepTextField = new TextField("100",15);

	this.startButton = new Button("Start");
	this.stopButton = new Button("Stop");	
	this.resetButton = new Button("Reset Fields");

	this.initLabel(9);
	this.initMouseListener();
	this.setLayout(new BorderLayout());
	this.northPanel = new Panel(new BorderLayout());
	this.centerPanel = new Panel(new BorderLayout());
	this.northCenterPanel = new Panel(new GridLayout(4,2));
	this.northNorthPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.northEastPanel = new Panel(new GridLayout(4,1));
	this.centerCenterPanel = new Panel(new GridLayout(4,2));
	this.centerNorthPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.centerEastPanel = new Panel(new GridLayout(4,1));
	this.southPanel = new Panel(new GridLayout(2,1));
	this.southSouthPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.southNorthPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	this.centerGridPanel = new Panel(new GridLayout(2,1));
	this.southGridPanel = new Panel(new FlowLayout(FlowLayout.CENTER));
	
    }	
    
    /**
     * The constructor. It creates a new RetineConfigurationPanel with all the upper components.
     * Its background color is white.
     */
    public RetineConfigurationPanel() {
	this(Color.white);
    }
    
    /**
     * Initialization of the RetineConfigurationPanel.
     * @see PanelConfiguration.ToolPanel
     * @see PanelConfiguration.FormsMakerPanel
     * @see PanelConfiguration.ModifiedPaintSurfacePanel
     * @param toolPanel                 The panel on which we display the help message.
     * @param formsMakerPanel           The panel we are going to get if the user is drawing a form or not.
     * @param modifiedPaintSurfacePanel The panel we are going to tell if the simulation could start or not.
     */
    public void init(ToolPanel toolPanel, FormsMakerPanel formsMakerPanel, ModifiedPaintSurfacePanel modifiedPaintSurfacePanel) {
	this.coneSynapticResistanceTextField.addMouseListener(this.coneSynapticResistanceMouseListener);
	this.coneMembranarResistanceTextField.addMouseListener(this.coneMembranarResistanceMouseListener);
	this.coneMembranarCapacityTextField.addMouseListener(this.coneMembranarCapacityMouseListener);
	this.coneJonctionResistanceTextField.addMouseListener(this.coneJonctionResistanceMouseListener);
	this.horizontalSynapticResistanceTextField.addMouseListener(this.horizontalSynapticResistanceMouseListener);
	this.horizontalMembranarResistanceTextField.addMouseListener(this.horizontalMembranarResistanceMouseListener);
	this.horizontalMembranarCapacityTextField.addMouseListener(this.horizontalMembranarCapacityMouseListener);
	this.horizontalJonctionResistanceTextField.addMouseListener(this.horizontalJonctionResistanceMouseListener);	
	this.numberOfMillisToSleepTextField.addMouseListener(this.numberOfMillisToSleepMouseListener);

	this.coneSynapticResistanceLabel.setForeground(Color.lightGray);
	this.coneMembranarResistanceLabel.setForeground(Color.cyan);
	this.coneMembranarCapacityLabel.setForeground(new Color(255,102,51));
	this.coneJonctionResistanceLabel.setForeground(Color.red);
	this.horizontalSynapticResistanceLabel.setForeground(Color.magenta);
	this.horizontalMembranarResistanceLabel.setForeground(Color.blue);
	this.horizontalMembranarCapacityLabel.setForeground(Color.black);
	this.horizontalJonctionResistanceLabel.setForeground(Color.green);	

	this.startButton.addMouseListener(this.startMouseListener);
	this.stopButton.addMouseListener(this.stopMouseListener);
	this.resetButton.addMouseListener(this.resetMouseListener);
		
	this.addMouseListener(this);	
	
	this.coneLayerLabel.addMouseListener(this);
	this.horizontalLayerLabel.addMouseListener(this);
	this.coneSynapticResistanceLabel.addMouseListener(this);
	this.coneMembranarResistanceLabel.addMouseListener(this);
	this.coneJonctionResistanceLabel.addMouseListener(this);
	this.coneMembranarCapacityLabel.addMouseListener(this);
	this.horizontalSynapticResistanceLabel.addMouseListener(this);
	this.horizontalMembranarResistanceLabel.addMouseListener(this);
	this.horizontalJonctionResistanceLabel.addMouseListener(this);
	this.horizontalMembranarCapacityLabel.addMouseListener(this);
	this.numberOfMillisToSleepLabel.addMouseListener(this);
	
	this.toolPanel = toolPanel;
	this.formsMakerPanel = formsMakerPanel;
	this.modifiedPaintSurfacePanel = modifiedPaintSurfacePanel;	

	this.coneLayerLabel.setFont(new Font("Dialog",Font.ITALIC,15));
	this.horizontalLayerLabel.setFont(new Font("Dialog",Font.ITALIC,15));

	for (int i=0; i<unitLabel.length; i++) {
	    this.unitLabel[i].addMouseListener(this);
	}

	this.add(this.centerGridPanel,BorderLayout.CENTER);
	this.add(this.southGridPanel,BorderLayout.SOUTH);

	this.centerGridPanel.add(this.northPanel);
	this.centerGridPanel.add(this.centerPanel);
	this.southGridPanel.add(this.southPanel);

	this.northPanel.add(this.northNorthPanel,BorderLayout.NORTH);
	this.northNorthPanel.add(this.coneLayerLabel);
	this.northPanel.add(this.northCenterPanel,BorderLayout.CENTER);
	this.northPanel.add(this.northEastPanel,BorderLayout.EAST);

	this.centerPanel.add(this.centerNorthPanel,BorderLayout.NORTH);
	this.centerNorthPanel.add(this.horizontalLayerLabel);
	this.centerPanel.add(this.centerCenterPanel,BorderLayout.CENTER);
	this.centerPanel.add(this.centerEastPanel,BorderLayout.EAST);
	this.centerPanel.add(this.southSouthPanel,BorderLayout.SOUTH);

	this.northCenterPanel.add(this.coneSynapticResistanceLabel);
	this.northCenterPanel.add(this.coneSynapticResistanceTextField);
	this.northCenterPanel.add(this.coneMembranarResistanceLabel);
	this.northCenterPanel.add(this.coneMembranarResistanceTextField);
	this.northCenterPanel.add(this.coneMembranarCapacityLabel);
	this.northCenterPanel.add(this.coneMembranarCapacityTextField);
	this.northCenterPanel.add(this.coneJonctionResistanceLabel);
	this.northCenterPanel.add(this.coneJonctionResistanceTextField);

	this.centerCenterPanel.add(this.horizontalSynapticResistanceLabel);
	this.centerCenterPanel.add(this.horizontalSynapticResistanceTextField);
	this.centerCenterPanel.add(this.horizontalMembranarResistanceLabel);
	this.centerCenterPanel.add(this.horizontalMembranarResistanceTextField);
	this.centerCenterPanel.add(this.horizontalMembranarCapacityLabel);
	this.centerCenterPanel.add(this.horizontalMembranarCapacityTextField);
	this.centerCenterPanel.add(this.horizontalJonctionResistanceLabel);
	this.centerCenterPanel.add(this.horizontalJonctionResistanceTextField);
	for (int i=0; i<this.unitLabel.length; i++) {
	    
	    if (i<(this.unitLabel.length/2)) {
		this.northEastPanel.add(this.unitLabel[i]);
	    }
	    else {
		this.centerEastPanel.add(this.unitLabel[i]);
	    }
	}       

	this.southPanel.add(this.southNorthPanel);
	this.southPanel.add(this.southSouthPanel);

	this.southNorthPanel.add(this.numberOfMillisToSleepLabel);
	this.southNorthPanel.add(this.numberOfMillisToSleepTextField);
	this.southNorthPanel.add(this.unitLabel[this.unitLabel.length-1]);

	this.southSouthPanel.add(this.startButton);
	this.southSouthPanel.add(this.stopButton);
	this.southSouthPanel.add(this.resetButton);

    }    

    /**
     * This method create a number of label with a specified layout.
     * @param number The number of panel you want ot create.
     */
    protected void initLabel(int number) {
	unitLabel = new Label[number];
	for (int i=0; i<unitLabel.length; i++) {
	    if (i==2 || i ==6) {
		unitLabel[i] = new Label("NFarad");
	    }
	    else if (i==unitLabel.length-1) {
		unitLabel[i] = new Label("ms");
	    }
	    else {
		unitLabel[i] = new Label("MOhm");
	    }
	}	
    }
    

    /** 
     * Initialization of all the mouse listeners of the different components.
     */
    protected void initMouseListener() {		
	this.coneSynapticResistanceMouseListener= new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("Enter a number for the synaptic resistance of the cone layer.\nFor better results don't put a value below 0.007 with the others default values.");	
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.coneMembranarResistanceMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("Enter a number for the membrane resistance of the cone layer.\nFor better results don't put a value below 0.015 with the others default values.");	
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.coneMembranarCapacityMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("Enter a number for the membrane capacity  of the cone layer.\nFor better results don't put a value below 0.6 with the others default values.");	
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.coneJonctionResistanceMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("Enter a number for the junction resistance of the cone layer.\nFor better results don't put a value below 0.08 with the others default values.");	
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.horizontalSynapticResistanceMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("Enter a number for the synaptic resistance of the horizontal layer.\nFor better results don't put a value below 0.015 with the others default values.");	
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.horizontalMembranarResistanceMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) { }
	    public void mouseEntered(MouseEvent e) {
		setText("Enter a number for the membrane resistance of the horizontal layer.\nFor better results don't put a value below 0.015 with the others default values.");	
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.horizontalMembranarCapacityMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("Enter a number for the membrane capacity  of the horizontal layer.\nFor better results don't put a value below 0.7 with the others default values.");	
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.horizontalJonctionResistanceMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("Enter a number for the junction resistance of the horizontal layer.\nFor better results don't put a value below 0.04 with the others default values.");			
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.startMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {	
		if (formsMakerEnable()) {
		    if (!formsMakerPaintingDisable()) {
			setText("You must finish creating your shape before beginning the simulation.",Color.red);
		    }
		    else {
			initSimulation();
			if (initSucceed) {
			    startSimulation();
			}
		    }	
		}
		else {
		    setText("The simulation is already running. Stop it and then you can restart it.",Color.red);
		}
	    }
	    public void mouseEntered(MouseEvent e) {
		setText("Click on this button to start the simulation when you have finished creating shapes.");
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.stopMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {
		if (formsMakerPaintingDisable()) {
		    stopSimulation();
		}	
	    }
	    public void mouseEntered(MouseEvent e) {
		setText("Click on this button to stop the simulation.");
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.resetMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {
		if (formsMakerEnable()) {
		    resetAllTextFields();
		}
		else {
		    setText("You can't reset the fields when the simulation is running.",Color.red);
		}
	    }
	    public void mouseEntered(MouseEvent e) {
		setText("Click on this button to reset all the fields of the simulation.");
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
	this.numberOfMillisToSleepMouseListener = new MouseListener() {
	    public void mouseClicked(MouseEvent e) {}
	    public void mouseEntered(MouseEvent e) {
		setText("Enter the time you want to sleep in the algorithm (in milli-seconds).\nThis is helpful to see what the algorithme does step by step if you have a very fast machine.");
	    }
	    public void mouseExited(MouseEvent e) {}
	    public void mousePressed(MouseEvent e) {}
	    public void mouseReleased(MouseEvent e) {}
	};
    }	
	
    /**
     * This method is called when the user clicked on the start button.
     * We take all the parameters defined by the user and we give them to algorithme if the their are valids.
     * If the parameters are not valid we display a message in a the help panel.
     */
    public void initSimulation() {
	try {
	    this.coneSynapticResistanceFloat = new Float(coneSynapticResistanceTextField.getText()).floatValue();
	    this.coneMembranarResistanceFloat = new Float(coneMembranarResistanceTextField.getText()).floatValue();
	    this.coneMembranarCapacityFloat = new Float(coneMembranarCapacityTextField.getText()).floatValue();
	    this.coneJonctionResistanceFloat = new Float(coneJonctionResistanceTextField.getText()).floatValue();	    
	    this.horizontalSynapticResistanceFloat = new Float(horizontalSynapticResistanceTextField.getText()).floatValue();
	    this.horizontalMembranarResistanceFloat = new Float(horizontalMembranarResistanceTextField.getText()).floatValue();
	    this.horizontalMembranarCapacityFloat = new Float(horizontalMembranarCapacityTextField.getText()).floatValue();		
	    this.horizontalJonctionResistanceFloat = new Float(horizontalJonctionResistanceTextField.getText()).floatValue();	
	    this.numberOfMillisToSleepLong = new Long(this.numberOfMillisToSleepTextField.getText()).longValue();
	    if (this.coneSynapticResistanceFloat < 0.0f) {
		this.initSucceed = false;		
		this.setText("The synaptic resistance of the cone layer is negative.\nThe algorithm won't work with this value. Please change it.",Color.red);
	    }
	    else if (this.coneMembranarResistanceFloat <= 0.0f) {
		this.initSucceed = false;
		this.setText("The membrane resistance of the cone layer is negative or null.\nThe algorithm won't work with this value. Please change it.",Color.red);
	    }
	    else if (this.coneMembranarCapacityFloat <= 0.0f) {
		this.initSucceed = false;
		this.setText("The membrane capacity of the cone layer is negative or null.\nThe algorithm won't work with this value. Please change it.",Color.red);
	    }
	    else if (this.coneJonctionResistanceFloat <= 0.0f) {
		this.initSucceed = false;
		this.setText("The junction resistance of the cone layer is negative or null.\nThe algorithm won't work with this value. Please change it.",Color.red);
	    }	
	    else if (this.horizontalSynapticResistanceFloat < 0.0f) {
		this.initSucceed = false;
		this.setText("The synaptic resistance of the horizontal is negative or null.\nThe algorithm won't work with this value. Please change it.",Color.red);
	    }
	    else if (this.horizontalMembranarResistanceFloat <= 0.0f) {
		this.initSucceed = false;
		this.setText("The membrane resistance of the horizontal layer is negative or null.\nThe algorithm won't work with this value. Please change it.",Color.red);
	    }
	    else if (this.horizontalMembranarCapacityFloat <= 0.0f) {
		this.initSucceed = false;
		this.setText("The membrane capacity of the horizontal layer is negative or null.\nThe algorithm won't work with this value. Please change it.",Color.red);
	    }
	    else if (this.horizontalJonctionResistanceFloat <= 0.0f) {
		this.initSucceed = false;
		this.setText("The junction resistance of the horizontal layer is negative or null.\nThe algorithm won't work with this value. Please change it.",Color.red);
	    }		
	    else if (this.numberOfMillisToSleepLong < 0) {
		this.initSucceed = false;
		this.setText("The time you want to sleep is negative !!\nYou must specify a positive number.",Color.red);
	    }
	    else {
		this.initSucceed = true;
		this.modifiedPaintSurfacePanel.initSimulation(this.coneSynapticResistanceFloat,
							      this.coneMembranarResistanceFloat,
							      this.coneMembranarCapacityFloat,
							      this.coneJonctionResistanceFloat,
							      this.horizontalSynapticResistanceFloat,
							      this.horizontalMembranarResistanceFloat,
							      this.horizontalMembranarCapacityFloat,
							      this.horizontalJonctionResistanceFloat,
							      this.numberOfMillisToSleepLong);
		this.setText("The algorithm is initialized with the specified values and has started.");
	    }
	}
	catch (NumberFormatException e) {
	    this.initSucceed = false;
	    this.setText("You must specify valid fields.\nThe fields allowed are float, double, integer and long.",Color.red);
	}
    }
	
    /**
     * We start the simulation. Called when the user clicks on the start button.
     */
    public void startSimulation() {
	this.disableWhenSimulating();
	this.modifiedPaintSurfacePanel.startSimulation();
    }

    /**
     * We stop the simulation. Called when the user clicks on the stop button.
     */	
    public void stopSimulation() {
	this.enableAfterSimulating();		
	this.modifiedPaintSurfacePanel.stopSimulation();		
    }
	
    /**
     * This method draw a help message in a text area.
     * @param string The help message.
     */
    public void setText(String string) {		
	this.toolPanel.setText(string);
    }

    /**
     * This method draw a help message in a text area.
     * @param string The help message.
     * @param color  The color of the help message.
     */	
    public void setText(String string,Color color) {		
	this.toolPanel.setText(string,color);
    }
	
    /**
     * Return if we can start the simulation or not.
     * @return True if the simulation can start. False otherwise.
     */
    public boolean initSucceed() {
	return this.initSucceed;
    }
    
    /**
     * This method returns if the FormsMakerPanel is enable or not (used when the user is painting).
     * @see PanelConfiguration.FormsMakerPanel
     * @return True if the user is painting a form. False otherwise.
     */
    public boolean formsMakerPaintingDisable() {
	return this.formsMakerPanel.formsMakerPaintingDisable();
    }

    /**
     * This method returns if the FormsMakerPanel is enable or not.
     * @see PanelConfiguration.FormsMakerPanel
     * @return True if the user is painting a form. False otherwise.
     */
    public boolean formsMakerEnable() {
	return this.formsMakerPanel.formsMakerEnable();
    }
	
    /**
     * This method disable all the panel when we are simulating the retina algorithme.
     */
    protected void disableWhenSimulating() {
	this.formsMakerPanel.disableFormsMaker();
	this.coneSynapticResistanceTextField.setEnabled(false);
	this.coneMembranarResistanceTextField.setEnabled(false);
	this.coneJonctionResistanceTextField.setEnabled(false);
	this.coneMembranarCapacityTextField.setEnabled(false);
	this.horizontalSynapticResistanceTextField.setEnabled(false);
	this.horizontalMembranarResistanceTextField.setEnabled(false);
	this.horizontalJonctionResistanceTextField.setEnabled(false);
	this.horizontalMembranarCapacityTextField.setEnabled(false);
	this.numberOfMillisToSleepTextField.setEnabled(false);		
	this.startButton.setEnabled(false);	
	this.resetButton.setEnabled(false);
    }
	
    /**
     * This method enable all the panel when the simulation stopped.
     */
    protected void enableAfterSimulating() {
	this.formsMakerPanel.enableFormsMaker();
	this.coneSynapticResistanceTextField.setEnabled(true);
	this.coneMembranarResistanceTextField.setEnabled(true);
	this.coneJonctionResistanceTextField.setEnabled(true);
	this.coneMembranarCapacityTextField.setEnabled(true);
	this.horizontalSynapticResistanceTextField.setEnabled(true);
	this.horizontalMembranarResistanceTextField.setEnabled(true);
	this.horizontalJonctionResistanceTextField.setEnabled(true);
	this.horizontalMembranarCapacityTextField.setEnabled(true);
	this.numberOfMillisToSleepTextField.setEnabled(true);		
	this.startButton.setEnabled(true);
	this.resetButton.setEnabled(true);	
    }

    /**
     * This method resets all the textfields.
     */
    protected void resetAllTextFields() {
	this.coneSynapticResistanceTextField.setText("0.01");
	this.coneMembranarResistanceTextField.setText("1");
	this.coneJonctionResistanceTextField.setText("0.6");
	this.coneMembranarCapacityTextField.setText("1");
	this.horizontalSynapticResistanceTextField.setText("0.08");
	this.horizontalMembranarResistanceTextField.setText("1");
	this.horizontalJonctionResistanceTextField.setText("0.06");
	this.horizontalMembranarCapacityTextField.setText("1");
	this.numberOfMillisToSleepTextField.setText("100");	
    }

    /**
     * When the mouse is clicked.
     * @param e The MouseEvent.
     */
    public void mouseClicked(MouseEvent e) {}
    /**
     * When the mouse enters.
     * @param e The MouseEvent.
     */
    public void mouseEntered(MouseEvent e) {
  	this.setText("Here you can specify all the parameters for the configuration of the retina algorithm.");
    }
    /**
     * When the mouse exits.
     * @param e The MouseEvent.
     */
    public void mouseExited(MouseEvent e) {}
    /**
     * When the mouse is pressed.
     * @param e The MouseEvent.
     */
    public void mousePressed(MouseEvent e) {}
    /**
     * When the mouse is released.
     * @param e The MouseEvent.
     */
    public void mouseReleased(MouseEvent e) {}
	
}
