/**
 * This package provides useful tools for creating and manipulating textures.
 * @author Sebastien Baehni
 * @version 1.0.
 */
package Texture;

import java.awt.Color;

/**
 * This class provides tools for creating textures.
 * @see Texture.TextureError
 * @author Sebastien Baehni
 * @version 1.0.
 */
public class Texture {
    
    /**
     * The array in which is the texture (rgb mode)
     */
    protected int[] arraytexture;
    /**
     * The width of the array texture.
     */
    protected int width;
    /**
     * The height of the array texture.
     */
    protected int height;
    /**
     * The mode of the texture.
     */
    protected int mode;
    /**
     * The color of the texture.
     */
    protected Color color;
    /**
     * The width increment for the texture.
     */
    protected final float widthGrayIncrement;
    /**
     * The height increment for the texture.
     */
    protected final float heightGrayIncrement;

    /**
     * The first constructor. It creates a uniform texture with a specified color.
     * @param color The color of the texture.
     */
    public Texture(int color) {	
	this.mode = color;
	this.width = 0;
	this.height = 0;
	this.widthGrayIncrement = 0;
	this.heightGrayIncrement = 0;
	this.color = new Color(-color,-color,-color);
    }
    
    /**
     * The second constructor. It creates a "true texture" in the specified mode.
     * @param width  The width of the texture.
     * @param height The height of the texture.
     * @param mode   The texture's type.
     */
    public Texture(int width, int height, int mode) {
	this.mode = mode;
	/**
	 * If the mode is equal to 1 then the user wants to create a forms without a texture.
	 */
	if (mode != 1) {
	    this.width = width;
	    this.height = height;
	    arraytexture = new int[this.width*this.height];
	
	    /**
	     * The "degrade" of the texture.
	     */
	    this.widthGrayIncrement = Math.max((255.0f/this.width),1.0f);
	    this.heightGrayIncrement = Math.max((255.0f/this.height),1.0f);       	
	
	    switch (mode) {	   
	    case 2:
		/**
		 * Texture black-white from the higher left corner to the lower right corner.
		 */
		for (int y=0; y<this.height; y++) {
		    for (int x=0; x<this.width; x++) {
			arraytexture[(y*this.width)+x] = ( 255 << 24 |
							   Math.min(255,(int)(y*this.heightGrayIncrement)/2+(int)(x*this.widthGrayIncrement)/2) << 16 |
							   Math.min(255,(int)(y*this.heightGrayIncrement)/2+(int)(x*this.widthGrayIncrement)/2) << 8  |
							   Math.min(255,(int)(y*this.heightGrayIncrement)/2+(int)(x*this.widthGrayIncrement)/2) );		    
		    }
		}		    
		break;
	    case 3:
		/**
		 * Texture black-white from the higher right corner to the lower left corner.
		 */
		for (int y=0; y<this.height; y++) {
		    for (int x=this.width-1; x>=0; x--) {
			arraytexture[(y*this.width)+x] = ( 255 << 24 |
							   Math.min(255,(int)(y*this.heightGrayIncrement)/2+(int)((this.width-x)*this.widthGrayIncrement)/2) << 16 |
							   Math.min(255,(int)(y*this.heightGrayIncrement)/2+(int)((this.width-x)*this.widthGrayIncrement)/2) << 8  |
							   Math.min(255,(int)(y*this.heightGrayIncrement)/2+(int)((this.width-x)*this.widthGrayIncrement)/2) );		    
		    }
		}		    
		break;
	    case 4:
		/**
		 * Texture black-white from the lower left corner to the higher right corner.
		 */
		for (int y=this.height-1; y>=0; y--) {
		    for (int x=0; x<this.width; x++) {
			arraytexture[(y*this.width)+x] = ( 255 << 24 |
							   Math.min(255,(int)((this.height-y)*this.heightGrayIncrement)/2+(int)(x*this.widthGrayIncrement)/2) << 16 |
							   Math.min(255,(int)((this.height-y)*this.heightGrayIncrement)/2+(int)(x*this.widthGrayIncrement)/2) << 8  |
							   Math.min(255,(int)((this.height-y)*this.heightGrayIncrement)/2+(int)(x*this.widthGrayIncrement)/2) );		    
		    }
		}		    
		break;
	    case 5:
		/**
		 * Texture black-white from the lower right corner to the higher left corner.
		 */
		for (int y=this.height-1; y>=0; y--) {
		    for (int x=this.width-1; x>=0; x--) {
			arraytexture[(y*this.width)+x] = ( 255 << 24 |
							   Math.min(255,(int)((this.height-y)*this.heightGrayIncrement)/2+(int)((this.width-x)*this.widthGrayIncrement)/2) << 16 |
							   Math.min(255,(int)((this.height-y)*this.heightGrayIncrement)/2+(int)((this.width-x)*this.widthGrayIncrement)/2) << 8  |
							   Math.min(255,(int)((this.height-y)*this.heightGrayIncrement)/2+(int)((this.width-x)*this.widthGrayIncrement)/2) );		    
		    }
		}	
		break;
	    case 6:
		/**
		 * Texture black-white from the left to the right.
		 */
		for (int y=0; y<this.height; y++) {
		    for (int x=0; x<this.width; x++) {
			arraytexture[(y*this.width)+x] = ( 255 << 24 |
							   Math.min(255,(int)(x*this.widthGrayIncrement)) << 16 |
							   Math.min(255,(int)(x*this.widthGrayIncrement)) << 8  |
							   Math.min(255,(int)(x*this.widthGrayIncrement)) );		    
		    }
		}		
		break;
	    case 7:
		/**
		 * Texture black-white from right to the left.
		 */
		for (int y=0; y<this.height; y++) {
		    for (int x=this.width-1; x>=0; x--) {
			arraytexture[(y*this.width)+x] = ( 255 << 24 |
							   Math.min(255,(int)((this.width-x)*this.widthGrayIncrement)) << 16 |
							   Math.min(255,(int)((this.width-x)*this.widthGrayIncrement)) << 8  |
							   Math.min(255,(int)((this.width-x)*this.widthGrayIncrement)) );		    
		    }
		}		

		break;
	    case 8:
		/**
		 * Texture black-white from up to down.
		 */
		for (int y=0; y<this.height; y++) {
		    for (int x=0; x<this.width; x++) {
			arraytexture[(y*this.width)+x] = ( 255 << 24 |
							   Math.min(255,(int)(y*this.heightGrayIncrement)) << 16 |
							   Math.min(255,(int)(y*this.heightGrayIncrement)) << 8  |
							   Math.min(255,(int)(y*this.heightGrayIncrement) ));		    
		    }
		}		
		break;
	    case 9:
		/**
		 * Texture black-white from down to up.
		 */
		for (int y=this.height-1; y>=0; y--) {
		    for (int x=0; x<this.width; x++) {		  
			arraytexture[(y*this.width)+x] = ( 255 << 24 |
							   Math.min(255,(int)((this.height-y)*this.heightGrayIncrement)) << 16 |
							   Math.min(255,(int)((this.height-y)*this.heightGrayIncrement)) << 8  |
							   Math.min(255,(int)((this.height-y)*this.heightGrayIncrement)) );		    
		    }
		}		
		break;
	    case 10:
		/**
		 * A random texture.
		 */
		for (int y=0; y<this.height; y++) {
		    for (int x=0; x<this.width; x++) {
			int color_value = (int)(Math.random()*255);
			arraytexture[(y*this.width)+x] = ( 255 << 24 |
							   Math.min(255,color_value) << 16 |
							   Math.min(255,color_value) << 8  |
							   Math.min(255,color_value) );		    
		    }
		}		
		break;
	    case 11:
		/**
		 * A texture with sinus.
		 */
		for (int y=0; y<this.height; y++) {
		    for (int x=0; x<this.width; x++) {
			arraytexture[(y*this.width)+x] = ( 255 << 24 |
							   Math.min(255,(int)(Math.abs(Math.sin(y))*128)+(int)(Math.abs(Math.sin(x))*128)) << 16 |
							   Math.min(255,(int)(Math.abs(Math.sin(y))*128)+(int)(Math.abs(Math.sin(x))*128)) << 8  |
							   Math.min(255,(int)(Math.abs(Math.sin(y))*128)+(int)(Math.abs(Math.sin(x))*128)) );		    
		    }
		}	    
		break;
	    default:
		break;
	    }
	}
	else {
	    /**
	     * If there is no texture, the variables of the textures are set to zero.
	     */
	    this.width = 0;
	    this.height = 0;
	    this.widthGrayIncrement = 0;
	    this.heightGrayIncrement = 0;
	}
	
    }     
    
    /**
     * This method return the width of the texture.    
     * @throws Texture.TextureError
     * @return The width of the texture or an exception if the texture is a uniform texture or if there is no texture.
     */
    public int width() throws TextureError {
	/**
	 * If there is no texture of a uniform texture than the texture has no width.
	 */
	if (this.mode <=1) {
	    throw new TextureError("The texture is a color and don't have a width");
	}
	else {
	    return this.width;
	}
    }
    
    /**
     * This method return the height of the texture.   
     * @throws Texture.TextureError
     * @return The height of the texture or an exception if the texture is a uniform texture or if there is no texture.
     */
    public int height()throws TextureError  {
	/**
	 * If there is no texture of a uniform texture than the texture has no height.
	 */
	if (this.mode <=1) {
	    throw new TextureError("The texture is a color and don't have a height");
	}
	else {
	    return this.height;
	}
    }    

    /**
     * This method return the array of the texture.  
     * @throws Texture.TextureError
     * @return The array of the texture or an exception if the texture is a uniform texture or if there is no texture.
     */
    public int[] arraytexture() throws TextureError {
	/**
	 * If there is no texture of a uniform texture than the texture has no array.
	 */
	if (this.mode <=1) {
	    throw new TextureError("The texture is a color and don't have an array");
	}
	else {
	    return this.arraytexture;
	}
    }
    
    /**
     * This method returns the mode of the texture.
     * @return The mode of the texture. If the mode is <= 0 then we have a uniform texture. 
     *                                  If the mode is == 1 then we have no texture.
     *                                  If the mode is > 1 then we have a "true" texture.
     */
    public int mode() {
	return this.mode;
    }

    /**
     * This method returns the color of the texture.
     * @throws Texture.TextureError
     * @return The color of the texture or an error if there is no texture or if the texture is a "true" texture.
     */
    public Color color() throws TextureError {
	/**
	 * If the mode if upper than zero then we have a "true" texture or we have no texture.
	 */
	if (this.mode > 0) {
	    throw new TextureError("The texture is a color.");
	}
	else {
	    return this.color;
	}
    }		

}
