/**
 * This package provides useful tools for creating and manipulating textures.
 * @author Sebastien Baehni
 * @version 1.0.
 */
package Texture;

import java.util.Vector;

/**
 * This class provides tools to administrate the textures. The textures are in a vector and we can manipulate
 * them.
 * @author Sebastien Baehni
 * @see Texture.Texture
 * @see Texture.TextureError
 * @see Texture.TextureAdministratorError
 * @version 1.0
 */
public class TextureAdministrator {
	
    /**
     * The vector for the circles.
     */
    protected Vector circleTextureVector;
    /**
     * The vector for the rectangles.
     */	
    protected Vector rectangleTextureVector;
    /**
     * The vector for the polygons.
     */
    protected Vector polygonTextureVector;
	
    /**
     * The constructor. It creates the vectors.
     */
    public TextureAdministrator() {
	this.circleTextureVector = new Vector();	
	this.rectangleTextureVector = new Vector();
	this.polygonTextureVector = new Vector();
    }
    
    /**
     * This method add a "true" texture to a circle.
     * @param width  The width of the texture.
     * @param height The height of the texture.
     * @param mode   The mode of the texture.
     */
    public void addCircleTexture(int width, int height, int mode) {
	Texture texture = new Texture(width, height, mode);	
	this.circleTextureVector.insertElementAt(texture,this.indexCircle());
    }

    /**
     * This method add a uniform texture to a circle. 
     * @param color The color of the texture.
     */
    public void addCircleTexture(int color) {
	Texture texture = new Texture(color);	
	this.circleTextureVector.insertElementAt(texture,this.indexCircle());
    }
    
    /**
     * This method add a "true" texture to a rectangle.
     * @param width  The width of the texture.
     * @param height The height of the texture.
     * @param mode   The mode of the texture.
     */
    public void addRectangleTexture(int width, int height, int mode) {
	Texture texture = new Texture(width, height, mode);	    
	this.rectangleTextureVector.insertElementAt(texture,this.indexRectangle());
    }

    /**
     * This method add a uniform texture to a rectangle. 
     * @param color The color of the texture.
     */
    public void addRectangleTexture(int color) {
	Texture texture = new Texture(color);	    
	this.rectangleTextureVector.insertElementAt(texture,this.indexRectangle());
    }
    
    /**
     * This method add a "true" texture to a polygon.
     * @param width  The width of the texture.
     * @param height The height of the texture.
     * @param mode   The mode of the texture.
     */
    public void addPolygonTexture(int width, int height, int mode) {
	Texture texture = new Texture(width, height, mode);	
	this.polygonTextureVector.insertElementAt(texture,this.indexPolygon());
    }

    /**
     * This method add a uniform texture to a polygon. 
     * @param color The color of the texture.
     */
    public void addPolygonTexture(int color) {
	Texture texture = new Texture(color);	
	this.polygonTextureVector.insertElementAt(texture,this.indexPolygon());
    }
    
    /**
     * Removes a texture from the circle vector. 
     * @throws Texture.TextureAdministratorError
     * @param indexCircle The index of the vector we want to remove the texture.
     */
    public void removeCircleTexture(int indexCircle) throws TextureAdministratorError {
	/**
	 * If the indexCircle isn't in the scope of the vector then we raise an exception.
	 */
	if(indexCircle >= this.indexCircle() || indexCircle < 0) {
	    throw new TextureAdministratorError("The texture you want to choose doesn't exist");
	}
	else {
	    this.circleTextureVector.removeElementAt(indexCircle);
	}
    }

    /**
     * Removes a texture from the rectangle vector. 
     * @throws Texture.TextureAdministratorError
     * @param indexRectangle The index of the vector we want to remove the texture.
     */
    public void removeRectangleTexture(int indexRectangle) throws TextureAdministratorError {
	/**
	 * If the indexRectangle isn't in the scope of the vector then we raise an exception.
	 */
	if(indexRectangle >= this.indexRectangle() || indexRectangle < 0) {
	    throw new TextureAdministratorError("The texture you want to choose doesn't exist");
	}
	else {
	    this.rectangleTextureVector.removeElementAt(indexRectangle);
	}
    }

    /**
     * Removes a texture from the polygon vector. 
     * @throws Texture.TextureAdministratorError
     * @param indexPolygon The index of the vector we want to remove the texture.
     */
    public void removePolygonTexture(int indexPolygon) throws TextureAdministratorError {
	/**
	 * If the indexPolygon isn't in the scope of the vector then we raise an exception.
	 */
	if(indexPolygon >= this.indexPolygon() || indexPolygon < 0) {
	    throw new TextureAdministratorError("The texture you want to choose doesn't exist");
	}
	else {
	    this.polygonTextureVector.removeElementAt(indexPolygon);
	}
    }

    /**
     * This method removes all the element of the different vectors.
     */
    public void clearAll() {
	this.circleTextureVector.removeAllElements();
	this.rectangleTextureVector.removeAllElements();
	this.polygonTextureVector.removeAllElements();
    }	  

    /**
     * This method  return the texture associated to a circle. 
     * @see Texture.Texture
     * @throws Texture.TextureAdministratorError
     * @param indexCircle The index in the circle vector we want to acceed the texture.
     * @return The texture if the index is in the scope of the circle vector, a TextureAdministratorError otherwise.
     */
    public Texture returnCircleTexture(int indexCircle) throws TextureAdministratorError {
	/**
	 * If the indexCircle isn't in the scope of the vector then we raise an exception.
	 */
	if(indexCircle >= this.indexCircle() || indexCircle < 0) {
	    throw new TextureAdministratorError("The texture you want to choose doesn't exist");
	}
	else {
	    return (Texture)circleTextureVector.elementAt(indexCircle);
	}
    }

    /**
     * This method  return the texture associated to a rectangle. 
     * @see Texture.Texture
     * @throws Texture.TextureAdministratorError
     * @param indexRectangle The index in the rectangle vector we want to acceed the texture.
     * @return The texture if the index is in the scope of the rectangle vector, a TextureAdministratorError otherwise.
     */
    public Texture returnRectangleTexture(int indexRectangle) throws TextureAdministratorError {
	/**
	 * If the indexRectangle isn't in the scope of the vector then we raise an exception.
	 */
	if(indexRectangle >= this.indexRectangle() || indexRectangle < 0) {
	    throw new TextureAdministratorError("The texture you want to choose doesn't exist");
	}
	else {
	    return (Texture)rectangleTextureVector.elementAt(indexRectangle);
	}
    }

    /**
     * This method  return the texture associated to a polygon. 
     * @see Texture.Texture
     * @see Texture.TextureAdministratorError
     * @param indexPolygon The index in the polygon vector we want to acceed the texture.
     * @return The texture if the index is in the scope of the polygon vector, a TextureAdministratorError otherwise.
     */
    public Texture returnPolygonTexture(int indexPolygon) throws TextureAdministratorError {
	/**
	 * If the indexPolygon isn't in the scope of the vector then we raise an exception.
	 */
	if(indexPolygon >= this.indexPolygon() || indexPolygon < 0) {
	    throw new TextureAdministratorError("The texture you want to choose doesn't exist");
	}
	else {
	    return (Texture)polygonTextureVector.elementAt(indexPolygon);
	}
    }   

    /**
     * This method returns the size of the circle vector.
     * @return The size of the circle vector.
     */
    public int indexCircle() {
	return this.circleTextureVector.size();
    }
    
    /**
     * This method returns the size of the rectangle vector.
     * @return The size of the rectangle vector.
     */
    public int indexRectangle() {
	return this.rectangleTextureVector.size();
    }

    /**
     * This method returns the size of the polygon vector.
     * @return The size of the polygon vector.
     */
    public int indexPolygon() {
	return this.polygonTextureVector.size();
    }

}
