package spikingneuron;

import java.awt.*;
import java.applet.*;
import java.awt.event.*;
import javax.swing.*; 
import javax.swing.border.*; 

import com.objectspace.jgl.Pair;

import spikingneuron.math.*;
import spikingneuron.tools.*;
import spikingneuron.neurons.*;
import spikingneuron.drawable.*;
import spikingneuron.generators.*;
import spikingneuron.userinterface.*;

/**
*<FONT SIZE=2>
* @version 1.1, Lausanne le 2 Juillet 1998 
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Classe de base de l'application 'SingleNeuron', permettant de visualiser le potentiel membranaire
* d'un neurone unique, et l'influence des diffrents paramtres des modles implments.
* </TT></STRONG><FONT SIZE=3>
* <P>
* L'implmentation de la classe permet l'excution de l'application via l'interpreteur Java (avec la
* commande <CODE>java spikingneuron.SingleNeuron</CODE> (en supposant que le classpath soit correct)),
* ou via un navigateur (butineur), la classe tant une extension de JApplet. Attention, il faut utiliser
* un navigateur qui supporte java 1.1, ainsi que les JFC (<I>swing</I>). <BR>
* Rem: lors de l'excution sous forme d'application (via l'interprteur), la rservation pralable de
* suffisemment de mmoire, ainsi que la dsactivation de la vrification des classes permettent d'acclrer
* sensiblement l'excution. <BR>
* (<B>Commande:</B><CODE> java -noverify -ms20m -mx40m spikingneuron.SingleNeuron</CODE>).
* <P>
*/

public class SingleNeuron extends JApplet implements Runnable {

    public static final double EPSILON = 1E-15; // -> 0
    private static final int NB_DOTS = 10000; // Points calculs (fractions temporelles)

    // Attributs divers .................................................
    public Container cp; // Conteneur global.

    NeuronBaseModel neuron; // le neuron implmentant le modle dsir.
    SignalGenerator generator; // idem avec la source externe de courant.


    // Elements de l'interface utilisateur ..........................
	

    // Les polices de l'application:
    public Font defaultFont;
    public Font boldFont;
    public Font bigFont;
    public Font bigBoldFont;
    public Font smallMonoFont;

    // Les controles globaux:
    SingleNeuron.StartJButton computeBtn; // Le bouton permettant de lancer la simulation.
    SingleNeuron.StopJButton stopBtn; //Le bouton permettant d'arreter la simulation
    DoubleGI tMax; // Controle permettant  l'utilisateur de fixer la borne max de la simulation.
    LongGI	 spikeCounter; // Compteur d'impulsions (interface)
    DoubleGI spikeFrequence; // Frquence des impulsions (interface)
    /*
      spikeCounter et spikeFrequence devraient appartenir  une classe particulire,
      implmentant soit DataFlowAgent et pouvant etre connecte  un neurone,
      soit (mieux) une interface du genre SpikeListener...
      Il faudrait alors comme source d'vnement un DataFlowAgent qui filtre les spikes
      d'un ou des neurones du systme.
    */

    // Les composants de la fenetre:
    SingleNeuron.TitleCanvas headerCnv;
    SingleNeuron.NeuronUI    neuronUI;
    SingleNeuron.StimulusUI  stimulusUI;

    // Les diffrentes courbes & canvas d'affichage:
    GraphCanvas    stimulusCnv;
    GraphCanvas    neuronCnv;
    HLine          thresholdCurve; // A changer ! (Hirarchie d'objets graphiques plus complte)
    LineOfPoints   stimulusCurve;
    SpikesCurve    spikesCurve;
    PotentialCurve neuronCurve, holdedCurve;

    SingleNeuron.HoldButton         holdBtn;
    SingleNeuron.ResetHoldedButton  resetBtn;

    // Une reference sur moi
    SingleNeuron singleNeuron;

    // Les diffrentes interfaces utilisateur
    SignalGeneratorGI generatorInterface; // interface utilisateur de la source de courant
    NeuronBaseModelGI neuronInterface; // interface utilisateur du modle de neurone



    // Application start point.............
    public static void main(String args[]) {
	// On cherche les dimensions de l'ecran de maniere a placer la fenetre au milieu de
	// l'ecran. Rajoute par Sebastien Baehni
	Dimension dimension = Toolkit.getDefaultToolkit().getScreenSize();
	// On recr l'environnement d'une applet, except l'appel  init().	
	SingleNeuron applet = new SingleNeuron();
	SingleNeuron.PrimaryFrame window = applet.new PrimaryFrame("Spiking: neurone simple");
	applet.realInit(); // applet.init() pour simuler un browser.
	//applet.start();
	window.getContentPane().add(BorderLayout.CENTER,applet); 
	window.pack();
	window.setLocation(Math.abs((dimension.width-window.getSize().width)/2),Math.abs((dimension.height-window.getSize().height)/2));
	window.show();
    }

    // Constructeur ..................	
    public SingleNeuron()  {
	Tools.isAnApplet = false;
    }

    // Applet Life-Cycle: Start Point
    public final void init() {
	Tools.isAnApplet = true;
	realInit();
    }

    public final void start() {}
	

    // Initialisation .........................
    public final void realInit() {
	// Elements globaux susceptible d'etre utilis par tous les objets de l'app.
	defaultFont		= new Font("SanSerif", Font.PLAIN, 11);
	boldFont		= new Font("SanSerif", Font.BOLD, 11);
	bigFont			= new Font("Dialog", Font.PLAIN, 10);
	bigBoldFont		= new Font("Dialog", Font.BOLD, 10);
	smallMonoFont	= new Font("Monospaced", Font.PLAIN, 9);

	/* Creation des elements de l'interface */
	cp = getContentPane();
	cp.setLayout(new GridBagLayout());
	holdBtn = new SingleNeuron.HoldButton();
	resetBtn = new SingleNeuron.ResetHoldedButton();
	headerCnv = new SingleNeuron.TitleCanvas();
	computeBtn = new SingleNeuron.StartJButton(); 
	stopBtn = new SingleNeuron.StopJButton();
		
	tMax = new DoubleGI(200.,10,"Temps  [ms]",DoubleGI.LABEL_AT_LEFT,JLabel.LEFT);
	tMax.setValidityDomaine(EPSILON,Double.MAX_VALUE);
	tMax.setToolTipText("Temps maximum pour la simulation.");
	tMax.setFontLabels(defaultFont);
	spikeCounter = new LongGI(0,10,"Nb. de Spikes",DoubleGI.LABEL_AT_LEFT,JLabel.LEFT);
	spikeCounter.setFontLabels(defaultFont);
	spikeCounter.field.setOpaque(true);
	spikeCounter.field.setEditable(false);
	spikeCounter.field.setHorizontalAlignment(JLabel.CENTER);
	spikeCounter.setToolTipText("Nombre d'implusions.");
	spikeFrequence = new DoubleGI(0.0,10,"Frequence [Hz]",DoubleGI.LABEL_AT_LEFT,JLabel.LEFT);
	spikeFrequence.setFontLabels(defaultFont);
	spikeFrequence.setLimitedDecimales(true);
	spikeFrequence.field.setOpaque(true);
	spikeFrequence.field.setEditable(false);
	spikeFrequence.field.setHorizontalAlignment(JLabel.CENTER);
	spikeFrequence.setToolTipText("Frequence des impulsions [Hz].");
		
	//		Abandon (provisoire ?) du groupe de controles: alignement (disposition) irralisable.
	//		PrimitiveTypeGroup controlGrp = new PrimitiveTypeGroup();
	//		controlGrp.add(tMax);
	//		controlGrp.add(spikeCounter);
	//		controlGrp.add(spikeFrequence);


	// Courbes et canvas du potentiel membranaire
	neuronUI = new SingleNeuron.NeuronUI();
	holdedCurve = new PotentialCurve(Color.gray,Color.gray,Color.gray); // couleurs non utilise.
	neuronCurve = new PotentialCurve(new Color(50,50,100), // trace
					 new Color(190,70,20), // spike
					 new Color(160,160,160)); // refract trace
	neuronCurve.enableRefractoryTrace();
	thresholdCurve = new HLine(new Color(50,50,200));
	neuronCnv = new GraphCanvas(400, 250,
				    0.02,0.0,tMax.getValue()*1E-3,
				    0.1,0.0, 70e-3);
	neuronCnv.setAxesLabelFactor(1e3,1e3);
	neuronCnv.setFontLabels(smallMonoFont);
	neuronCnv.setBorder(new TitledBorder(new EmptyBorder(0,0,0,0), //LineBorder.createGrayLineBorder(),
					     "Potentiel membranaire du neurone [mV]", TitledBorder.DEFAULT_JUSTIFICATION,
					     TitledBorder.DEFAULT_POSITION, bigFont));
	neuronCnv.addItem(holdedCurve);
	neuronCnv.addItem(thresholdCurve);
	neuronCnv.addSynchronizedItem(neuronCurve);

	// Courbes et canvas de la source externe
	stimulusUI = new SingleNeuron.StimulusUI();
	spikesCurve = new SpikesCurve(Color.lightGray);
	spikesCurve.setHeightLineRatio(0.9);
	stimulusCurve = new LineOfPoints(Color.red);
	stimulusCnv = new GraphCanvas(400,250,
				      0.02,0.0,tMax.getValue()*1E-3,
				      0.1,0.0,10e-6);
	stimulusCnv.setAxesLabelFactor(1e3,1e6);
	stimulusCnv.setFontLabels(smallMonoFont);
	stimulusCnv.setBorder(new TitledBorder(new EmptyBorder(0,0,0,0),
					       "Signal externe d'entree [uA]", TitledBorder.DEFAULT_JUSTIFICATION,
					       TitledBorder.DEFAULT_POSITION, bigFont));
	stimulusCnv.addSynchronizedItem(spikesCurve);
	stimulusCnv.addSynchronizedItem(stimulusCurve);


	/* Composition de la fenetre */
	Constraints gbConstraints = new Constraints();
	cp.add(headerCnv,
	       gbConstraints.setAndGet(0,0,
				       1,3,
				       Constraints.BOTH,
				       0,0,
				       0,0,0,0,
				       Constraints.WEST,
				       0.8, 0.0));

	//  Composition du groupe des controles................

	//		cp.add(controlGrp,
	//			gbConstraints.setAndGet(1,0,
	//									1,1,
	//									Constraints.BOTH,
	//									0,0,
	//									0,0,0,5,
	//									Constraints.CENTER,
	//									0.0, 0.0));

	// Composition 'manuelle' ................................
	
	cp.add(tMax.label,
	       gbConstraints.setAndGet(1,0,
				       1,1,
				       Constraints.BOTH,
				       0,0,
				       0,0,0,0,
				       Constraints.CENTER,
				       0.05, 0.0));
	cp.add(tMax.signifiant,
	       gbConstraints.setAndGet(2,0,
				       1,1,
				       Constraints.BOTH,
				       0,0,
				       0,0,2,2,
				       Constraints.CENTER,
				       0.1, 0.0));
	cp.add(spikeCounter.label,
	       gbConstraints.setAndGet(1,1,
				       1,1,
				       Constraints.BOTH,
				       0,0,
				       0,0,0,0,
				       Constraints.CENTER,
				       0.05, 0.0));
	cp.add(spikeCounter.signifiant,
	       gbConstraints.setAndGet(2,1,
				       1,1,
				       Constraints.BOTH,
				       0,0,
				       0,0,2,2,
				       Constraints.CENTER,
				       0.1, 0.0));
	cp.add(spikeFrequence.label,
	       gbConstraints.setAndGet(1,2,
				       1,1,
				       Constraints.BOTH,
				       0,0,
				       0,0,0,0,
				       Constraints.CENTER,
				       0.05, 0.0));
	cp.add(spikeFrequence.signifiant,
	       gbConstraints.setAndGet(2,2,
				       1,1,
				       Constraints.BOTH,
				       0,0,
				       0,0,2,2,
				       Constraints.CENTER,
				       0.1, 0.0));


	//   .... Suite de la composition de l'interface.

	cp.add(computeBtn,
	       gbConstraints.setAndGet(3,0,
				       Constraints.RELATIVE,3,
				       Constraints.BOTH,
				       0,0,
				       0,0,1,0,
				       Constraints.CENTER,
				       0.05, 0.0));
	cp.add(stopBtn,
	       gbConstraints.setAndGet(4,0,
				       Constraints.REMAINDER,3,
				       Constraints.BOTH,
				       0,0,
				       0,0,1,0,
				       Constraints.CENTER,
				       0.05, 0.0));
	cp.add(stimulusCnv,
	       gbConstraints.setAndGet(0, 4,
				       2, 1,
				       Constraints.BOTH,
				       0,0,
				       10,5,0,0,
				       Constraints.CENTER,
				       0.85, 0.5));
	cp.add(stimulusUI,
	       gbConstraints.setAndGet(2, 4,
				       Constraints.REMAINDER, 1,
				       Constraints.BOTH,
				       0,0,
				       10,5,0,0,
				       Constraints.CENTER,
				       0.15, 0.5));

	cp.add(neuronCnv,
	       gbConstraints.setAndGet(0, 5,
				       2, Constraints.REMAINDER,
				       Constraints.BOTH,
				       0,0,
				       5,0,0,0,
				       Constraints.CENTER,
				       0.85, 0.5));
	cp.add(neuronUI,
	       gbConstraints.setAndGet(2,5,
				       Constraints.REMAINDER, Constraints.REMAINDER,
				       Constraints.BOTH,
				       0,0,
				       5,0,0,0,
				       Constraints.CENTER,
				       0.15, 0.5));

	// Activation & Paramtrage des tooltip.
	ToolTipManager.sharedInstance().setEnabled(true);
	ToolTipManager.sharedInstance().setInitialDelay(1000);
	ToolTipManager.sharedInstance().setDismissDelay(8000);
	ToolTipManager.sharedInstance().setReshowDelay(2000);
	singleNeuron = this;
    }
	

    /**
     * Ordonnancement des diffrentes actions  raliser pour effectuer la simulation proprement dite.
     */
    public void run() {
	long   spikes = 0; // nombre total de spikes.
	double maxT = tMax.getValue()*1E-3; // Dure maximume simule.
	double dt = maxT/NB_DOTS; // delta t. Intervalle entre calculs.

	// A chaque pas d'intgration, diffrents lments de l'interfaces doivent tre
	// rafraichis. On mmorise donc une fois en dbut de mthode les contextes d'affichage,
	// afin de ne pas ralentir inutilement le processus en les redemendant systmatiquement.
	// Meme 'truc' (pas tres lgant) que pour DisplayContext.
	/*
	 * Repris par Sbastien Baehni.
	 * Florian fait des update(Graphics g) o g sont les graphique qu'il obtient ci-dessous.
	 * Malheureusement, lors de l'appel  la mthode update (cf. plus bas), un nullpointerexception
	 * est lev => on fait appel  des repaint() tout simplement.
	 */
	//Graphics globalGrpCtx = getGraphics();
	//Graphics spkCntGrpCtx = spikeCounter.signifiant.getGraphics();
	//Graphics spkFrqGrpCtx = spikeFrequence.signifiant.getGraphics();

	spikeCounter.setValue(0);
	spikeFrequence.setValue(0.0);

	/*
	 * Repris par Sbastien Baehni.
	 * On fait un repaint  la place de l'update.
	 */
	//update(globalGrpCtx); // Changement d'tat du bouton 'start' et des compteurs.	
	this.repaint();
			
	// Initialisation de l'horloge:
	Clock.sharedInstance.setDeltaT(dt);
	Clock.sharedInstance.setMaximalTime(maxT);


	// Synchronisation des modles avec les interfaces:
	generatorInterface.synchronizeSignifie(generator);
	neuronInterface.synchronizeSignifie(neuron);

	// Rinitialisation du flux de donnes... par le biais d'une rinit. de ses agents.
	Clock.sharedInstance.resetTime();
	neuron.resetTime();
	generator.resetTime();
	spikesCurve.resetTime();
	neuronCurve.resetTime();
	stimulusCurve.resetTime();
	NeuronBaseModel.resetFireCounter();

	// Calcul des zones d'affichages des courbes... (estimation du rectangle englobant)

	// 1) Obtention des nouvelles plages d'affichage:
	Coordinate2D generatorRange = generator.getRange();
	Coordinate2D neuronRange = neuron.getRange();
	double genMax = generatorRange.getC1(); // fast !
	double genMin = generatorRange.getC0();
	double neuronMax = neuronRange.getC1(); 
	double neuronMin = neuronRange.getC0();

	// 2) Traitement des valeurs particulire, et assurance de cadrer les axes
	if ((genMax-genMin<EPSILON)&&(Math.abs(genMax)<EPSILON)) {
	    genMax = 1e-6;
	    genMin = -1e-6;
	} else {
	    genMax = Math.max(0.0, genMax);
	    genMin = Math.min(0.0, genMin);
	}
	neuronMax = Math.max(0.0, neuronMax);
	neuronMin = Math.min(0.0, neuronMin);

	neuronRange = holdedCurve.getVerticalBounds();
	neuronMax = Math.max(neuronMax, neuronRange.getC1());
	neuronMin = Math.min(neuronMin, neuronRange.getC0());
			

	// 3) Changement des zones d'affichage
	neuronCnv.changeVirtualBounds(0.02,0.,maxT,0.1,neuronMin,neuronMax);
	stimulusCnv.changeVirtualBounds(0.02,0.,maxT,0.1,genMin,genMax);

	// Initialisation diverses.....
	thresholdCurve.setOrdonnee(neuron.getMeanThreshold());
	neuronCurve.setLimits(neuron.getMeanThreshold(),0.0,10);
	stimulusCurve.ensureCapacity(NB_DOTS); 
	neuronCurve.ensureCapacity(NB_DOTS);

	/*
	  L'invocation de ensureCapacity() n'est pas ncessaire avec l'actuelle implmentation, dans laquelle
	  le nombre de points calculs est fixe. Dans le cas o cette grandeure est dynamique (soit dterminable
	  par l'utilisateur, soit calcule en fonction du facteur de zoom p. ex.), il est recommend d'invoquer
	  ensureCapacity() afin de disposer d'un trac fluide... (toutefois, il semble que sur certaines
	  architectures, l'effet dsir n'est pas obtenu... le problme peut provenir de la gestion virtuelle
	  de la mmoire: le systme d'exploitation accepte l'allocation, mais ne la ralise pas physiquement.
	  Ce n'est que lorque la mmoire est rellement utilise que le swapper libre l'espace mmoire disponible,
	  ce qui conduit  interrompre le trac pendant un temps relativement long.
	  (Pour pallier  ce problme, il suffirait de ne pas tracer les courbes en 'temps rel', mais d'afficher
	  une barre de progression, ce qui rsoudrait en mme temps le problme de l'estimation du domaine 
	  afficher).
	  Une autre solution (si lancement de l'applic. via la ligne de commande) est de spcifi  la machine
	  virtuelle une taille initial de tas suffisante.
	*/
		 
			
	// Mise  jours des canvas
	neuronCnv.update();
	stimulusCnv.update();
			
	// Rinitialisation des connextions ........................
	spikesCurve.setInput(neuron);
	neuronCurve.setInput(neuron);
	stimulusCurve.setInput(generator);
	neuron.removeExternalInputs();
	neuron.addExternalInput(generator);

	long tic = 0;
				
	while (Clock.sharedInstance.tic()) {
	    // Tous les DataFlowAgent doivent traiter la nouvelle plage temporelle:
	    tic++;
	    generator.computeNextTic();
	    neuron.computeNextTic();
	    spikesCurve.computeNextTic();
	    neuronCurve.computeNextTic();
	    stimulusCurve.computeNextTic();

	    if (tic%50 == 0) {
				// Mise  jours des 'statistiques' d'impulsions:
		spikeCounter.setValue(NeuronBaseModel.getFireCounter());
		spikeFrequence.setValue(NeuronBaseModel.getFireCounter()/Clock.sharedInstance.getTime());
		/* 
		 * Repris par Sbastien Baehni.
		 * Idem qu'avant. 
		 */
		//spikeCounter.signifiant.update(spkCntGrpCtx);
		//spikeFrequence.signifiant.update(spkFrqGrpCtx);
		spikeCounter.signifiant.repaint();				
		spikeFrequence.signifiant.repaint();
	    }

	    // Mise  jour des sortie de tous les RealFlowProducer (pseudo-simultanit)
	    Clock.sharedInstance.updateOutput();
	    generator.updateOutput();
	    neuron.updateOutput();
	}

	// Pour terminer, il faut (r)afficher les courbes, re-centres et r-chelonnes.

	spikeCounter.setValue(NeuronBaseModel.getFireCounter());
	spikeFrequence.setValue(NeuronBaseModel.getFireCounter()/maxT);

	// 1) Valeurs limites exactes, obtenues via les 'courbes'
	neuronRange = neuronCurve.getVerticalBounds();
	generatorRange = stimulusCurve.getVerticalBounds();

	// 2) Modif. pour inclure les axes
	if ((generatorRange.getC1()-generatorRange.getC0()<EPSILON) &&
	    (Math.abs(generatorRange.getC0())<EPSILON)) {
	    generatorRange.setC0(-1e-6);
	    generatorRange.setC1(1e-6);
	} else {
	    generatorRange.setC0(Math.min(0.0,generatorRange.getC0()));
	    generatorRange.setC1(Math.max(0.0,generatorRange.getC1()));
	}
	neuronMin = Math.min(0.0,neuronRange.getC0());
	neuronMax = Math.max(0.0,neuronRange.getC1());
	neuronRange = holdedCurve.getVerticalBounds();
	neuronMax = Math.max(neuronMax, neuronRange.getC1());
	neuronMin = Math.min(neuronMin, neuronRange.getC0());
		
	// 3) Mise  jour de la fonction affine de tranf. des coord.
	neuronCnv.changeVirtualBounds(0.02,0.0, maxT,
				      0.1,neuronMin,neuronMax);
	stimulusCnv.changeVirtualBounds(0.02,0.0, maxT,
					0.1,generatorRange.getC0(),generatorRange.getC1());
	// 4) R-affichage.
	repaint();
    }
	
	
    /**
     *<FONT SIZE=2>
     * @version 1.1, Lausanne le 3 septembre 1998 
     * @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.)
     * <P><FONT SIZE=4><TT><STRONG>
     * Classe assurant l'instanciation et le paramtrage d'un gnrateur externe de courant.
     * </TT></STRONG><FONT SIZE=3><P>
     * Les diffrents type de gnrateurs sont tous instancis 1x, ainsi que leur interface utilisateur.
     * Un ComboBox permet la slection du gnrateur, et l'interface permettant de le paramtrer est affiche
     * via un 'Card layout'. <BR>
     * Vu le nombre de gnrateurs, cette solution est  la limite de l'acceptable. Avec un nombre encore plus
     * lev de gnrateurs (ds 10), il serait prfrable de crer une classe englobant un ensemble quelconque
     * d'interface et d'objet, et utilisant une hashtable pour effectuer le stockage des paires d'objet,
     * en conservant ainsi la relation objet-interface (la reprsentation symbolique de l'interface tant la cl
     * de hachage...  explorer galement, les possibilits offertes par la <I>'Reflexion API'</I>.
     * <P>
     */
    private class StimulusUI extends JPanel implements ItemListener {
	JComboBox	generatorsChoice;
	JPanel 		cards;
	/* Les differentes instances de generateurs */
	CircularGenerator circularGenerator;
	CircularGeneratorGI circularGenInterface;
	PositiveGenerator positiveGenerator;
	PositiveGeneratorGI positiveGenInterface;
	SquareGenerator squareGenerator;
	SquareGeneratorGI squareGenInterface;
	PulsesGenerator pulsesGenerator;
	PulsesGeneratorGI pulsesGenInterface;
	RandomPulsesGenerator randomPulsesGenerator;
	RandomPulsesGeneratorGI randomPulsesGenInterface;
	NoisyGenerator noisyGenerator;
	NoisyGeneratorGI noisyGenInterface;
	DiscreteNoisyGenerator discreteNoisyGenerator;
	DiscreteNoisyGeneratorGI discreteNoisyGenInterface;
	StepGenerator stepGenerator;
	StepGeneratorGI stepGenInterface;
	// ... les autres generateurs & interfaces
		

	StimulusUI() {
	    setLayout(new GridBagLayout());
	    Constraints contraintes = new Constraints();

	    // Construction du comboBox .................................			
	    TitledBorder title = new TitledBorder(new EmptyBorder(0,0,0,0),
						  "Courbe courante",	TitledBorder.CENTER, TitledBorder.DEFAULT_POSITION,	bigFont);
	    setBorder(title);
	    generatorsChoice = new JComboBox();
	    generatorsChoice.setBorder(new CompoundBorder(
							  generatorsChoice.getBorder(), new EmptyBorder(2,2,2,2) ));
	    generatorsChoice.setRenderer(new SingleNeuron.MyCellRenderer());
	    generatorsChoice.setMaximumRowCount(4);
	    generatorsChoice.setLightWeightPopupEnabled(true);
	    generatorsChoice.addItemListener(this);
			
	    // Cration du 'CardLayout' contenant les diffrentes interfaces disponibles:
	    // 1) Cration d'un JPanel gr par un CardLayout
	    // 2) Ajouter au panel les differents GUI des differents generateurs, en specifiant
	    //    le hashCode de la resprsentation utilise dans le comboBox comme identifiant.
	    // 3) Afficher dans le panel cr le GUI determin par le choix dans le JComboBox.
	
	    cards = new JPanel();
	    cards.setLayout(new CardLayout());

	    /* Creation des generateurs */
	    // Signal priodique, fct. circulaire (sinusoidal)
	    circularGenerator = new CircularGenerator(); // instanciation
	    circularGenerator.init(11.e-8,0.1e-7,100.0,0.); // initialisation
	    circularGenInterface = new CircularGeneratorGI(circularGenerator); // interface
	    circularGenInterface.setFontLabels(defaultFont);
	    cards.add(circularGenInterface.getInterface(),
		      Integer.toString(circularGenInterface.getSymbolic().hashCode()));
	    // Signal step
	    stepGenerator = new StepGenerator();
	    stepGenerator.init(11e-8,0.1e-6,0.5);
	    stepGenInterface = new StepGeneratorGI(stepGenerator);
	    stepGenInterface.setFontLabels(defaultFont);
	    cards.add(stepGenInterface.getInterface(),
		      Integer.toString(stepGenInterface.getSymbolic().hashCode()));
	    // Signal impulsions temps constants
	    pulsesGenerator = new PulsesGenerator(11e-8,3,0.01e-6,1);	 
	    pulsesGenInterface = new PulsesGeneratorGI(pulsesGenerator);
	    pulsesGenInterface.setFontLabels(defaultFont);
	    cards.add(pulsesGenInterface.getInterface(),
		      Integer.toString(pulsesGenInterface.getSymbolic().hashCode()));
	    // Signal impulsions loi exponentielle
	    randomPulsesGenerator = new RandomPulsesGenerator(11e-8,75.0,0.01e-6,1,System.currentTimeMillis());	  
	    randomPulsesGenInterface = new RandomPulsesGeneratorGI(randomPulsesGenerator);
	    randomPulsesGenInterface.setFontLabels(defaultFont);
	    cards.add(randomPulsesGenInterface.getInterface(),
		      Integer.toString(randomPulsesGenInterface.getSymbolic().hashCode()));
	    // Modle fct continue de bruit
	    noisyGenerator = new NoisyGenerator(11e-8,0.5e-6,10,System.currentTimeMillis()); // instanciation	  
	    noisyGenInterface = new NoisyGeneratorGI(noisyGenerator); // interface
	    noisyGenInterface.setFontLabels(defaultFont);
	    cards.add(noisyGenInterface.getInterface(),
		      Integer.toString(noisyGenInterface.getSymbolic().hashCode()));
	    // Modle fct dicrete de bruit
	    discreteNoisyGenerator = new DiscreteNoisyGenerator(); // instanciation
	    discreteNoisyGenerator.setOffset(11.e-8);
	    discreteNoisyGenerator.setAmplitude(0.5e-6);
	    discreteNoisyGenInterface = new DiscreteNoisyGeneratorGI(discreteNoisyGenerator); // interface
	    discreteNoisyGenInterface.setFontLabels(defaultFont);
	    cards.add(discreteNoisyGenInterface.getInterface(),
		      Integer.toString(discreteNoisyGenInterface.getSymbolic().hashCode()));
	    // Signal priodique, fct. circulaire positive (sinusoide redresse)
	    positiveGenerator = new PositiveGenerator();
	    positiveGenerator.init(11e-8,3e-7,100.0,0.);
	    positiveGenInterface = new PositiveGeneratorGI(positiveGenerator);
	    positiveGenInterface.setFontLabels(defaultFont);
	    cards.add(positiveGenInterface.getInterface(),
		      Integer.toString(positiveGenInterface.getSymbolic().hashCode()));
	    // Signal priodique, fct. carre (crnaux)
	    squareGenerator = new SquareGenerator();
	    squareGenerator.init(11.e-8, 5.e-6, 0.e-6, new double[]
		{0.03, 0.01, 0.02, 0.01, 0.03, 0.01, 0.02, 0.01});
	    squareGenInterface = new SquareGeneratorGI(squareGenerator);
	    squareGenInterface.setFontLabels(defaultFont);
	    cards.add(squareGenInterface.getInterface(),
		      Integer.toString(squareGenInterface.getSymbolic().hashCode()));
	    // ...

	    // Ajout des diffrents modles de gnrateurs au ComboBox:
	    generatorsChoice.addItem(circularGenInterface.getSymbolic());
	    generatorsChoice.addItem(stepGenInterface.getSymbolic());
	    generatorsChoice.addItem(pulsesGenInterface.getSymbolic());
	    generatorsChoice.addItem(randomPulsesGenInterface.getSymbolic());
	    generatorsChoice.addItem(noisyGenInterface.getSymbolic());
	    generatorsChoice.addItem(positiveGenInterface.getSymbolic());
	    generatorsChoice.addItem(squareGenInterface.getSymbolic());
	    generatorsChoice.addItem(discreteNoisyGenInterface.getSymbolic());

	    // Composition du Panel global:
	    add(generatorsChoice,
		contraintes.setAndGet(0,0,
				      Constraints.REMAINDER,1,
				      Constraints.NONE,
				      0,0,
				      0,5,0,0,
				      Constraints.NORTH,
				      1.0, 0.0));
	    add(cards,
		contraintes.setAndGet(0,1,
				      Constraints.REMAINDER,1,
				      Constraints.BOTH,
				      0,0,
				      5,0,0,0,
				      Constraints.CENTER,
				      1.0, 1.0));

	    // Generateur & Interface initiaux ..........................
	    generator = circularGenerator;
	    generatorInterface = circularGenInterface;
	}
		
	public void itemStateChanged(ItemEvent e) {
	    ((CardLayout)cards.getLayout()).show(cards,
						 Integer.toString(generatorsChoice.getSelectedItem().hashCode()));
	    switch (generatorsChoice.getSelectedIndex()) {
	    case 0 : 	generator = circularGenerator;
		generatorInterface = circularGenInterface;
		break;
	    case 1 : 	generator = stepGenerator;
		generatorInterface = stepGenInterface;
		break;
	    case 2 : 	generator = pulsesGenerator;
		generatorInterface = pulsesGenInterface;
		break;
	    case 3 : 	generator = randomPulsesGenerator;
		generatorInterface = randomPulsesGenInterface;
		break;
	    case 4 : 	generator = noisyGenerator;
		generatorInterface = noisyGenInterface;
		break;
	    case 5 : 	generator = positiveGenerator;
		generatorInterface = positiveGenInterface;
		break;
	    case 6 : 	generator = squareGenerator;
		generatorInterface = squareGenInterface;
		break;
	    case 7 : 	generator = discreteNoisyGenerator;
		generatorInterface = discreteNoisyGenInterface;
		break;
	    }
	}
    }


    /**
     *<FONT SIZE=2>
     * @version 1.1, Lausanne le 3 septembre 1998 
     * @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.)
     * <P><FONT SIZE=4><TT><STRONG>
     * Classe assurant l'instanciation et le paramtrage d'un neurone.
     * </TT></STRONG><FONT SIZE=3><P>
     * La technique est exactement la mme que pour les gnrateurs de courant.
     * <P>
     */
    private class NeuronUI extends JPanel implements ItemListener {
	JComboBox	neuronsChoice;
	JPanel		cards;
	/* Les differentes instances de neurones */
	IntegrateAndFire ifNeuron;
	IntegrateAndFireGI ifInterface;
	SRMLongMemory srNeuron;
	SRMLongMemoryGI srInterface;
	SRMShortMemory srShortNeuron;
	SRMShortMemoryGI srShortInterface;
	// ... les autres modeles de neurones

	NeuronUI() {
	    setLayout(new GridBagLayout());
	    Constraints contraintes = new Constraints();

	    TitledBorder title = new TitledBorder(new EmptyBorder(0,0,0,0),
						  "Modele neuronal", TitledBorder.CENTER, TitledBorder.DEFAULT_POSITION, bigFont);
	    setBorder(title);

	    // Panel de gestion (rudimentaire) des courbes ......
	    JPanel ctrl = new JPanel();
	    ctrl.setLayout(new GridBagLayout());
	    ctrl.add(holdBtn,contraintes.setAndGet(	0,0,
							Constraints.REMAINDER,1,
							Constraints.HORIZONTAL,
							0,0,
							0,1,0,0,
							Constraints.CENTER,
							1.0, 0.5));
	    ctrl.add(resetBtn,contraintes.setAndGet(0,1,
						    Constraints.REMAINDER,1,
						    Constraints.HORIZONTAL,
						    0,0,
						    1,0,0,0,
						    Constraints.CENTER,
						    1.0, 0.5));

	    // Construction du comboBox .................................			
	    neuronsChoice = new JComboBox();
	    neuronsChoice.setBorder(new CompoundBorder(
						       neuronsChoice.getBorder(),
						       new EmptyBorder(5,5,5,5)
							   ));
	    neuronsChoice.setRenderer(new SingleNeuron.MyCellRenderer());
	    //			neuronsChoice.setMaximumRowCount(3);
	    neuronsChoice.setLightWeightPopupEnabled(true);
	    neuronsChoice.addItemListener(this);
			
	    // Cration du 'CardLayout' contenant les diffrentes interfaces disponibles:
	    // 1) Cration d'un JPanel gr par un CardLayout
	    // 2) Ajouter au panel les differents GUI des differents generateurs, en specifiant
	    //    le hashCode de la resprsentation utilise dans le comboBox comme identifiant.
	    // 3) Afficher dans le panel cr le GUI determin par le choix dans le JComboBox.
	
	    cards = new JPanel();
	    cards.setLayout(new CardLayout());

	    /* Cration des modles de neurones */
	    // Modele Integrate&Fire
	    ifNeuron = new IntegrateAndFire(0); // Instancie le modle,
	    ifNeuron.nbmInit(0.1,  0.,	// Threshold average & gap
			     0.005, 0.,	// RefractoryTime average & gap
			     1.e6,		// Resistance
			     0.02,		// Tau S
			     0.0);		// NoisyCurrent amplitude
	    ifNeuron.ifInit(0.02, 0.0);	// Tau M, ResetPotential
	    ifInterface = new IntegrateAndFireGI(ifNeuron); // Interface
	    ifInterface.setFontLabels(defaultFont);
	    neuronsChoice.addItem(ifInterface.getSymbolic()); // Ajout au JComboBox
	    cards.add(ifInterface.getInterface(),
		      Integer.toString(ifInterface.getSymbolic().hashCode()));
	    // Spike Response (Long Memory)
	    srNeuron = new SRMLongMemory(0); // Instancie le modle,
	    srNeuron.nbmInit(0.1,  0.,		// Threshold average & gap
			     0.005, 0.,		// RefractoryTime average & gap
			     1.e6,			// Resistance
			     0.02,			// Tau S
			     0.0);			// NoisyCurrent amplitude
	    srNeuron.srInit(-0.07, 0.05);	// Eta 0, Tau Eta
	    srInterface = new SRMLongMemoryGI(srNeuron); // Interface
	    srInterface.setFontLabels(defaultFont);
	    neuronsChoice.addItem(srInterface.getSymbolic()); // Ajoute  la slection
	    cards.add(srInterface.getInterface(),
		      Integer.toString(srInterface.getSymbolic().hashCode()));
	    // Spike Response (Short Memory)
	    srShortNeuron = new SRMShortMemory(0); // Instancie le modle,
	    srShortNeuron.nbmInit(0.1,  0.,	// Threshold average & gap
				  0.005, 0.,	// RefractoryTime average & gap
				  1.e6,			// Resistance
				  0.02,			// Tau S
				  0.00);		// NoiseCurrent amplitude
	    srShortNeuron.srInit(-0.07, 0.05);	// Eta 0, Tau Eta
	    srShortInterface = new SRMShortMemoryGI(srShortNeuron); // Interface
	    srShortInterface.setFontLabels(defaultFont);
	    neuronsChoice.addItem(srShortInterface.getSymbolic()); // Ajoute  la slection
	    cards.add(srShortInterface.getInterface(),
		      Integer.toString(srShortInterface.getSymbolic().hashCode()));
	    // ...
			
	    add(neuronsChoice,
		contraintes.setAndGet(	1,0,
					Constraints.REMAINDER,1,
					Constraints.HORIZONTAL,
					0,0,
					0,5,5,5,
					Constraints.CENTER,
					0.95, 0.0));
	    add(cards,
		contraintes.setAndGet(	0,1,
					Constraints.REMAINDER,Constraints.REMAINDER,
					Constraints.BOTH,
					0,0,
					5,0,0,0,
					Constraints.CENTER,
					1.0, 1.0));
	    add(ctrl,
		contraintes.setAndGet(	0,0,
					1,1,
					Constraints.HORIZONTAL,
					4,0,
					0,0,0,5,
					Constraints.WEST,
					0.05, 0.0));
	    /* Neurone & Interface initiaux */
	    neuron = ifNeuron;
	    neuronInterface = ifInterface;
	}
		
	public void itemStateChanged(ItemEvent e) {
	    ((CardLayout)cards.getLayout()).show(cards,
						 Integer.toString(neuronsChoice.getSelectedItem().hashCode()));
	    switch (neuronsChoice.getSelectedIndex()) {
	    case 0 : 	neuron = ifNeuron;
		neuronInterface = ifInterface;
		break;
	    case 1 : 	neuron = srNeuron;
		neuronInterface = srInterface;
		break;
	    case 2 : 	neuron = srShortNeuron;
		neuronInterface = srShortInterface;
		break;
	    }
	}
    }
	

    /* Bouton de lancement de la simulation */

    private class StartJButton extends JButton implements ActionListener {
	Border loweredBorder;
	Border raisedBorder;
		
	StartJButton() {
	    super();
	    loweredBorder = BorderFactory.createLoweredBevelBorder();
	    raisedBorder = BorderFactory.createRaisedBevelBorder();
	    setIcon(Tools.loadImageIcon(getClass(),"/images/computeBtn_R.gif", null));
	    setPressedIcon(Tools.loadImageIcon(getClass(),"/images/computeBtn_P.gif", null));
	    setDisabledIcon(Tools.loadImageIcon(getClass(),"/images/computeBtn_D.gif",null));
	    setRolloverIcon(Tools.loadImageIcon(getClass(),"/images/computeBtn_S.gif",null));
	    //			setBorder(raisedBorder);
	    setBorderPainted(false);
	    setOpaque(false);
	    setFocusPainted(false);
	    setRolloverEnabled(true);
	    addActionListener(this);
	    getAccessibleContext().setAccessibleName("Start");
	    setToolTipText("Demarre la simulation.");
	}

	// Interface: ActionListener...
	public void actionPerformed(ActionEvent event) {
	    setEnabled(false); // Dsactivation du bouton
	    //			setBorder(loweredBorder);
	    new Thread(singleNeuron).start(); // Effectuer la simulation
	    //			setBorder(raisedBorder);
	    setEnabled(true); // Ractivation du bouton
	    holdBtn.setEnabled(true);
	}
    }

    /* Bouton d'arret de la simulation */    
    private class StopJButton extends JButton implements ActionListener {
	Border loweredBorder;
	Border raisedBorder;
		
	StopJButton() {
	    super();
	    loweredBorder = BorderFactory.createLoweredBevelBorder();
	    raisedBorder = BorderFactory.createRaisedBevelBorder();
	    setIcon(Tools.loadImageIcon(getClass(),"/images/stop.gif", null));
	    setPressedIcon(Tools.loadImageIcon(getClass(),"/images/stop_p.gif", null));
	    setDisabledIcon(Tools.loadImageIcon(getClass(),"/images/stop_d.gif",null));
	    setRolloverIcon(Tools.loadImageIcon(getClass(),"/images/stop_r.gif",null));
	    //			setBorder(raisedBorder);
	    setBorderPainted(false);
	    setOpaque(false);
	    setFocusPainted(false);
	    setRolloverEnabled(true);
	    addActionListener(this);
	    getAccessibleContext().setAccessibleName("Stop");
	    setToolTipText("Stoppe la simulation.");
	}
	
	// Interface: ActionListener...
	public void actionPerformed(ActionEvent event) {
	    setEnabled(false); // Dsactivation du bouton
	    Clock.sharedInstance.setTime(Clock.sharedInstance.getMaximalTime());
	    setEnabled(true); // Ractivation du bouton
	    holdBtn.setEnabled(true);
	}
    }
	
    /* Bouton permettant de mmoriser une courbe */
    private class HoldButton extends JButton implements ActionListener {
	Border loweredBorder;
	Border raisedBorder;
		
	HoldButton() {
	    super("Memorise");
	    setFont(defaultFont);
	    loweredBorder = BorderFactory.createLoweredBevelBorder();
	    raisedBorder = BorderFactory.createRaisedBevelBorder();
	    //			setIcon(Tools.loadImageIcon(getClass(),"/images/holdBtn_R.gif", null));
	    //			setPressedIcon(Tools.loadImageIcon(getClass(), "/images/holdBtn_P.gif",null));
	    //			setDisabledIcon(Tools.loadImageIcon(getClass(),	"/images/holdBtn_D.gif", null));
	    //			setRolloverIcon(Tools.loadImageIcon(getClass(), "/images/holdBtn_S.gif", null));
	    setBorderPainted(true);
	    setOpaque(true);
	    setFocusPainted(false);
	    setRolloverEnabled(false);
	    addActionListener(this);
	    getAccessibleContext().setAccessibleName("Memorise");
	    setToolTipText("Memorise la courbe courante.");
	    setEnabled(false);
	}

	public void setEnabled(boolean state) {
	    super.setEnabled(state);
	    setBorder( (state ? raisedBorder : loweredBorder));
	} 

	// Interface: ActionListener...
	public void actionPerformed(ActionEvent event) {
	    // version pas propre... (prob. si autre rfrence  neuronCurve... couleurs init)
	    setEnabled(false);
	    /*
	     * Repris par Sbastien Baehni.
	     * On fait un repaint au lieu d'un update.
	     */
	    //update(getGraphics());
	    repaint();
	    holdedCurve.holdCurve(neuronCurve);
	    resetBtn.setEnabled(true);
	}
		
    }

    /* Bouton permettant d'annuler la mmorisation d'une courbe */
    private class ResetHoldedButton extends JButton implements ActionListener {
	Border loweredBorder;
	Border raisedBorder;
		
	ResetHoldedButton() {
	    super("Efface");
	    setFont(defaultFont);
	    //			setIcon(Tools.loadImageIcon(getClass(), "/images/resetBtn_R.gif", ""));
	    //			setPressedIcon(Tools.loadImageIcon(getClass(), "/images/resetBtn_P.gif",""));
	    //			setDisabledIcon(Tools.loadImageIcon(getClass(), "/images/resetBtn_D.gif",""));
	    //			setRolloverIcon(Tools.loadImageIcon(getClass(), "/images/resetBtn_S.gif",""));
	    loweredBorder = BorderFactory.createLoweredBevelBorder();
	    raisedBorder = BorderFactory.createRaisedBevelBorder();
	    setBorderPainted(true);
	    setOpaque(true);
	    setFocusPainted(false);
	    setRolloverEnabled(false);
	    addActionListener(this);
	    getAccessibleContext().setAccessibleName("Efface");
	    setToolTipText("Efface la courbe memorisee.");
	    setEnabled(false);
	}

	public void setEnabled(boolean state) {
	    super.setEnabled(state);
	    setBorder( (state ? raisedBorder : loweredBorder));
	} 

	// Interface: ActionListener...
	public void actionPerformed(ActionEvent event) {
	    holdedCurve.resetTime();
	    neuronCnv.update();
	    setEnabled(false);
	    holdBtn.setEnabled(true);
	}
    }

    /* Canvas de titre: bouton  2 tat, affichant 2 images (dessin + auteur & date). */
    private class TitleCanvas extends JButton implements ActionListener {
	private final int NB_STATES = 2;
	int state;
	Icon[] icons;
	Border[]  borders;
		
	TitleCanvas() {
	    super();
	    state = 0;
	    icons = new Icon[NB_STATES];
	    borders = new Border[NB_STATES];
	    for (int i = 0; i < NB_STATES; i++)
		icons[i] = Tools.loadImageIcon(getClass(), "/images/Title_s_"+i+".gif",null);
	    borders[0] = BorderFactory.createLoweredBevelBorder();
	    borders[1] = BorderFactory.createRaisedBevelBorder();
			
	    setOpaque(false);
	    setFocusPainted(false);
	    setBorderPainted(false);
	    setRolloverEnabled(false);
	    addActionListener(this);
	    getAccessibleContext().setAccessibleName("Title");
	    setIcon(icons[state]);
	    //			setBorder(borders[state]);
	}

	// Interface: ActionListener...
	public void actionPerformed(ActionEvent event) {
	    state = (state + 1) % NB_STATES;
	    setIcon(icons[state]);
	    //			setBorder(borders[state]);
	}
    }

    /* Fentre de l'application, si excution via la ligne de commande */
    private class PrimaryFrame extends JFrame implements WindowListener { 
	PrimaryFrame(String title) {
	    super(title);
	    addWindowListener(this);
	}
	public void windowActivated(WindowEvent we) {}
	public void windowDeactivated(WindowEvent we) {}
	public void windowDeiconified(WindowEvent we) {}
	public void windowIconified(WindowEvent we) {}
	public void windowOpened(WindowEvent we) {}
	public void windowClosing(WindowEvent we) {
	    dispose();
	}
	public void windowClosed(WindowEvent we) {
	    System.exit(0);
	}
    }

    /**
     *<FONT SIZE=2>
     * @version 1.0, Lausanne le 18 mai 1998 
     * @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.)
     * <P><FONT SIZE=4><TT><STRONG>
     * Classe ralisant l'affichage d'une cellule de combo-box.
     * </TT></STRONG><FONT SIZE=3><P>
     * Dfinit le composant affich dans une cellule de JComboBox, en l'occurence un JLabel compos
     * d'une icone. Les objets  stocker dans la liste/combo Box sont des paires (jgl.Pair) d'icones,
     * la 1re tant l'icone de l'objet simple, et la seconde celle de l'objet slectionn.
     * <P>
     */
    public class MyCellRenderer implements ListCellRenderer {
	
	AbstractBorder border;
	public MyCellRenderer() {
	    border = new EmptyBorder(4,4,4,4);
	}
			
	public Component getListCellRendererComponent(JList list, Object value,
						      int index, boolean isSelected, boolean cellHasFocus) {
	    JLabel symbol = null;						 
	    if (isSelected) {
		symbol = (JLabel) ((Pair)value).second;
		symbol.setBackground(UIManager.getColor("ComboBox.selectedBackground"));
		symbol.setForeground(UIManager.getColor("ComboBox.selectedForeground"));
	    } else {
		symbol = (JLabel) ((Pair)value).first;
		symbol.setBackground(UIManager.getColor("ComboBox.background"));
		symbol.setForeground(UIManager.getColor("ComboBox.foreground"));
	    }
	    symbol.setBorder(border);
	    if(UIManager.getLookAndFeel().getName().equals("CDE/Motif"))
		symbol.setOpaque((index!=-1));
	    else
		symbol.setOpaque(true);
	    return symbol;
	}
    }
}
