package spikingneuron.drawable;

import java.awt.Point;
import java.awt.Color;
import java.awt.Graphics;
import java.util.Vector;
import java.util.Enumeration;
import spikingneuron.math.Screen2D;
import spikingneuron.math.Coordinate2D;
import spikingneuron.tools.Tools;
import spikingneuron.tools.Clock;
import spikingneuron.tools.DataFlowAgent;
import spikingneuron.tools.RealFlowProducer;
import spikingneuron.neurons.NeuronBaseModel;

/**
*<FONT SIZE=2>
* @version 1.0, Lausanne le 18 Mai 1998 
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Modlise la courbe du potentiel membranaire d'un neurone.
* </TT></STRONG><FONT SIZE=3>
* <P>
* La courbe est constitue d'un ensemble de points, d'impulsions et de priode rfractaire, pendant
* lesquelles le trac peut vent. ne pas tre ralis. <BR>
* L aussi, il y a possibilit de mmoriser (dupliquer) une courbe afin de la redessiner de manire
* attnue.
* <P>
*/
public class PotentialCurve extends Drawable implements DataFlowAgent, Cloneable {
    /*
      Pour un trace fluide, il faut appeler ensureCapacity avant d'ajouter
      des points au graphique. (Attention au depassement d'alloc. memoire)
	
    */
    public static final int GROWING_INCREMENT = 1000;
    public static final int INITIAL_CAPACITY = 5000;

    // Gestion des couleurs  amliorer, en fct du nb de courbe, (pls sur meme graphe),
    // et en fonction de l'actualit des donnes (ancienne courbe mmorise).

    public Color curveColor;
    public Color spikeColor;
    public Color refractColor;
    public boolean valuedRefractPeriode;
	
    protected int spikeUpLimits;
    protected double spikeVoltage;
    protected double thresholdVoltage;
    protected double refractoryVoltage;
    protected boolean absoluteLimits;

    protected Vector pointsStock; // of Coordinate2D
    protected Coordinate2D lastPoint; //Point lastPoint;
    protected RealFlowProducer input; // input producer
    protected Coordinate2D verticalBounds; // BoundingBox

    protected static final int START_POINT_TAG = 1;
    protected static final int CURVE_POINT_TAG = 2;
    protected static final int SPIKE_POINT_TAG = 3;
	
    protected class CurveItem {
	int tag;
	Coordinate2D point;
	CurveItem() {};
	CurveItem(int tag, Coordinate2D point) {
	    this.tag = tag;
	    this.point = point;
	}
    }

    // Constructeurs .........................
    protected PotentialCurve() {
	super();
    }
	
    public PotentialCurve(Color colorCurve, Color colorSpike, Color colorRefract) {
	super();
	lastPoint = null;
	this.input = null;
	this.curveColor = colorCurve;
	this.spikeColor = colorSpike;
	this.refractColor = colorRefract;
	this.spikeVoltage = 0.;
	this.spikeUpLimits = 0;
	this.thresholdVoltage = 0.;
	this.refractoryVoltage = 0.;
	this.absoluteLimits = false;
	this.valuedRefractPeriode = false;
	this.verticalBounds = new Coordinate2D(0.0,0.0);
	pointsStock = new java.util.Vector(INITIAL_CAPACITY,GROWING_INCREMENT);
    }
	
    public PotentialCurve(Color colorLine, Color colorSpike, Color colorRefract, int nbPoints) {
	super();
	lastPoint = null;
	this.input = null;
	this.curveColor = colorLine;
	this.spikeColor = colorSpike;
	this.refractColor = colorRefract;
	this.spikeVoltage = 0.;
	this.spikeUpLimits = 0;
	this.thresholdVoltage = 0.;
	this.refractoryVoltage = 0.;
	this.absoluteLimits = false;
	this.valuedRefractPeriode = false;
	this.verticalBounds = new Coordinate2D();
	pointsStock = new Vector(nbPoints,nbPoints/2);
    }

    public java.lang.Object clone() {
	PotentialCurve newer = new PotentialCurve();
	newer.input = input;
	newer.curveColor = curveColor;
	newer.spikeColor = spikeColor;
	newer.refractColor = refractColor;
	newer.spikeVoltage = spikeVoltage;
	newer.spikeUpLimits = spikeUpLimits;
	newer.thresholdVoltage = thresholdVoltage;
	newer.refractoryVoltage = refractoryVoltage;
	newer.valuedRefractPeriode = valuedRefractPeriode;
	if (lastPoint != null)
	    newer.lastPoint = (Coordinate2D)lastPoint.clone();
	else
	    newer.lastPoint = null;
			
	if (verticalBounds != null)
	    newer.verticalBounds = (Coordinate2D)verticalBounds.clone();
	else
	    newer.verticalBounds = null;
	if (pointsStock != null)
	    newer.pointsStock = (Vector)pointsStock.clone();
	else
	    newer.pointsStock = null;
	return newer;
    }
		
		
    // Accesseurs ..........................
    // Il manque ceux des couleurs... attributs public.
    public void enableRefractoryTrace() {
	valuedRefractPeriode = true;
    }
    public void disableRefractoryTrace() {
	valuedRefractPeriode = false;
    }
	
    public void holdCurve(PotentialCurve stump) {
	pointsStock.removeAllElements();
	//		input = stump.input;
	spikeVoltage = stump.spikeVoltage;
	spikeUpLimits = stump.spikeUpLimits;
	absoluteLimits = stump.absoluteLimits;
	thresholdVoltage = stump.thresholdVoltage;
	refractoryVoltage = stump.refractoryVoltage;
	valuedRefractPeriode = stump.valuedRefractPeriode;
	if (stump.lastPoint != null)
	    lastPoint = (Coordinate2D)stump.lastPoint.clone();
	else
	    lastPoint = null;
			
	if (stump.verticalBounds != null)
	    verticalBounds = (Coordinate2D)stump.verticalBounds.clone();
	else
	    verticalBounds = null;
	if (pointsStock != null)
	    for (Enumeration e = stump.pointsStock.elements();e.hasMoreElements();)
		pointsStock.addElement(e.nextElement());
	//		curveColor = Tools.brighter(stump.curveColor,130);
	//		spikeColor = Tools.brighter(stump.spikeColor,130);
	//		refractColor = Tools.brighter(stump.refractColor,70);

	curveColor = Tools.brightness(Tools.brightness(stump.curveColor,80),1.5);
	spikeColor = Tools.brightness(Tools.brightness(stump.spikeColor,80),1.5);
	refractColor = Tools.brightness(stump.refractColor,1.5);
    }
	
    public void setLimits(double thresholdVoltage, double refractoryVoltage, double spikeVoltage) {
	this.spikeVoltage = spikeVoltage;
	this.thresholdVoltage = thresholdVoltage;
	this.refractoryVoltage = refractoryVoltage;
	this.absoluteLimits = false;
	verticalBounds.set(refractoryVoltage, thresholdVoltage);
    }
    public void setLimits(double thresholdVoltage, double refractoryVoltage, int spikeBorder) {
	this.spikeUpLimits = spikeBorder;
	this.thresholdVoltage = thresholdVoltage;
	this.refractoryVoltage = refractoryVoltage;
	this.absoluteLimits = true;
	verticalBounds.set(refractoryVoltage, thresholdVoltage);
    }
    public void setInput(RealFlowProducer input) {
	this.input = input;
    }

    // Ces deux methodes sont a placer dans la hierarchie Drawable...	
    public Coordinate2D getVerticalBounds() {
	/* Ne prend pas en compte la limite superieur du spike */
	return verticalBounds;
    }
    public Coordinate2D getHorizontalBounds() {
	if (pointsStock.size()>0)
	    return new Coordinate2D(0.0,((CurveItem)(pointsStock.lastElement())).point.getC0());
	else
	    return new Coordinate2D(0.0,0.0);
    }
	
    // Divers ................................
    public void ensureCapacity(int nbPoints) {
	pointsStock.ensureCapacity(nbPoints);
    }


    // Drawable extend ............................	
    public void redrawAt(Graphics g, Screen2D view) {
	CurveItem item;
	Point older, newer;
	int spikeScr, timeScr;
	int thresholdScr = view.scrY(thresholdVoltage);
	int refractScr = view.scrY(refractoryVoltage);
	if (absoluteLimits)
	    spikeScr = view.getScrMinY()+spikeUpLimits;
	else
	    spikeScr = view.scrY(spikeVoltage);
	older = null; // Init. des variables, le compilo ne connait pas la sequence dans le vecteur.
	newer = null;
		
	// Traitements des elements du vecteur...		
	if ((g!=null)&&(pointsStock.size()>0)) {
	    for (int i=0; i<pointsStock.size(); i++) {
		item = (CurveItem)pointsStock.elementAt(i);
		timeScr = view.scrX(item.point.getX());
		switch (item.tag) {
		case SPIKE_POINT_TAG :
		    if (valuedRefractPeriode) { 
			g.setColor(refractColor);
			g.drawLine(timeScr,thresholdScr,timeScr,refractScr);
		    }
		    g.setColor(spikeColor);
		    g.drawLine(timeScr,thresholdScr,timeScr,spikeScr);
		    break;
		case START_POINT_TAG :
		    g.setColor(curveColor);
		    older = view.scrPoint(item.point);
		    break;
		case CURVE_POINT_TAG :
		    newer = view.scrPoint(item.point);
		    g.drawLine(older.x,older.y,newer.x,newer.y);
		    older = newer;
		    break;
		}
	    }
	}
    }

    // DataFlowAgent interface implement .............................
    public void resetTime() {
	lastPoint = null;
	verticalBounds.set(0.0,0.0);
	pointsStock.removeAllElements();
	notifyChanges();
    }

    public void computeNextTic() {
	Graphics g;
	Screen2D view;
	DisplayContext dspCtx;
	double time = Clock.sharedInstance.getTime();
	double value = input.getRealOutput();
	Coordinate2D point = new Coordinate2D(time,value);
		
	if (value==NeuronBaseModel.SPIKE_POTENTIAL) {
	    // Trace du pulse, pour tous les contextDevices, et reinit. de lastPoint.
	    int startY, stopY, x;
	    if (synchronizedDisplay)
		for (Enumeration e = contexts.elements();e.hasMoreElements();) {
		    dspCtx = (DisplayContext)e.nextElement();
		    g = dspCtx.getGraphics();
		    view = dspCtx.getView();
		    if (g!=null) {
			startY = view.scrY(thresholdVoltage);
			if (absoluteLimits)
			    stopY = view.getScrMinY()+spikeUpLimits;
			else
			    stopY = view.scrY(spikeVoltage);
			x = view.scrX(time);
			if (lastPoint!=null) { // on bouche le trou de la courbe ...
			    g.setColor(curveColor);
			    g.drawLine(view.scrX(lastPoint.getX()),view.scrY(lastPoint.getY()),x,startY);
			}
			if (valuedRefractPeriode) { // ... on amene la courbe au niveau refractaire ...
			    g.setColor(refractColor);
			    g.drawLine(x,startY,x,view.scrY(refractoryVoltage));
			}
			// ... et on trace le spike.
			g.setColor(spikeColor);
			g.drawLine(x,startY,x,stopY);
		    }
		}
	    if (lastPoint==null)
		pointsStock.addElement(new CurveItem(START_POINT_TAG, new Coordinate2D(time,thresholdVoltage)));
	    else
		pointsStock.addElement(new CurveItem(CURVE_POINT_TAG, new Coordinate2D(time,thresholdVoltage)));
	    pointsStock.addElement(new CurveItem(SPIKE_POINT_TAG, point));
	    lastPoint = null;
	} else if (value != NeuronBaseModel.REFRACTORY_POTENTIAL) {
	    // Point de la courbe...
	    if (value<verticalBounds.getC0())
		verticalBounds.setC0(value);
	    else if (value>verticalBounds.getC1())
		verticalBounds.setC1(value);
		
	    if (lastPoint == null) // est-ce le premier point ?
		if (valuedRefractPeriode) {
		    Coordinate2D postPoint = new Coordinate2D(time,refractoryVoltage);
		    pointsStock.addElement(new CurveItem(START_POINT_TAG, postPoint));
		    pointsStock.addElement(new CurveItem(CURVE_POINT_TAG, point));
		    for (Enumeration e = contexts.elements();e.hasMoreElements();) {
			dspCtx = (DisplayContext)e.nextElement();
			g = dspCtx.getGraphics();
			view = dspCtx.getView();
			if (g!=null) {
			    g.setColor(curveColor);
			    g.drawLine(view.scrX(postPoint.getX()), view.scrY(postPoint.getY()),
				       view.scrX(point.getX()), view.scrY(point.getY()));
			}
		    }
		} else pointsStock.addElement(new CurveItem(START_POINT_TAG, point));

	    else { // Le point n'est pas en debut de courbe, on doit tracer le segment.
		if (synchronizedDisplay) {
		    Point lastScrPoint;
		    Point newScrPoint;
		    for (Enumeration e = contexts.elements();e.hasMoreElements();) {
			dspCtx = (DisplayContext)e.nextElement();
			view = dspCtx.getView();
			lastScrPoint = view.scrPoint(lastPoint);
			newScrPoint = view.scrPoint(point);
			if (!newScrPoint.equals(lastScrPoint)) {
			    g = dspCtx.getGraphics();
			    if (g!=null) {
				g.setColor(curveColor);
				g.drawLine(lastScrPoint.x,lastScrPoint.y,
					   newScrPoint.x,newScrPoint.y);
			    }
			}
		    }
		}
		pointsStock.addElement(new CurveItem(CURVE_POINT_TAG, point));
	    }
	    lastPoint = point;
	}
    }
}
