package spikingneuron.generators;

import java.lang.Math;
import java.util.Random;
import spikingneuron.math.Coordinate2D;
import spikingneuron.tools.Clock;
import spikingneuron.tools.DataFlowAgent;

/**
*<FONT SIZE=2>
* @version 1.0, Lausanne le 1 Juin 1998 
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Gnrateur de bruit blanc (pseudo), continu.
* </TT></STRONG><FONT SIZE=3>
* <P>
* Le signal est dfinit par la superposition de n sinusoides, (amplitude, frquence et angle initial
* alatoire). 
* Le signal total est compos d'un offset auquel est ajout la somme des signal sinusoidaux, mise 
* l'chelle  l'aide d'un paramtre dfinissant l'amplitude maximum de cette somme.
* <P>
*/
public class NoisyGenerator extends SignalGenerator {

    public static final String IDENTITY = "Generateur de bruit";
    protected static final double maxPulsation = 1E12;
    protected static final double maxAmplitude = 1E2;

    protected double offset;
    protected double amplitudeMax;
    protected double amplitudes[];
    protected double pulsations[];
    protected double dephasages[];
    protected double scaleFactor = 0.0;
    protected long seed;
    protected Random random;

    // Constructeurs ...........................	
    public NoisyGenerator() {
	super();
	offset = 0.0;
	amplitudeMax = 0.0;
	seed = System.currentTimeMillis();
	random = new Random(seed);
	pulsations = new double[10];
	dephasages = new double[10];
	amplitudes = new double[10];
	for (int i=0; i<10; i++) {
	    pulsations[i] = random.nextDouble()*maxPulsation;
	    dephasages[i] = random.nextDouble()*2*Math.PI;
	    amplitudes[i] = random.nextDouble()*maxAmplitude;
	}
	signal = offset;
	updateOutput();
    }

    public NoisyGenerator(double I0, double A, int nbSinus, long seed) {
	super();
	offset = I0;
	amplitudeMax = A;
	this.seed = seed;
	random = new Random(seed);
	pulsations = new double[nbSinus];
	dephasages = new double[nbSinus];
	amplitudes = new double[nbSinus];
	for (int i=0; i<nbSinus; i++) {
	    pulsations[i] = random.nextDouble()*maxPulsation;
	    dephasages[i] = random.nextDouble()*2*Math.PI;
	    amplitudes[i] = random.nextDouble()*maxAmplitude;
	}
	signal = I0;
	updateOutput();
    }
	
    // Accesseurs .............................
    public double getOffset() {
	return offset;
    }
    public double getAmplitude() {
	return amplitudeMax;
    }
    public int getNbSinus() {
	return pulsations.length;
    }
    public long getSeed() {
	return seed;
    }

    public void setOffset(double offset) {
	this.offset = offset;
    }
    public void setAmplitude(double amplitude) {
	this.amplitudeMax = amplitude;
    }
    public void setNbSinus(int nbSinus) {
	pulsations = new double[nbSinus];
	dephasages = new double[nbSinus];
	amplitudes = new double[nbSinus];
	for (int i=nbSinus-1; i>=0; i--) {
	    pulsations[i] = random.nextDouble()*maxPulsation;
	    dephasages[i] = random.nextDouble()*2*Math.PI;
	    amplitudes[i] = random.nextDouble()*maxAmplitude;
	}
    }

    public void setSeed(long seed) {
	this.seed = seed;
	random.setSeed(seed);
	for (int i=getNbSinus()-1; i>=0; i--) {
	    pulsations[i] = random.nextDouble()*maxPulsation;
	    dephasages[i] = random.nextDouble()*2*Math.PI;
	    amplitudes[i] = random.nextDouble()*maxAmplitude;
	}
    }
	

    // Divers ..............................
    protected void calculateScaleFactor() {
	double sum = 0.0;
	for (int i=getNbSinus()-1; i>=0; i--)
	    sum += amplitudes[i];
	scaleFactor = amplitudeMax / sum;
    }
	
    public String getIdentity() {
	return NoisyGenerator.IDENTITY;
    }

    public Coordinate2D getRange() {
	return new Coordinate2D(offset-amplitudeMax, offset+amplitudeMax);
    }

    // DataFlowAgent interface implement ...............
    public void resetTime() {
	calculateScaleFactor();
	signal = offset;
	updateOutput();
    }

    public void computeNextTic() {
	double sum = 0.0;
	double t = Clock.sharedInstance.getTime();
	for (int i=getNbSinus()-1; i>=0; i--)
	    sum += amplitudes[i] * Math.sin(dephasages[i] + t * pulsations[i]); 
	signal = offset + sum * scaleFactor;
    }
}
