package spikingneuron.generators;

import java.lang.Math;
import spikingneuron.math.Coordinate2D;
import spikingneuron.tools.Clock;
import spikingneuron.tools.DataFlowAgent;

/**
*<FONT SIZE=2>
* @version 1.0, Lausanne le 27 Mai 1998 
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Gnrateur d'impulsions de temps de survenance dterministe.
* </TT></STRONG><FONT SIZE=3>
* <P>
* Le signal est une suite d'impulsions de largeur <I>pulseWidth</I> (en Tics), rgulirement espaces.
* Le client peut dfinir le temps (sec) entre les impulsions (dure de simulation non-dterministe), ou
* le nombre d'impulsions (dure dterministe). <BR>
* L'offset est la valeur de base du signal, et l'amplitude est la valeur ajoute lors d'une impulsion.
* <P>
*/
public class PulsesGenerator extends SignalGenerator {

    public static final String IDENTITY = "Pulses Generator";

    protected int nbPulses;
    protected double offset;
    protected int pulseWidth; // [Tic]
    protected double pulseAmplitude; // [A]
    protected boolean withPulseCount; // Flag discriminant la simulation de dure dterministe ou non-dt.
    protected double deltaT; // temps entre pulse

    // La version avec flag indiquant l'initialisation est un peu
    // naive (et lourde, car un test pas vraiment ncessaire est fait 
    // chaque tic). L'autre mthode ncessite la synchro avec
    // l'interface AVANT le resetTime... c.f. RandomPulsesGenerator.
	
    private boolean initialized;
    private double nextPulse; // occurence du prochain pulse
    private int tics; // nb de tics restant

    // Constructeurs ...........................	
    public PulsesGenerator() {
	super();
	withPulseCount = false;
	offset = 0.0;
	nbPulses = 0;
	pulseWidth = 0;
	pulseAmplitude = 0.0;
	signal = offset;
	updateOutput();
	initialized = false;
	deltaT = 0.0;
	nextPulse = 0.0;
	tics = 0;
    }

    public PulsesGenerator(double I0, int nbPulses,
			   double A, int tics) {
	super();
	this.offset = I0;
	this.nbPulses = nbPulses;
	this.pulseWidth = tics;
	this.pulseAmplitude = A;
	signal = offset;
	updateOutput();
	initialized = false;
	deltaT = 0.0;
	nextPulse = 0.0;
	tics = 0;
	withPulseCount = true;
    }

    public PulsesGenerator(double I0, double delay,
			   double A, int tics) {
	super();
	this.offset = I0;
	this.pulseWidth = tics;
	this.pulseAmplitude = A;
	signal = offset;
	updateOutput();
	initialized = false;
	deltaT = delay;
	nbPulses = 0;
	nextPulse = 0.0;
	tics = 0;
	withPulseCount = false;
    }
	
    // Accesseurs .............................
    public double getOffset() {
	return offset;
    }
    public double getPulseAmplitude() {
	return pulseAmplitude;
    }
    public int getNbPulses() {
	return nbPulses;
    }
    public double getDeltaPulses() {
	return deltaT;
    }
    public int getPulseWidth() {
	return pulseWidth;
    }
    public boolean isNbPulsesFix() {
	return withPulseCount;
    }

    public void setOffset(double offset) {
	this.offset = offset;
    }
    public void setPulseAmplitude(double amplitude) {
	this.pulseAmplitude = amplitude;
    }
    public void setNbPulses(int nbPulses) {
	this.nbPulses = nbPulses;
	withPulseCount = true;
    }
    public void setPulsesDelta(double delta) {
	deltaT = delta;
	withPulseCount = false;
    }
    public void setPulseWidth(int tics) {
	this.pulseWidth = tics;
    }
	

    // Divers ..............................
    public String getIdentity() {
	return PulsesGenerator.IDENTITY;
    }

    public Coordinate2D getRange() {
	return new Coordinate2D(Math.min(offset, offset+pulseAmplitude),
				Math.max(offset, offset+pulseAmplitude));
	// On laisse au compilo le soin d'optimiser tout cela... !?!
    }

    // DataFlowAgent interface implement ...............
    public void resetTime() {
	signal = offset;
	updateOutput();
	tics = 0;
	nextPulse = Clock.sharedInstance.getInitialTime();
	initialized = false;
    }

    public void computeNextTic() {
	double t = Clock.sharedInstance.getTime();
	if (!initialized) {
	    initialized = true;
	    tics = 0;
	    if (withPulseCount) {
		deltaT = Clock.sharedInstance.getDuration() / nbPulses;
		nextPulse += deltaT/2;
	    }
	    else
		nextPulse += deltaT;
	    signal = offset;
	}
	if (tics>0) {
	    tics--;
	    if (tics==0)
		signal = offset;
	    // else signal = offset + amplitudePulse;
	}
	if (t>=nextPulse) {
	    // emmission du pulse
	    nextPulse = t + deltaT;
	    tics = pulseWidth;
	    signal = offset + pulseAmplitude;
	}
    }
}
