package spikingneuron.generators;

import java.lang.Math;
import spikingneuron.math.Coordinate2D;
import spikingneuron.math.RandomDistribution;
import spikingneuron.tools.Clock;
import spikingneuron.tools.DataFlowAgent;

/**
*<FONT SIZE=2>
* @version 1.0, Lausanne le 27 Mai 1998 
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Gnrateur d'impulsions de temps de survenance non-dterministe.
* </TT></STRONG><FONT SIZE=3>
* <P>
* Le signal est une suite d'impulsions de largeur <I>pulseWidth</I> (en Tics).
* Le temps entre les impulsions est une ralisation d'une variable alatoire, suivant une distribution
* exponentielle, de paramtre <I>lambda</I>. <BR>
* L'offset est la valeur de base du signal, et l'amplitude est la valeur ajoute lors d'une impulsion.
* <P>
*/
public class RandomPulsesGenerator extends SignalGenerator {

    public static final String IDENTITY = "Random Pulses Generator";

    protected double offset;
    protected double lambda;
    protected int pulseWidth; // [Tic]
    protected double pulseAmplitude; // [A]
    protected long seed;
    protected RandomDistribution random;

    private double nextPulse; // occurence du prochain pulse
    private int tics; // nb de tics restant

    // Constructeurs ...........................	
    public RandomPulsesGenerator() {
	super();
	seed = System.currentTimeMillis();
	random = new RandomDistribution(seed);
	offset = 0.0;
	lambda = 0.0;
	pulseWidth = 0;
	pulseAmplitude = 0.0;
	signal = offset;
	updateOutput();
	nextPulse = 0.0;
	tics = 0;
    }

    public RandomPulsesGenerator(double I0, double lambda,
				 double A, int tics, long seed) {
	super();
	this.offset = I0;
	this.lambda = lambda;
	this.pulseWidth = tics;
	this.pulseAmplitude = A;
	this.seed = seed;
	random = new RandomDistribution(seed);
	signal = offset;
	updateOutput();
	nextPulse = 0.0;
	tics = 0;
    }
	
    // Accesseurs .............................
    public double getOffset() {
	return offset;
    }
    public double getPulseAmplitude() {
	return pulseAmplitude;
    }
    public double getLambda() {
	return lambda;
    }
    public int getPulseWidth() {
	return pulseWidth;
    }
    public long getSeed() {
	return seed;
    }

    public void setOffset(double offset) {
	this.offset = offset;
    }
    public void setPulseAmplitude(double amplitude) {
	this.pulseAmplitude = amplitude;
    }
    public void setLambda(double lambda) {
	this.lambda = lambda;
    }
    public void setPulseWidth(int tics) {
	this.pulseWidth = tics;
    }
    public void setSeed(long seed) {
	this.seed = seed;
	random.setSeed(seed);
    }

    // Divers ..............................
    public String getIdentity() {
	return RandomPulsesGenerator.IDENTITY;
    }

    public Coordinate2D getRange() {
	return new Coordinate2D(Math.min(offset, offset+pulseAmplitude),
				Math.max(offset, offset+pulseAmplitude));
	// On laisse au compilo le soin d'optimiser tout cela....						
    }

    // DataFlowAgent interface implement ...............
    public void resetTime() {
	// Synchronisation avec interface AVANT resetTime !!!!!
	random.setSeed(seed);
	signal = offset;
	updateOutput();
	tics = 0;
	nextPulse = Clock.sharedInstance.getInitialTime() + random.nextExponential(lambda);
    }

    public void computeNextTic() {
	double t = Clock.sharedInstance.getTime();
	if (tics>0) {
	    tics--;
	    if (tics==0)
		signal = offset;
	}
	if (t>=nextPulse) {
	    // emmission du pulse
	    nextPulse = t + random.nextExponential(lambda);
	    tics = pulseWidth;
	    signal = offset + pulseAmplitude;
	}
    }
}
