package spikingneuron.generators;

import java.lang.Math;
import spikingneuron.tools.Clock;
import spikingneuron.tools.DataFlowAgent;
import spikingneuron.math.Coordinate2D;

/**
*<FONT SIZE=2>
* @version 1.0, Lausanne le 1 Juin 1998 
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Gnrateur de signal (pseudo)carr (crnaux).
* </TT></STRONG><FONT SIZE=3>
* <P>
* Le signal est dfinit sur 3 niveaux (moyen (= valeur initiale), bas et haut), et dcompos en 8 segments:<BR>
* t1 : I = IMoyen (maintient) <BR> 
* t2 : Temps de monte (IMoyen <= I <= IHaut) <BR>
* t3 : I = IHaut <BR>
* t4 : Temps de descente (IHaut >= I >= IMoyen) <BR>
* t5 : I = IMoyen (maintient) <BR>
* t6 : Temps de descente (IMoyen >= I >= IBas) <BR>
* t7 : I = IBas (maintient) <BR>
* t8 : Temps de monte (IBas <= I <= IMoyen) <BR>
* Tous les temps sont dfinis en secondes.
* <P>
* Pour simplifier l'implmentation, une valeur de courant est dfinie pour chaque phase.
*/
public class SquareGenerator extends SignalGenerator {

    public static final String IDENTITY = "Square Waveforme";

	
    protected double[] currents;
    protected double[] timing;

    private int phase; // Phase actuelle
    private double nextPhase; // temps de dbut de la prochaine phase
    private double startPhase; // temps de dbut de la phase actuelle
    private double[] slope; // pente des diffrentes phases


    // Constructeurs ...........................	
    public SquareGenerator() {
	super();
	currents = new double[8];
	timing = new double[8];
	slope = new double[8];
	phase = 0;
	signal = 0.0;
	nextPhase = Double.POSITIVE_INFINITY;
	startPhase = 0.0;
	updateOutput();
    }

    /**
     * Initialisation du gnrateur.
     * @param timing Tableau de 8 double, spcifiant la dure de chaque phase, en secondes.
     */
    public void init(double IOffset, double IHigh, double ILow, double[] timing) {
	phase = 0;
	startPhase = 0.0;
	this.timing = timing;
	nextPhase = timing[0];
	currents[2] = currents[3] = IHigh;
	currents[6] = currents[7] = ILow;
	signal = currents[0] = currents[1] = currents[4] = currents[5] = IOffset;
	for (int i=0;i<8;i+=2)
	    this.slope[i] = (currents[(i+1)%8]-currents[i])/timing[i];
	updateOutput();
    }
	
    public double getIOffset() {
	return currents[0];
    }
    public double getIHigh() {
	return currents[2];
    }
    public double getILow() {
	return currents[6];
    }
    public double getTiming(int timingId) {
	return timing[timingId];
    }

    public void setIOffset(double current) {
	currents[0] = currents[1] = currents[4] = currents[5] = current;
	for (int i=1;i<8;i+=2)
	    slope[i] = (currents[(i+1)%8]-currents[i])/timing[i];
    }
    public void setIHigh(double current) {
	currents[2] = currents[3] = current;
	slope[1] = (currents[2]-currents[1])/timing[1];
	slope[3] = (currents[4]-currents[3])/timing[3];
    }
    public void setILow(double current) {
	currents[6] = currents[7] = current;
	slope[5] = (currents[6]-currents[5])/timing[5];
	slope[7] = (currents[0]-currents[7])/timing[7];
    }
    public void setTiming(int timingId, double time) {
	timing[timingId] = time;
	slope[timingId] = (currents[(timingId+1)%8]-currents[timingId])/time;
    }
	

    // Divers ..............................
    public String getIdentity() {
	return SquareGenerator.IDENTITY;
    }

    public Coordinate2D getRange() {
	double max = Double.NEGATIVE_INFINITY;
	double min = Double.POSITIVE_INFINITY;
	for (int i=0;i<8;i++) {
	    if (max<currents[i])
		max = currents[i];
	    if (min>currents[i])
		min = currents[i];
	}
	return new Coordinate2D(min,max);
    }

    // DataFlowAgent interface implement ...............
    public void resetTime() {
	phase = 0;
	startPhase = Clock.sharedInstance.getInitialTime();
	signal = currents[0];
	nextPhase = startPhase + timing[0];
	updateOutput();
    }

    public void computeNextTic() {
	double t = Clock.sharedInstance.getTime();
	if (t>=nextPhase) {
	    phase = ++phase % 8;
	    signal = currents[phase];
	    startPhase = nextPhase;
	    nextPhase += timing[phase];
	}
	else // y = b + at
	    signal = currents[phase]+slope[phase]*(t-startPhase);
    }
}

