package spikingneuron.math;

import java.lang.Math;
import java.util.Random;

/**
*<FONT SIZE=2>
* @version 1.0, Lausanne le 27 Mai 1998 
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Cette classe permet de gnrer des variables alatoires </STRONG>(<I> Ralisations
* d'expriences  stochastiques</I>) <STRONG> suivant diffrentes distributions.
* </STRONG></TT><FONT SIZE=3>
* <P>
* Base sur le gnrateur congruentiel de jdk, cette classe offre des extensions tire de:
* <CITE>Brian D. Ripley, Stochastic Simulation, Wiley 1987</CITE>, et <CITE> Donald Knuth,
* The Art of Computer Programming, Volume 2</CITE>.
* Une instance, <CODE><B>sharedInstance</B></CODE> est dfinie en tant qu'attribut publique de la classe, avec
* choix non dterministe de la graine.
* <P>
* <TT><B>Distributions Discrtes: </B></TT><UL TYPE = disc>
* <LI> Uniforme U[0, max),
* <LI> Bernoulli B(1,p) = succs-chec
* <LI> Gomtrique G(p) = nb d'exp. de Bernoulli B(p) -> 1er succs
* <LI> Binomiale B(n,p) = nb succs sur n exp de Bernoulli B(p)
* <LI> Poisson P(l) = loi des vnements rares.
* </UL><TT><B>Distributions Continues: </B></TT><UL TYPE = disc>
* <LI> Uniforme continue U[0,max)
* <LI> Normale rduite (gaussienne rduite) N(0,1)
* <LI> Normale (gaussienne) N(Esprance, ecart-type^2)
* <LI> Exponentielle Exp(lambda)
* <LI> Cauchy (1) demi-cercle
* <LI> Beta (alpha, beta)
* </UL><I>Manque: Binomiale ngative, hypergomtrique, Student, KhiDeux, Rayleigh, F, Gamma ...
* et mthode pour distributions particulires / mixtes, par exemple rjection gnralise, ou alias </I>.
* <P>
*/

public class RandomDistribution extends Random {

	/**
	* Instance public, partage, et initialise avec une graine alatoire.
	*/
	public static final RandomDistribution sharedInstance = new RandomDistribution();


// Constructeurs .......................................................................

	/**
	* Construction avec choix alatoire (time dependant) de la graine du gnrateur congruentiel
	*/
	public RandomDistribution() { super(); }
	
	/**
	* Construction du gnrateur avec initialisation de la graine.
	* @param	seed	La graine du gnrateur congruentiel (48 bits).
	*/
	public RandomDistribution(long seed) { super(seed); }


// Distributions discrtes .........................................................

	/**
	* X ~ Unif[0,max-1]
	*/
	public int nextUnif(int max) { return nextInt()%max; }

	/**
	* X ~ Unif[0,max-1]
	*/
	public long nextUnif(long max) { return nextLong()%max; }

	public boolean nextBernoulli(double probability) { return (nextDouble()<probability); }
	

	/**
	* X ~ G(p) : X = [X'~E(-ln(1-p))]
	*/
	public long nextGeometrical(double probability) {
		return ( (probability>=1.0) ? 1 :
				 (long) Math.ceil(nextExponential(-Math.log(1.0-probability))));
	}

	public int nextBinomial(int nbExp, double probability) {
		if (probability<=1E-20) return 0;
		if (probability>=1.0) return nbExp;
		if (nbExp<200) {
			int success = 0;
			for (int i=0; i<nbExp; i++)
				if (nextBernoulli(probability))
					success++;
			return success;
		}
		final int K = 200;
		int k = nbExp;
		int i, X = 0;
		double V;
		double phi = probability;
		do {
			i = (int)Math.ceil(k*phi)+1;
			V = nextBeta((double)i,(double)(k+1-i));
			if (phi<V) {
				phi /= V;
				k = i-1;
			} else {
				X+=i;
				phi = (phi-V)/(1.0-V);
				k-=i;
			}
		} while (k>K);
		for (i=0;i<k;k++) {
			V = nextDouble();
			if (V<probability) X++;
		}
		return (X);
	}

	public int nextPoisson(int lambda) {
		if (lambda<=50) {
			float U;
			float P = 1.0f;
			float c = (float)Math.exp(-lambda);
			int N = 0;
			do {
				U = nextFloat();
				P *= U;
				N++;
			} while (P>=c);
			return (--N);
		}
		if (lambda<=250) {
			double c = 0.767 - 3.36/lambda;
			double b = Math.PI/Math.sqrt(3.0*lambda);
			double a = b*lambda;
			double k = Math.log(c)-lambda-Math.log(b);
			double U1,U2,X;
			double N;
			do {
				do {
					U1 = nextDouble();
					X = (a-Math.log(1.0-U1)/U1)/b;
				} while (X<=-0.5);
				N = Math.ceil(X+0.5);
				U2 = nextDouble();
			} while ((a-b*X+Math.log(U2/Math.sqrt(1.+Math.exp(a-b*X)))) 
			  > (k+N*Math.log((double)lambda)-Math.log(RandomDistribution.fact(N))));
			return ((int)N);
		} else
			return ((int) nextNormal(lambda,lambda));
	}

// Distributions continues .................................................................

	/**
	* X ~ Unif[0,max)
	*/
	public float nextUnif(float max) { return nextFloat()*max; }

	/**
	* X ~ Unif[0,max)
	*/
	public double nextUnif(double max) { return nextDouble()*max; }
	
	/**
	* X ~ N(m,v^2) : X = N(0,1)*v + m
	*/
	public double nextNormal(double esperance, double ecartType) { 
		return ((nextGaussian()*ecartType) + esperance);
		// Autre possibilit: la mthode Box-Muller:
		// Z1, Z2 -> Unif[0,1)
		// X1 = (Sqrt(-2ln(Z1)) * Cos(2*PI*Z2)) * ecartType + esperance
		// X2 = (Sqrt(-2ln(Z1)) * Sin(2*PI*Z2)) * ecartType + esperance
		// X1 et X2 sont deux ralisations 'indpendantes'...
	}

	/**
	* X ~ Exp(l): approx par: X = ln( U[0,1) / -l).
	*/
	public double nextExponential(double lambda) {
		return (Math.log(nextDouble())/(-lambda));
	}

	/**
	* Ralisation exponentielle, approxime par la mthode de VonNeumann.
	*/
	public double nextVNexponential(double lambda) { // Von Neumann algo
		int I = 0;
		double U, Up, T;
		while (true) {
			T = U = nextDouble();
			do
				if (U<=(Up=nextDouble()))
					return ((double)(I+T));
			while ((U=nextDouble())<Up);
			I++;
		}
	}

	public double nextCauchy() { // Cauchy et rjection
		double U1,U2,V;
		do {
			U1 = nextDouble();
			U2 = nextDouble();
			V  = 2*U2 - 1.0;
		} while ((U1*U1+V*V)>=1.);
		return (V/U1);
	}
	
	
	public double nextBeta(double alpha, double beta) {
		double U1,U2,v1,v2;
		alpha = 1.0/alpha;
		beta  = 1.0/beta;
		do {
			U1 = nextDouble();
			U2 = nextDouble();
			v1 = Math.pow(U1,alpha);
			v2 = Math.pow(U2,beta);
		} while (v1+v2>1.0);
		return (v1/(v1+v2));
	}

/* Reste  implmenter:  
		nextStudent(t);
		nextKhiDeux(d);
		nextRayleigh(x); 
		nextF();
		nextGamma(alpha);
		...
*/ 
	

	// Tools... .......................................................
	public static double fact(double num) {
		if ((num<0.)||(Math.ceil(num)!=num))
			return 0.;
		double i = 1.;
		long u;
		for (u=2;u<=(long)num;u++)
			i*=u;
		return (i);
	}
	
	public static long fact(long num) {
		long u,i = 1;
		for (u=2;u<=num;u++)
			i*=u;
		return (i);
	}
}
