package spikingneuron.neurons;

import java.lang.Math;
import spikingneuron.tools.Clock;
import spikingneuron.math.Coordinate2D;

/**
*<FONT SIZE=2>
* @version 1.1, Lausanne le 3 septembre 1998
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Classe modlisant les neurones appartenant au modle <I>Integrate & Fire</I>.
* </TT></STRONG><FONT SIZE=3>
* <P>
* Dans ce modle (driv du modle plus gnral <I>Spike Response Model</I>, les lments constitutifs
* du neurone sont apparents  des composants lectriques simples: Le rle de la synapse est ralis
* par un filtre passe-bas (Intgrateur passif). Un couple R-C parallle reprsente le soma, et lorsque
* la tension au borne de ces deux derniers lments atteint la valeur de seuil, le circuit est schunt,
* et une impulsion est mise. <BR>
* <TT><B>Equation:</B>  Tau(m) * du/dt = -u(t) + RI(t)* </TT><BR>
* <P>
*/
public class IntegrateAndFire extends NeuronBaseModel {

    public static final String IDENTITY = "Integrate & Fire Model";

    protected double resetPotential; // [V] Valeur du potentiel aprs ref. absolue.
    protected double tauMembrane;	// [s] Constante de temps de la membrane.
    private double integrationStartTime; // [s] Dbut de la proch. phase d'int.
	
    // Constructors ..............................................................

    public IntegrateAndFire(int id) {
	super(id);
	resetPotential = 0.0;
    }

    public void ifInit(double tauMembrane, double resetPotential) {
	this.tauMembrane = tauMembrane;
	this.resetPotential = resetPotential;
	integrationStartTime = 0.0;
	membranePotential = resetPotential;
    }

    // Accessors ....................................................................

    public double getResetPotential() {
	return resetPotential;
    }
    public void setResetPotential(double potential) {
	this.resetPotential = potential;
    }

    /**
     * Retourne la constante de temps de la membrane du modle de neurone.
     * (Rsistance * Capacit membranaire).
     * @return		Constante de temps, en secondes.
     */
    public double getTauMembrane() {
	return tauMembrane;
    }

    /**
     * Modifie la constante de temps de la membrane du modle de neurone.
     * @param	newTau	Nouvelle constante de temps, en secondes.
     */	
    public void setTauMembrane(double newTau) {
	this.tauMembrane = newTau;
    }


	
    // Divers ........................................................................
    public String getIdentity() {
	return IntegrateAndFire.IDENTITY;
    }

    public void resetTime() {
	super.resetTime();
	integrationStartTime = Clock.sharedInstance.getInitialTime();
	membranePotential = resetPotential;
	updateOutput();
    }

    public Coordinate2D getRange() {
	return new Coordinate2D(resetPotential,meanThreshold+gapThreshold);
    }
	
    // DataFlowAgent interface implement ................................................
    public void computeNextTic() {
	/* Il faut traiter la plage temporelle [t-dt,t[
	   l'horloge nous donne la borne temporelle droite (maximum), et les donnes
	   en entres proviennent de la prcdente plage temporelle, donc borne gauche.
	   
	   -K si (t) < (t(f)+ tabs)
	   U(t)   =   u(t-dt)+(dt*(R*i(t-dt)-u(t-dt))/tau) si (t) > (t(f)+tabs)
	   Spike puis Ur si (t=t(f), soit u(t)>= seuil)
		
	   On dtermine dans quelle phase se situe la tranche de temps  simuler.
	   3 Cas sont  distinguer: (IT = integrationStartTime)
	   1) dans la priode rfractaire : IT > t
	   2) dans la phase d'intgration : IT < t-dt
	   3) A cheval entre les 2 phases : IT c [t-dt,t[
		
	*/
	super.computeNextTic();
		
	double u;
	double i;
	double t = Clock.sharedInstance.getTime();
	double dt = Clock.sharedInstance.getDeltaT();

	if (integrationStartTime>t) // priode rfractaire, cas (1)
	    membranePotential = REFRACTORY_POTENTIAL;
	else {
	    if (integrationStartTime < (t-dt))  // phase d'integration, cas (2)
		u = membranePotential;
	    else { //  cheval entre les 2 phases, cas (3)
		u = resetPotential; // r-initialisation du potentiel membranaire
		dt = integrationStartTime-t; // priode 'active'
	    }
	    i = calculateExternalCurrent() + calculatePreSynapticCurrent();
	    u += dt*(resistance * i - u) / tauMembrane; // Euler
	    if ((u >= threshold)&&(membranePotential<u)) {
				// Emission du spike...
		membranePotential = SPIKE_POTENTIAL;
				// ... et dtermination de la prochaine phase d'intgration
		computeEffectiveRefractoryTime();
		integrationStartTime = t + refractoryTime;
	    }
	    else membranePotential = u;
	}
    }
}
