package spikingneuron.neurons;

import java.awt.*;
import java.util.*;
import java.awt.event.*;
import java.awt.image.ImageObserver;
import javax.swing.*; 

import spikingneuron.tools.Tools;
import spikingneuron.drawable.LayeredDrawable;

/**
*<FONT SIZE=2>
* @version 1.1, Lausanne le 3 septembre 1998 
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Paneau d'affichage du rseau de neurones.
* </TT></STRONG><FONT SIZE=3>
* <P>
* Conteneur pour l'affichage du rseau, sur 2 couche: le fond contient les connexions,
* et la surface les neurones. Il serait prfrable d'employer un layeredPane (c.f. <I>Swing</I>),
* afin de permettre aux connexions de modifier leurs tats sans affecter la reprsentation des
* neurones...
* <P>
* Version brouillon: Pour les 3 topologies possible, une image de fond est charge en mmoire. Cette
* image contient la reprsentation des connexions, et elle est affiche (avec resize) pour chaque neurone.
* La classe dfinit galement un tableau de couleurs utilise pour reprsenter un neurone (la couleur
* est choisie en fonction du potentiel membranaire)... JDK 1.1.x ne permettant pas de modifier la
* couleur d'une instance de Color, ces couleurs sont instancies une fois pour toutes, en tant qu'attribut
* de classe.
*/

public class NetworkView extends JPanel implements ComponentListener {

    public static final Color DEFAULT_BK_COLOR = UIManager.getColor("Panel.background");

    public static final int MAX_COLORS = 128; 			// Nb de couleurs disponible pour l'affichage des neurones
    public static final int SPIKE_COLOR = MAX_COLORS;   // Indice de la couleur d'affichage lors d'un fire
    public static final int REFRACTORY_COLOR = MAX_COLORS + 1; // Indice de la couleur d'aff. de la priode rfract.
    public static Color neuronsColor[]; // Tableau de couleurs.

    protected Image[] neuronImg;        // Images de fond pour chacune des topologies possible.
    protected Image buffer;             // Buffer Image pour le dessin off-screen
    protected Graphics graphicsBuffer;  // Context d'affichage dans le buffer off-screen

    protected Dimension dim;            // Dimension de la zone d'affichage du rseau
    protected Insets borderSize;        // Taille des bordures
    protected int userWidth;            // Espace util
    protected int userHeight;
    protected int netWidth;             // Dimensions du rseau
    protected int netHeight;

    protected Font font;

    protected java.util.Vector items;   // Elments devant tre reprsents.
    protected java.util.Vector changed; // Elements  redessiner (dont l'tat  chang)
	
    protected Dimension neuronBoundle;  // Dimension de la zone d'affichage d'un neurone
	
    protected int currentItem;
    protected int topologieMode;

    public NetworkView(int initialWidth, int initialHeight, Font font) {
	super(true);
	setOpaque(true);
	borderSize = getInsets();
	dim = new Dimension(initialWidth, initialHeight);
	neuronBoundle = new Dimension();
	this.font = font;
		
	netWidth = 0;
	netHeight = 0;
	currentItem = 0;
	topologieMode = 0;
	addComponentListener(this);
	userWidth = initialWidth - borderSize.right - borderSize.left;
	userHeight = initialHeight - borderSize.top - borderSize.bottom;
	neuronImg = new Image[3];
	neuronImg[0] = Tools.loadImageIcon(getClass(),"/images/full.gif","").getImage();
	neuronImg[1] = Tools.loadImageIcon(getClass(),"/images/4neighbor.gif","").getImage();
	neuronImg[2] = Tools.loadImageIcon(getClass(),"/images/8neighbor.gif","").getImage();
	buffer = null; // La cration doit etre diffre.
	graphicsBuffer = null;
	items = new java.util.Vector(500,200);
	changed = new java.util.Vector(500,200);
    }

    public Dimension getPreferredSize() {
	return dim;
    }

    public synchronized Dimension getMinimumSize() {
	return dim;
    }

    /* imprativement appele avant de crer les neurons. */	
    public void setNetworkSize(int width, int height) {
	netWidth = width;
	netHeight = height;
	currentItem = 0;
	neuronBoundle.width = (int)(userWidth/netWidth);
	neuronBoundle.height = (int)(userHeight/netHeight);
	drawBackground();
	// Version plus labore, avec place pour les connexions...
	//		neuronBoundle.width = (int)(userWidth / (0.5+1.5*netWidth));
	//		neuronBoundle.height = (int)(userHeight / (0.5+1.5*netHeight));
    }
	
    public Graphics getBufferedGraphics() { return graphicsBuffer; }

    public Dimension getNeuronSpace() { return neuronBoundle; }
		
    public Rectangle addItem(LayeredDrawable item) {
	Point p = new Point();
	//		p.x = (int)( ((currentItem % netWidth) * 1.5 + 0.5) * neuronBoundle.width)
	//		p.y = (int)( ((currentItem / netHeight) * 1.5 + 0.5) * neuronBoundle.height)
	p.x = (int)(currentItem % netWidth) * neuronBoundle.width;
	p.y = (int)(currentItem / netWidth) * neuronBoundle.height;
	items.addElement(item);
	currentItem++;
	return new Rectangle(p,neuronBoundle);
    }

    public void addChangedItem(LayeredDrawable item) { changed.addElement(item); }
    public void removeItems() {
	items.removeAllElements();
	currentItem = 0;
    }
	
    public void setTopologie(int topologieMode) { this.topologieMode=topologieMode; }
    public Image getNeuronBackImage() { return neuronImg[topologieMode]; }
	
    public void paint(Graphics g) {
	if (!dim.equals(getSize()))
	    componentResized(null);
	if (graphicsBuffer == null) {
	    buffer = createImage(userWidth,userHeight);
	    graphicsBuffer = buffer.getGraphics();
	    graphicsBuffer.setFont(font);
	    for (Enumeration e = items.elements(); e.hasMoreElements(); )
		((LayeredDrawable)e.nextElement()).setBufferGraphics(graphicsBuffer);
	}
	drawBackground();
	for (Enumeration e = items.elements(); e.hasMoreElements(); )
	    ((LayeredDrawable)e.nextElement()).paintAt(LayeredDrawable.LAYER_ONE);
	super.paint(g); // Border, Child & Background
	g.drawImage(buffer,borderSize.left,borderSize.right,this);
    }

    public void repaintNeurons() {
	Graphics g = getGraphics();
	for (Enumeration e = changed.elements(); e.hasMoreElements(); )
	    ((LayeredDrawable)e.nextElement()).paintAt(LayeredDrawable.LAYER_ONE);
	changed.removeAllElements();
	g.drawImage(buffer,borderSize.left,borderSize.right,this);
    }
		
    // ItemListener interface ............................
    public void componentResized(ComponentEvent e) {
	Point p = new Point();
	dim = getSize();
	borderSize = getInsets();
	userWidth = dim.width - borderSize.right - borderSize.left;
	userHeight = dim.height - borderSize.top - borderSize.bottom;
	neuronBoundle.width = (int)(userWidth/netWidth);
	neuronBoundle.height = (int)(userHeight/netHeight);
	buffer = createImage(userWidth,userHeight);
	if (buffer !=null) { // sinon, ce sera fait plus tard.
	    graphicsBuffer = buffer.getGraphics();
	    graphicsBuffer.setFont(font);
	    drawBackground();
	    for (int i = 0; i<currentItem; i++) {
		p.x = (int)( (i % netWidth) * neuronBoundle.width);
		p.y = (int)( (i / netWidth) * neuronBoundle.height);
		((LayeredDrawable)items.elementAt(i)).resize(graphicsBuffer,new Rectangle(p,neuronBoundle));
	    }
	}
    }
	
    protected void drawBackground() {
	if (graphicsBuffer != null) {
	    graphicsBuffer.setColor(DEFAULT_BK_COLOR);
	    graphicsBuffer.fillRect(0,0,userWidth,userHeight);
	    for (int i = 0; i<netHeight*netWidth; i++)
		graphicsBuffer.drawImage(neuronImg[topologieMode],
					 (int)( (i % netWidth) * neuronBoundle.width),
					 (int)( (i / netWidth) * neuronBoundle.height),
					 neuronBoundle.width,neuronBoundle.height,this);
	}
    }

    public void componentMoved(ComponentEvent e) {}
    public void componentShown(ComponentEvent e) {}
    public void componentHidden(ComponentEvent e) {}

    static {
	neuronsColor = new Color[MAX_COLORS+2];
	for (int i = 0; i<MAX_COLORS; i++)
	    neuronsColor[i] = new Color(256*i/MAX_COLORS, 255-256*i/MAX_COLORS, 0);
	neuronsColor[SPIKE_COLOR] = Color.white;
	neuronsColor[REFRACTORY_COLOR] = Color.black;
    }

}
