package spikingneuron.neurons;

import java.util.Vector;
import java.util.Enumeration;
import spikingneuron.tools.Clock;
import spikingneuron.tools.RealFlowProducer;
import spikingneuron.tools.SpikeFlowProducer;
import spikingneuron.generators.DiscreteNoisyGenerator;
import spikingneuron.math.Coordinate2D;
import spikingneuron.math.RandomDistribution;

/**
*<FONT SIZE=2>
* @version 1.1, Lausanne le 3 septembre 1998
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Classe de base (<B>abstraite</B>) des diffrents modles de neurones.
* </TT></STRONG><FONT SIZE=3>
* <P>
* La classe implment l'interface <CODE>RealFlowProducer</CODE>, le flux de rels tant la squence
* des valeurs du potentiel membranaire, en Volts. <BR>
* La classe implmente galement l'interface <CODE>SpikeFlowProducer</CODE>, la squence des temps
* de survenance (d'occurrence) des impulsions, en secondes, pouvant tre obtenue.
* <P>
* Dfinition publique des constantes utilises pour reprsenter le potentiel lors de l'mission
* d'une impulsion, ainsi que pendant l'ventuelle priode rfractaire absolue.<BR>
* La classe contient les lments d'instances communs  tous les modles implments,  savoir:
* <UL TYPE = disc>
* <LI> La valeur moyenne de seuil (threshold), ainsi que son cart-type (voir plus bas),
* <LI> La dure moyenne de la priode rfractaire, ainsi que son cart-type (refractoryTime),
* <LI> La rsistivit de la membrane (R),
* <LI> La constante de temps de la synapse, Tau synaptic. (la doc. de W.Gerstner n'est pas trs claire
*      ( = confuse) au sujet des constantes  utiliser pour les diffrents kernel).
* <LI> Un gnrateur de courant de bruit (discret), d'amplitude fixe. Voir  ce propos les
*      remarques de la classe <CODE>DiscreteNoisyGenerator</CODE>.
* <HR CLEAR="60 en">
* <LI> Le potentiel membranaire actuel (<B>attribut devant tre dtermin par les sous-classes</B>),
* <LI> Le train d'impulsion du neurone,
* <LI> L'identification de l'instance,
* <LI> Un ensemble de dendrites (connexion  d'autre neurones, en fait  d'autre
*      <CODE>SpikeFlowProducer</CODE>),
* <LI> Un ensemble de gnrateurs externe de courant (implmentant <CODE>RealFlowProducer</CODE>),
* <LI> Un <I>tampon</I> mmorisant l'tat du neurone (potentiel membranaire); c'est par le biais de
*      ce tampon que les obets environnants perevront l'tat du neurone, ce qui permet de simuler
*      une simultanit des actions (Interface <CODE>RealFlowProducer</CODE>).
* </UL>
* Un attribut de classe implmente le compteur d'impulsions, partag par toutes les instances, et
* servant  sommer les impulsions misent par une population de neurones (dans ce cas, la population
* tant l'ensemble des instances). Les mthodes (de classe) permettant la consultation/modification de
* ce compteur sont fournie. <I>(Voir  ce propos le commentaire de <CODE>PopulationActivity</CODE></I>.
* <P>
* La dtermination des paramtres initiaux (activit initiale (=antrieur) de la population & tat
* initial (antrieur) du neurone) m'a pos quelques problmes: Pour l'activit initiale d'un neurone,
* et bien que toutes les options soient plausibles, deux choix semblent logiques: <OL TYPE = 1>
* <LI> Le dernier spike  eu lieu en t = -refTime, ce qui conduit  dbuter la phase d'intgration 
*      en t = t<SUB>o</SUB> = 0.
* <LI> Le dernier spike  eu lieu en t -> -infini (ou pas de spike), donc on est en phase 
*      d'intgration, avec un stimuli initial = 0...   <B>Choix adopt!</B>
* </OL>
* L'activit initial de la population, ncessaire pour que les neurones ragissent lors de la
* simulation en rseau, est plus difficile  dfinir. La topologie n'tant pas fixe, cette grandeur
* n'est pas utilise pour dterminer l'tat d'un neurone (ralisable uniquement avec une population
* compltement connecte); c'est bel et bien l'ensemble des impulsions reues qui dterminent cet
* tat. Par consquent, on ne peut fixer l'activit initial  une valeur particulire, et influencer
* ainsi l'activit des neurones. Il faut explicitement dfinir les moments d'occurences des derniers
* spikes. Il y a alors (quoique galement dans le cas prcdant) conflit avec l'activit initial d'un
* neurone, qui ne devrait plus tre l'un des deux choix prcdemment cits, mais bel et bien dtermine
* en fonction de l'activit initiale de la population. (En fait, le problme vient du fait que ces
* deux paramtres sont considrs sparments, alors qu'ils sont totalement dpendants puisque
* reprsentant la mme chose). <BR>
* Il a t dcid que l'activit initiale serait dtermine par simulation: l'utilisateur choisit le
* temps pendant lequel un courant externe est appliqu aux neurones (ces derniers tant initialement
* inactifs, comme dfinit prcedemment), ainsi que la forme du signal. Puis, ce courant est 
* 'supprim' (ventuellement remplac par un autre), et la simulation se poursuit (Cette phase initiale
* s'inscrivant dans la portion ngative de l'axe du 'temps', t = t<SUB>0</SUB> = 0 reprsentant le
* dbut de la simulation proprement dite). <BR>
* <B>Il est laiss  l'objet grant la simulation la responsabilit de l'implmentation de cette
* phase initiale. </B>
* <P>
* <TT><B>Remarques: </B></TT><UL TYPE = disc>
* <LI> Le bruit introduit sur le niveau de seuil n'est pas vraiment conforme (utilisation d'une
*      gausienne centre sur le seuil, et d'cart-type donn), mais  t choisi pour sa simplicit,
*      tant au niveau de l'implmentation (calcul) que du nombre de paramtres.
*      C.f. <CITE>Spiking Neurons, W. GERSTNER,  Chapter 1, p. 33 </CITE>.
* <LI> Pour gnrer les valeurs de seuil et de priode rfractaire absolue, les neurones utilisent
*      l'instance partage de <CODE>RandomDistribution</CODE>. L'ventuelle rinitialisation de la
*      graine de ce gnrateur est laisse  la responsabilit de l'objet grant la simulation.
* <LI> Pour des questions de performances, les neurones sont implments sous forme d'objets simples,
*      et pas en tant qu'objets actifs (threads), bien que cela aurait conduit, j'en suis certain,  
*      une modlisation extrmement propre, et forcment  une implmentation plus claire (au niveau 
*      du cycle de vie (c.f. remarques de la classe <CODE>Clock</CODE>).
* <LI> A explorer: le comportement du neurone devrait pouvoir tre dcrit  l'aide d'un AFD, ce qui
*	   rendrait le code plus lisible...
* </UL><BR>
* @see spikingneuron.tools.Clock
* @see spikingneuron.tools.RealFlowProducer
* @see spikingneuron.tools.SpikeflowProducer
* @see spikingneuron.generators.DiscreteNoisyGenerator
* <P>
*/
public abstract class NeuronBaseModel implements RealFlowProducer, SpikeFlowProducer {

    // Attributs de classe ...........................................................................

    /**
     *  Potentiel membranaire lors de l'mission d'un spike.
     */
    public static final double SPIKE_POTENTIAL = Double.POSITIVE_INFINITY;

    /**
     *  Potentiel membranaire pendant la priode rfractaire absolue. 
     */
    public static final double REFRACTORY_POTENTIAL = Double.NEGATIVE_INFINITY;
	
    /** 
     * Identit du modle. (Originellement, utilis comme intitul de la liste de slection).
     */
    public static final String IDENTITY = "Abstract Neural Model";
	
    /**
     * Compte les impulsions gnres par l'ensemble des neurones instancis.
     * (Utilis pour dterminer l'activit de la population, et la frquences des impulsions).
     * @see spikingneuron.neurons.PopulationActivity
     */
    protected static int nbSpike = 0;

    // Attributs d'instances .............................................................................

    // Paramtres du modle .....................................................

    /**
     * Moyenne du seuil au-dela duquel le neurone met un spike.
     * Lorsque le potentiel membranaire dpasse un certain seuil, une impulsion
     * est mise. Ce seuil est une ralisation d'une variable alatoire, obissant
     *  une distribution Normale, de moyenne <CODE>meanThreshold</CODE>.
     * Ce seuil est dfinit en Volts.
     */
    protected double meanThreshold; // moyenne du seuil [V]

    /**
     * Ecart-type de la valeur de seuil, en Volts.
     * (Introduction du bruit par le biais d'un seuil stochastique.)
     */
    protected double gapThreshold; // ecart-type du seuil [V]

    /**
     * Dure moyenne de la priode rfractaire, en secondes.
     */
    protected double meanRefractoryTime;

    /**
     * Ecart-type de la duree de la priode rfractaire, en secondes.
     */
    protected double gapRefractoryTime;

    /**
     * Rsistance ohmique de la membrane ([Ohm]).
     */
    protected double resistance;
	
    /**
     * Constante de temps de la synapse, en secondes.
     */
    protected double tauSynaptic;

    /**
     * Gnrateur de courant de bruit.
     */
    protected DiscreteNoisyGenerator noisyCurrent;
	

    // Internal state ...................................................

    /**
     * <B>Prochaine</B> valeur de seuil, en Volts.
     * Ralisation d'une variable altoire ~ N(meanThreshold, etThreshold^2).
     */
    protected double threshold;
	
    /**
     * Dure effective de la <B>prcdente</B> priode rfractaire.
     * Ralisation d'une variable altoire ~ N(meanRefractoryTime, etRefractoryTime^2).
     */
    protected double refractoryTime;


    /**
     * Potentiel membranaire du neurone [V].
     */
    protected double membranePotential;
	
    /**
     * Train d'impulsions: ensemble de Double reprsentant les moments d'occurences
     * des impulsions.
     */
    protected Vector spikeTrain;

    /**
     * Identificateur de l'instance.
     * Cet identificateur simplifie l'identification d'une instance dans un rseau (sa localisation).
     */
    protected int id;
	
    /**
     * Tampon mmorisant le potentiel membranaire en t - dt.
     */
    protected double output;
	
    /**
     * Ensemble de dendrites.
     * Permet la connextion d'autre neurones, via un objet de type Dendrites.
     * @See spikingneuron.neurons.NeuronBaseModel.Dendrites
     */
    protected Vector dendrites;
	
    /**
     * Ensemble de gnrateur externe de courant (<CODE>RealFlowProducer</CODE>).
     * Ces gnrateurs doivent fournir en sortie une valeur de courant, en Ampres.
     * @See spikingneuron.tools.RealFlowProducer
     */ 
    protected Vector externalStimulus;


    // Mthodes de Classe ...................................................................................
	
    /**
     * Incrmente le compteur d'impulsions d'une unit.
     * Cette mthode ne devrait normalement pas tre invoque depuis une  classe ne faisant
     * pas partie de la hirarchie de NeuronBaseModel (sous-classes); et par consquent
     * devrait tre <I>protected</I> plutt que <I>public</I>.
     */
    public static synchronized void incrementFireCounter() { nbSpike ++; }
		
    /**
     * Retourne la valeur du compteur d'impulsions, soit le nombre total d'impulsions mises
     * par l'ensemble des instances de la classes, depuis la dernire (r)initialisation.
     * @return		Nombre d'impulsions depuis la dernire initialisation.
     */
    public static synchronized int getFireCounter() { return nbSpike; }
		
    /**
     * Rinitialisation du compteur d'impulsion.
     * Le total des impulsions mises est rinitialis  0.
     */
    public static synchronized void resetFireCounter() { nbSpike = 0; }
	
    // Mthodes d'instances ......................................................................................

    // Constructeurs ..............................................................
	
    /**
     * Construit (partiellement)  une nouvelle instance de neurone.
     * Tous les paramtres du modle sont initialiss  0.
     * @param	id	Identificateur de l'instance (Entier).
     */
    protected NeuronBaseModel(int id) {
	this.id = id;
	meanThreshold		= 0.0; 	gapThreshold		= 0.0;
	meanRefractoryTime	= 0.0;	gapRefractoryTime	= 0.0;
	tauSynaptic			= 0.0;	resistance 			= 0.0;
	refractoryTime 		= 0.0;	threshold 			= 0.0;
	membranePotential	= 0.0;	output				= 0.0;

	noisyCurrent 		= new DiscreteNoisyGenerator();
	externalStimulus	= new Vector(2,5);
	dendrites			= new Vector(10,50);
	spikeTrain			= new Vector(SPIKE_TRAIN_SIZE, SPIKE_TRAIN_SIZE);
    }

    /**
     * (R)Initialisation des paramtres du modle.
     * Permet de paramtrer totalement le modle, en n'invoquant qu'une mthode. Toutes les
     * grandeurs sont  exprimer dans leurs units de bases (Volts, secondes, Ohms, Ampres).
     * Cette mthode est  tendre (surcharge avec <B>extension de la visibilit</B>) dans les
     * classes drives (la mthode publique <I>init(...)</I> d'une sous-classe de
     * <CODE>NeuronBaseModel</CODE> doit invoquer cette mthode: <CODE><I>super.init(...)</I></CODE>).
     * @param	meanThreshold		Valeur moyenne, en volts, du seuil dterminant l'mission d'impulsions [V].
     * @param	varThreshold		Variance, en volts, du seuil dterminant l'mission d'impulsions [V].
     * @param	meanRefractoryTime	Dure moyenne, en secondes, de la priode rfractaire absolue [s].
     * @param	varRefractoryTime	Variance, en secondes, de la dure de la priode rfractaire absolue [s].
     * @param	resistance			Rsistance lectrique, en ohms, de la membrane du neurone [Ohm].
     * @param	tauSynaptic			Constante de temps de la synapse, en secondes [s].
     * @param	noiseAmplitude		Amplitude maximale, en ampres, du courant de bruit [A].
     */
    public void nbmInit(double meanThreshold,		double gapThreshold,
			double meanRefractoryTime,	double gapRefractoryTime,
			double resistance,			double tauSynaptic,
			double noiseAmplitude) {
	this.meanThreshold = meanThreshold;
	this.gapThreshold = gapThreshold;
	this.meanRefractoryTime = meanRefractoryTime;
	this.gapRefractoryTime = gapRefractoryTime;
	this.resistance = resistance;
	this.tauSynaptic = tauSynaptic;
	noisyCurrent.setAmplitude(noiseAmplitude);
    }

    // Accesseurs & modificateurs ................................................................................
	
    /**
     * Retourne l'identificateur de l'instance.
     * @return		Entier identifiant l'instance.
     */
    public int getId() { return (id); }

    /**
     * Modifie l'identificateur de l'instance.
     * @param	newId	Nouvel identificateur d'instance (Entier).
     */
    public void setId(int newId) { this.id = newId; }

    /**
     * Retourne la moyenne de la loi Normale dterminant la tension de seuil du neurone.
     * @return		La valeur moyenne du potentiel membranaire limite, en Volts.
     * @see spikingneuron.neurons.NeuronBaseModel#meanThreshold
     */
    public double getMeanThreshold() { return meanThreshold; }

    /**
     * Retourne l'cart-type de la loi Normale dterminant la tension de seuil du neurone.
     * @return		Ecart-type du potentiel membranaire limite, en Volts.
     * @see spikingneuron.neurons.NeuronBaseModel#gapThreshold
     */
    public double getGapThreshold() { return gapThreshold; }

    /**
     * Modifie les paramtres de la distribution normale dterminant le seuil du potentiel membranaire.
     * Pour ne pas introduire de bruit (<I>'Noisy Threshold'</I>), l'cart-type doit valoire 0.
     * @param	newMean	Moyenne de la normale = valeur de seuil moyenne (Volts).
     * @param	newGap	Ecart-type de la normale = ecart maximum autours de la moyenne (Volts).
     */
    public void setThreshold(double newMean, double newGap) {
	this.meanThreshold = newMean;
	this.gapThreshold = newGap;
    }

    /**
     * Retourne la moyenne de la loi Normale dterminant la dure de la priode rfractaire absolue.
     * @return		Dure moyenne de la priode rfractaire absolue, en secondes.
     * @see spikingneuron.neurons.NeuronBaseModel#meanRefractoryTime
     */
    public double getMeanAbsoluteRefractoryTime() {
	return meanRefractoryTime;
    }

    /**
     * Retourne l'cart-type de la loi Normale dterminant la dure de la priode rfractaire absolue.
     * @return		Ecart-type de la priode rfractaire absolue, en secondes.
     * @see spikingneuron.neurons.NeuronBaseModel#gapRefractoryTime
     */
    public double getGapAbsoluteRefractoryTime() {
	return gapRefractoryTime;
    }

    /**
     * Modifie les paramtres de la distribution normale dterminant la dure de la priode rfractaire absolue.
     * Pour ne pas introduire de bruit (<I>'Noisy Reset'</I>), l'cart-type doit valoire 0.
     * @param	newMean	Moyenne de la normale = dure moyenne de la priode rfractaire (secondes).
     * @param	newGap	Ecart-type de la normale = ecart maximum autours de la moyenne (secondes).
     */
    public void setAbsoluteRefractoryTime(double newMean, double newGap) {
	this.meanRefractoryTime = newMean;
	this.gapRefractoryTime = newGap;
    }

    /**
     * Retourne la valeur de rsistance (lectrique) de la membrane du modle de neurone.
     * @return		Rsistance membranaire, en Ohms.
     */
    public double getResistance() { return resistance; }

    /**
     * Modifie la valeur de la rsistance lectrique de la membrane du modle de neurone.
     * @param	newResistance	Nouvelle valeur de la rsistance membranaire, en Ohms.
     */
    public void setResistance(double newResistance) {
	this.resistance = newResistance;
    }

    /**
     * Retourne la constante de temps de la synapse du modle de neurone.
     * @return		Constante de temps, en secondes.
     */
    public double getTauSynaptic() { return tauSynaptic; }

    /**
     * Modifie la constante de temps de la synapse du modle de neurone.
     * @param	newTau	Nouvelle constante de temps, en secondes.
     */	
    public void setTauSynaptic(double newTau) { this.tauSynaptic = newTau; }

    /**
     * Retourne l'amplitude du courant de bruit.
     * @return Amplitude maximale du courant (externe) de bruit, en Ampres.
     */
    public double getNoiseAmplitude() { return noisyCurrent.getAmplitude(); }

    /**
     * Modifie l'amplitude maximale du courant de bruit.
     * Le courant de bruit, assimil  un courant 'externe', est produit par une instance de 
     * <CODE>NoisyGenerator</CODE>. Pour ne pas introduire de courant de bruit (<I>'Noisy current'</I>),
     * l'amplitude maximale doit tre fixe  0.
     * @param	newAmplitude		Amplitude maximale du courant de bruit, en Ampres.
     * @see spikingneuron.generators.NoisyGenerator
     */
    public void setNoiseAmplitude(double newAmplitude) {
	noisyCurrent.setAmplitude(newAmplitude);
    }
	

    // Connexions manipulators ........................................................................................

    /**
     * Retourne l'ensemble des sources externes (gnrateurs) de courant auxquelles l'instances est
     * raccorde.  Le gnrateur utilis pour la production du courant de bruit n'est pas compris dans
     * cet ensemble.
     * @return		Enumration de <CODE>RealFlowProducers</CODE>.
     * @see spikingneuron.tools.RealFlowProducer
     */
    public Enumeration getExternalInputs() {
	return externalStimulus.elements();
    }
		
    /**
     * Retourne l'ensemble des sources (pre-synaptic spiking neurons) d'impulsions
     * auxquelles l'instances est raccorde.
     * @return	Enumration de <CODE>Dendrites</CODE> (description des paramtres des 'synapses').
     * @see spikingneuron.neurons.NeuronBaseModel.Dendrite
     */
    public java.util.Enumeration getInputs() {
	return dendrites.elements();
    }
	
    /**
     * Ralise l'ajout d'une connexion.
     * Les instanciation et destruction des connexions sont ralises en interne,
     * l'utilisateur n'a donc pas  s'en procuper.
     * @param	input	le neurone  connecter en entre (axon du neurone). 
     * @param	weight	coefficient d'efficacit de la synapse; D.V:  ]-INF;+INF[
     * @param	delay	dlais de transmission de l'axone, en secondes; D.V: [0;+INF[
     * @return	Rfrence  la connexion cre, utilisable pour sa suppression/modification.
     */
    public NeuronBaseModel.Dendrite addInput(SpikeFlowProducer input, double weight, double delay) {
	NeuronBaseModel.Dendrite connexion = new NeuronBaseModel.Dendrite(input, weight, delay);
	dendrites.addElement(connexion);
	return connexion;
    }
	
    /**
     * Modifie globalement les paramtres des connexions.
     * @param	newWeight	coefficient d'efficacit des synapses (facteur). 
     * @param	newDelay	dlais de transmission des axones, en secondes. 
     */
    public void setInputs(double newWeight, double newDelay) {
	NeuronBaseModel.Dendrite d;
	for (Enumeration e = dendrites.elements(); e.hasMoreElements(); ) {
	    d = (Dendrite)e.nextElement();
	    d.delay = newDelay;
	    d.efficacityFactor = newWeight;
	}
    }
	
    /**
     * Modifie les paramtres d'une connexion.
     * La connexion  modifier est identifie par la valeur retourne par <CODE>addInput</CODE>.
     * @param	connexion	Rfrence  la connexion, retourne lors de sa cration.
     * @param	newWeight	coefficient d'efficacit de la synapse.
     * @param	newDelay	dlais de transmission de l'axones.
     * @see spikingneuron.neurons.NeuronBaseModel#addInput
     */
    public void setInput(NeuronBaseModel.Dendrite connexion, double newWeight, double newDelay) {
	// Assert. connexion is in dendrites.
	connexion.delay = newDelay;
	connexion.efficacityFactor = newWeight;
    }


    /**
     * Suppression de toutes les dendrites.
     */
    public void removeInputs() {
	dendrites.removeAllElements();
    }

    /**
     * Suppression d'une dendrite particulire.
     * La dendrite rfrence par <CODE>connexion</CODE> est supprime.
     * @param	connexion	Rfrence  la dendrite  supprimer.
     */
    public void removeInput(NeuronBaseModel.Dendrite connexion) {
	dendrites.removeElement(connexion);
    }

    /**
     * Ajoute une connexion  un gnrateur externe de courant.
     * Le gnrateur doit fournir un courant de sortie en Ampre.
     * @param	input	Gnrateur externe de courant  connecter.
     */
    public void addExternalInput(RealFlowProducer input) {
	externalStimulus.addElement(input);
    }
	
    /**
     * Supprime la connexion au gnrateur externe de courant.
     * @param	input	Gnrateur externe de courant  dconnecter.
     */
    public void removeExternalInput(RealFlowProducer input) {
	externalStimulus.removeElement(input);
    }
	
    /**
     * Supprime toutes les connexions aux gnrateurs externes.
     * Le courant de bruit n'est pas affect par l'invocation de cette mthode.
     */
    public void removeExternalInputs() {
	externalStimulus.removeAllElements();
    }
	
    // Mthodes Diverses ..................................................................


    /**
     * Retourne l'identit du modle.
     * @return	Identit du modle, dpendant du type de l'instance.
     */
    public String getIdentity() { return NeuronBaseModel.IDENTITY;}
	
    /**
     * Retourne la reprsentation de l'instance, sous forme de chane de caractres.
     * L'instance est identifie par un nombre entier, retourn par cette mthode.
     * @return	Reprsentation textuelle de l'instance (de l'entier identifiant l'instance).
     */
    public String toString() { return Integer.toString(id); }
	
    /**
     * Estimation du domaine de validit du potentiel membranaire.
     * @return	estimation de la plage (en Volts) dans laquelle le potentiel membranaire peut prendre valeur.	
     */
    public abstract Coordinate2D getRange();
	
    /**
     * Dtermine la valeur de seuil, d'aprs une distribution normale.
     * L'attribut <CODE>threshold</CODE> est mis  jour, avec un nouveau tirage suivant une distribution
     * gaussienne, N(mean,gap^2). La valeur de seuil tant maintenue par cette classe, la mthode est
     * prive.
     */
    private void computeEffectiveThreshold() {
	threshold =	RandomDistribution.sharedInstance.nextNormal(
								     meanThreshold, gapThreshold
								     );
    }

    /**
     * Dtermine la dure de la priode rfractaire, d'aprs une distribution normale.
     * L'attribut <CODE>refractoryTime</CODE> est mis  jour, avec un nouveau tirage suivant la valeur absolue
     * d'une distribution gaussienne: abs(N(mean,gap^2)).
     * Les sous-classes ont la responsabilit de gnrer une nouvelle valeur de seuil aprs chaque
     * spike.
     */
    protected void computeEffectiveRefractoryTime(){
	refractoryTime =	Math.abs(
					 RandomDistribution.sharedInstance.nextNormal(
										      meanRefractoryTime, gapRefractoryTime
										      )
					 );
    }

    // DataFlowAgent interface implement .......................................................................

    /**
     * Rinitialise l'instance.
     * Signal  l'instance que le temps est 'initialis'. Cette mthode est  tendre dans les sous-classes,
     * afin de procder aux initialisations des paramtres propres  chaque modles, mais doit toujours tre
     * invoque (Rinitialisation du train d'impulsions et dtermination de la valeur initiale de seuil).
     * @see spikingneuron.tools.DataFlowAgent
     */
    public void resetTime() {
	noisyCurrent.resetTime();
	computeEffectiveThreshold();
	spikeTrain.removeAllElements(); // clearing firing times
    }

    /**
     * Calcul le prochain tat du neurone.
     * Cette mthode est  tendre dans les sous-classe (l encore, avec invocation de la mthode parente).
     */
    public void computeNextTic() {
	noisyCurrent.computeNextTic();
    }
	
    /**
     * Somme toutes les contributions des gnrateurs externes de courant, ainsi que le courant de bruit.
     * @return	Somme des courants dus  des sources autres que des neurones impulsionnels.
     */
    protected double calculateExternalCurrent() {
	double current = noisyCurrent.getRealOutput();
	for (Enumeration e = externalStimulus.elements();e.hasMoreElements();)
	    current += ((RealFlowProducer)e.nextElement()).getRealOutput();
	return current;
    }
	
    /**
     * Calcul le courant rsultant de l'activit pr-synaptique.
     * Pour chaque spike de chaque synapse, le kernel <CODE>e(s)</CODE> est calcul. <BR>
     * <B>Kernel:</B><TT> e(s) = w * (s-delay)/(TauS^2) * exp (- (s-delay)/TauS)</TT><BR>.
     * Remarque: A la place de recalculer  chaque fois toutes les contributions, on pourrait
     * utiliser l'eq. aux diffrences pour raliser une approximation par la drive, ce qui conduirait
     *  simplement ajouter une quantit  une 'mmoire'; mais la somme des calculs  raliser est
     * du mme ordre, et cette solution ncessiterai la mmorisation d'une quantit pour chaque spike
     * de chaque synapse... (donc sns avantages rels). <BR>
     * Par contre, on pourrait ne ternir compte que des n derniers spikes... dans ce cas, soit on limite le 
     * nombre de d'impulsion du <I>spiketrain</I> (limitation en interne pour chaque neurone, et perte
     * d'information dans le cas o l'on implmente la possibilit d'examiner l'tat des neurones en un temps
     * donn, aprs la simulation), soit on modifie ce que retournent les <CODE>SpikeFlowProducer</CODE>
     * (en retournant p. ex. une rfrence au vecteur... mais la modlisation en souffrirait!). <BR>
     * Il serait sans doute prfrable de dfinir une nouvelle classe pour chacun des kernels, et d'utiliser
     * l'aggrgation, afin de pouvoir facilement substituer un kernel  un autre... toutefois, il faut
     * rsoudre le problme du paramtrage (Sparer l'interface du modle de neurone en 2).
     */
    protected double calculatePreSynapticCurrent() {
	double s = 0.0;
	double t = Clock.sharedInstance.getTime();
	double current = 0.0;
	NeuronBaseModel.Dendrite synapse;

	for (Enumeration dendriteEnum = dendrites.elements(); dendriteEnum.hasMoreElements() ;) {
	    synapse = (NeuronBaseModel.Dendrite)dendriteEnum.nextElement();
	    for (Enumeration spikeTrain = synapse.axon.getSpikesOutput(); spikeTrain.hasMoreElements();) {
		s = t - ((Double)spikeTrain.nextElement()).doubleValue() - synapse.delay;
		if (s>0.0) {
		    s /= tauSynaptic;
		    current += synapse.efficacityFactor * s/tauSynaptic * Math.exp(-s);
		}
	    }
	}
	return current;
    }			

    // DataFlowProducer interface implement .....................................................................
    public void updateOutput() {
	noisyCurrent.updateOutput();
	output = membranePotential;
	if (output == SPIKE_POTENTIAL) {
	    computeEffectiveThreshold(); // Nouvelle valeur de seuil
	    NeuronBaseModel.incrementFireCounter(); // add a spike for population activity
	    // update firing times
	    if ((SPIKE_TRAIN_LIMITED_SIZE) && (spikeTrain.size()==SPIKE_TRAIN_SIZE))
		spikeTrain.removeElementAt(0);
	    spikeTrain.addElement(new Double(Clock.sharedInstance.getTime()));
	}
    }

    // RealFlowProducer interface implement ....................................................................
    public double getRealOutput() { return output; }

    // SpikeFlowProducer interface implement ....................................................................
    public Enumeration getSpikesOutput() { return spikeTrain.elements(); }

    /**
     *<FONT SIZE=2>
     * @version 1.1, Lausanne le 3 septembre 1998
     * @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
     * <P><FONT SIZE=4><TT><STRONG>
     * Classe modlisant une dendrite (<I>class wrapper</I>).
     * </TT></STRONG><FONT SIZE=3>
     * <P>
     * Cette classe ne sert qu' dfinir un objet par aggrgation. Elle ne met  disposition
     * aucune mthode particulire, hormis le constructeur d'instances (tous les attributs sont publics).<BR>
     * Les lments aggrgs dans cette classe sont: <UL TYPE = disc>
     * <LI> l'axon connect (entre du signal).
     * <LI> le coefficient d'efficacit de la synapse, i.e. coefficient par lequel est multipli
     *      le courant en provenance de ladite synapse.
     * <LI> le dlai de propagation du signal.
     * </UL><P>
     */
    public class Dendrite {
	public SpikeFlowProducer	axon;
	public double				efficacityFactor;
	public double				delay;
		
	// Constructeurs ...................
	Dendrite(SpikeFlowProducer axon, double weight, double delay) {
	    this.axon = axon;
	    this.efficacityFactor = weight;
	    this.delay = delay;
	}
    }
}

/*----------------------------------------------------------------------------------------------------------
  Mthodes abstraites:
  --------------------
  -	public abstract Coordinate2D getRange() : domaine de validit
	
  Mthodes  tendre:
  ------------------
  -	public void resetTime() 				: rinitialisation du modle.
  -	public void computeNextTic() 			: simulation de la prochaine plage temporelle.

  -----------------------------------------------------------------------------------------------------------*/
