package spikingneuron.neurons;

import java.lang.Math;
import java.util.Enumeration;
import spikingneuron.tools.Clock;
import spikingneuron.math.Coordinate2D;

/**
*<FONT SIZE=2>
* @version 1.1, Lausanne le 3 septembre 1998
* @author Florian Seydoux (EPFL-Lami-Mantra, projet <I>Spiking Neurons</I>.) <HR>
* <P><FONT SIZE=4><TT><STRONG>
* Classe modlisant les neurones appartenant au modle <I>Spike Response Model</I>.
* </TT></STRONG><FONT SIZE=3>
* <P>
* Ce modle gnral dcrit le potentiel membranaire comme la superposition linaire de 2 composantes,
* l'une reprsentant l'influence des impulsions reues par le neurone, l'autre tant la raction du
* neurone  ses propres impulsions. C'est cette dernire raction qui doit etre modlise dans les
* sous-classes: cela permet de rduire au maximum les diffrences entre les modles  mmoire courte
* et ceux  mmoire longue. <BR>
* <TT><B>Equation:</B> u(t) = eta(t - ti - rAbs) + h(t)) <BR>
* eta(s) = ? -> classe abstraite <BR>
* h(t) = (Somme (sur tj = pre-synaptic) wij * epsilon(t - tj)) + hext(t) <BR> </TT>
* <P>
* <TT><B>Remarques: </B></TT><UL TYPE = disc>
* <LI> L'intgration du signal externe n'est pas ralise pendant la priode rfractaire absolue.
* </UL><P>
*/ 


public abstract class SpikeResponseModel extends NeuronBaseModel {

    protected double eta0; // [V]
    protected double tauEta; // [s]

    protected double hExternal;	 // excitation engendre par la source externe,
    protected double integrationStartTime; // [s] Dbut de la proch. phase d'int.
	
    // Constructors .............................................................
    public SpikeResponseModel(int id) {
	super(id);
	eta0 = 0.0;
	tauEta = 0.0;
	hExternal = 0.0; 		 
	integrationStartTime = Clock.sharedInstance.getInitialTime();
	membranePotential = 0.0; 
    }
	
    public void srInit(double eta0, double tauEta) {
	this.eta0 = eta0;
	this.tauEta = tauEta;
	hExternal = 0.0; 		 
	integrationStartTime = Clock.sharedInstance.getInitialTime();
	membranePotential = 0.0; 
    }
	
    // Accessors ..........................................................
    public double getEta0() {
	return eta0;
    }
    public void setEta0(double eta0) {
	this.eta0 = eta0;
    }

    public double getTauEta() {
	return tauEta;
    }
    public void setTauEta(double tauEta) {
	this.tauEta = tauEta;
    }
	
    // Divers ..............................................................
    public void resetTime() {
	super.resetTime();
	hExternal = 0.0; 		 // C.I. (1) : hExternal = threshold;
	integrationStartTime = Clock.sharedInstance.getInitialTime();
	membranePotential = 0.0; // C.I. (1): membranePotential = eta0 + hExternal;
	updateOutput();
    }
	
    public Coordinate2D getRange() {
	return new Coordinate2D(
				Math.min(meanThreshold-gapThreshold, meanThreshold-gapThreshold + eta0),
				Math.max(meanThreshold+gapThreshold, meanThreshold+gapThreshold + eta0));
    }


    // DataFlowAgent interface implement ..........................................

    /**
     * Calcul le kernel eta.
     * @param s 
     * <DL><DD>	<CODE>t - ts - tabs</CODE>, avec <I>ts</I> l'occurrence du dernier spike, et
     *			<I>tabs</I> la dure de la priode rfractaire absolue.</DL>
     */
    protected double eta(double s) {
	return (eta0 * Math.exp(-s/tauEta));
    }
	
    protected void updateStimulus(double dt, double iext) {
	// approximation par Euler: h(t+dt)=h(t)+dt(R i(t)-h(t)/tauS)
	hExternal += dt*(resistance*iext - hExternal/tauSynaptic);
    }

    public void computeNextTic() {
	/* Il faut traiter la plage temporelle [t-dt,t[, c-a-d dterminer u(t), en
	   connaissant u(t-dt), i(t-dt), et en dterminant etat(t) et h(t).

	   -K si (t) < (t(f)+ tabs)
	   u(t) =  (eta(t-tfi...)  + h(t), h(t) = h(t-dt)+dt(i(t-dt)-h(t-dt)/tauM)
	   Spike puis -K si (t=t(f), soit u(t)>= seuil, u(t-dt)<seuil)

	   On dtermine dans quelle phase se situe la tranche de temps  simuler.
	   3 Cas sont  distinguer: (IT = integrationStartTime)
	   1) dans la priode rfractaire : IT > t
	   2) dans la phase d'intgration : IT < t-dt
	   3) A cheval entre les 2 phases : IT c [t-dt,t[
		
	*/

	super.computeNextTic();
		
	double h; // calcul diffr, gain si refract priode.
	double u;
	double t = Clock.sharedInstance.getTime();
	double dt = Clock.sharedInstance.getDeltaT();
		
	if (integrationStartTime > t) // Phase refractaire
	    membranePotential = REFRACTORY_POTENTIAL;
	else {
	    updateStimulus(dt,calculateExternalCurrent()); // Choix (1b)
	    // L encore, le papier de W. Gerstner n'est pas vraiment satisfaisant... le terme
	    // R n'apparat que dans l'quation gnrale du modle I&F... 
	    // on trouve donc qqch du genre u(t) = ... + h(t) o h est un courant! (et u une tension) !
	    h = hExternal + calculatePreSynapticCurrent() * resistance;
	    u = myFiringResponse(t) + h;
	    if ((u>=threshold)&&(membranePotential<u)) {
				// Emission du spike...
		membranePotential = SPIKE_POTENTIAL;
		computeEffectiveRefractoryTime();
		integrationStartTime = t + refractoryTime;
	    }
	    else membranePotential = u;
	}
    }

    protected abstract double myFiringResponse(double t);
}
/*----------------------------------------------------
  Mthodes abstraites:
 --------------------
 protected abstract double myFiringResponse(double) : calcul de la reponse aux impulsions
 propres (kernel eta) 
 ------------------------------------------------------- */
